/*******************************************************************
 *   Copyright (C) 2011-2018, Michael Anders, FH-Wedel
 *
 *  This file is part of the ECC-Implementation "Academic Signature"
 *  Maintainer: Michael Anders
 *
 *  All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <wx/thread.h>

#include <wx/string.h>
#include <wx/file.h>
#include <wx/ffile.h>
#include <wx/filename.h>
#include <wx/intl.h>
#include <wx/msgdlg.h>
#include <wx/progdlg.h>

#include "helpersxx.h"
#include "globals.h"
#include "mylogin.h"
#include "elliptic1.h" //nur für key
#include "thrower.h"
#include "AskDialog.h"

#include <stdio.h>
#if (SIZEOF_UNSIGNED_INT -2)
    #include "jh_ref_a64.h"      //for 64 bit systems
#else
    #include "jh_ref_a32.h"      //for 32 bit systems
#endif
//#include "threefishApi.h"
#include "skein.h"

/**********************thread classes****************/
/**************************************************/
class oneway_thr : public wxThread
    {
    public:
        oneway_thr(bool is_add, bool *retval, int blolen, char* bladwell, char* pertxt, int rounds, int spice,char fr)
            : wxThread(wxTHREAD_JOINABLE)
            {
                res=retval;
                tlen=blolen;
                tad=bladwell;
                tpad=pertxt;
                trounds=rounds;
                tspice=spice;
                tfr=fr;
                tis_add=is_add;
            }
        ~oneway_thr();

    protected:
        virtual ExitCode Entry();
        int tlen;
        char *tad;
        char *tpad;
        int trounds;
        int tspice;
        char tfr;
        bool *res;
        bool tis_add;
    };
/*******************************************************/
wxThread::ExitCode oneway_thr::Entry()
    {

        if(tis_add)
        {
           *res=bytadd_a(tlen,tad,tpad,trounds,tspice,tfr);
        }
        else
        {
           *res=bytmult_a(tlen,tad,tpad,trounds,tspice,tfr);
        }
        return (wxThread::ExitCode)0;     // success
    }

/*******************************************************/
oneway_thr::~oneway_thr()
{
    return;
}
/**************************************************/

/**************************************************/
class kdev_thr : public wxThread
    {
    public:
        kdev_thr( bool *retval, int blolen, char* bladwell, int cpa_blen, char* cpa_bl, int keylen, char* keytxt, int rounds, int algonum )

            : wxThread(wxTHREAD_JOINABLE)
            {
                res=retval;
                tblolen=blolen;
                tbladwell=bladwell;
                tcpa_blen=cpa_blen;
                tcpa_bl=cpa_bl;
                tkeylen=keylen;
                tkeytxt=keytxt;
                trounds=rounds;
                talgonum=algonum;
            }
        ~kdev_thr();

    protected:
        virtual ExitCode Entry();
        int tblolen;
        char *tbladwell;
        int tcpa_blen;
        char *tcpa_bl;
        int tkeylen;
        char *tkeytxt;
        int trounds;
        int talgonum;
        bool *res;
    };
/*******************************************************/
wxThread::ExitCode kdev_thr::Entry()
    {

        *res=k_develop_f4(tblolen, tbladwell, tcpa_blen, tcpa_bl, tkeylen, tkeytxt,  trounds, talgonum );

        return (wxThread::ExitCode)0;     // success
    }

/*******************************************************/
kdev_thr::~kdev_thr()
{
    return;
}
/**************************************************/


/**************************************************/
class badd_thr : public wxThread
    {
    public:
        badd_thr(bool *retval, int blolen, char* bladwell, char* pertxt, int rounds, int spice,char fr)
            : wxThread(wxTHREAD_JOINABLE)
            {
                res=retval;
                tlen=blolen;
                tad=bladwell;
                tpad=pertxt;
                trounds=rounds;
                tspice=spice;
                tfr=fr;
            }
        ~badd_thr();

    protected:
        virtual ExitCode Entry();
        int tlen;
        char *tad;
        char *tpad;
        int trounds;
        int tspice;
        char tfr;
        bool *res;
    };
/*******************************************************/
wxThread::ExitCode badd_thr::Entry()
    {

        *res=bytadd_a(tlen,tad,tpad,trounds,tspice,tfr);

        return (wxThread::ExitCode)0;     // success
    }

/*******************************************************/
badd_thr::~badd_thr()
{
    return;
}
/**************************************************/
/**************************************************/
class bmul_thr : public wxThread
    {
    public:
        bmul_thr(bool *retval, int blolen, char* bladwell, char* pertxt, int rounds, int spice,char fr)
            : wxThread(wxTHREAD_JOINABLE)
            {
                res=retval;
                tlen=blolen;
                tad=bladwell;
                tpad=pertxt;
                trounds=rounds;
                tspice=spice;
                tfr=fr;
            }
        ~bmul_thr();

    protected:
        virtual ExitCode Entry();
        int tlen;
        char *tad;
        char *tpad;
        int trounds;
        int tspice;
        char tfr;
        bool *res;
    };
/*******************************************************/
wxThread::ExitCode bmul_thr::Entry()
    {

        *res=bytmult_a(tlen,tad,tpad,trounds,tspice,tfr);

        return (wxThread::ExitCode)0;     // success
    }

/*******************************************************/
bmul_thr::~bmul_thr()
{
    return;
}
/************************************************/
/**************** memory block assignement for reuse in develop_x***********************/
/********************************************************************/
memblo::memblo(int size_of_blocks, int number_of_blocks)
{
    int i,j;
    bool snd=true;

    no_of_mb=number_of_blocks;
    blono= size_of_blocks;

    ppmb= (char **)malloc(no_of_mb * sizeof(char*));
    if(ppmb==NULL)
       {
          throwout(_("memblo assignement failed(1)"));
          return;
       }
    for(i=0;i<no_of_mb;i++)
    {
        *(ppmb+i) = (char *) malloc(blono*sizeof(char));
        if( *(ppmb+i) == NULL)  //error
        {
            throwout(_("memblo assignement failed(2)"));
            for(j=0;j<i;j++)
            {
                free( *(ppmb+j) );
            }
            free(ppmb);
            ppmb=NULL;
            snd=false;
            break;
        }
        if(!snd) break;
    }
    if(!snd){is_sound=false; return;}
    is_sound = true;
    return;
}
/********************************************************************/
memblo::~memblo()
{
    int j;

    for(j=0;j<no_of_mb;j++)
    {
        free( *(ppmb+j) );
    }
    free(ppmb);
    ppmb=NULL;
    is_sound=false;
    return;
}
/********************************************************************/
char* memblo::get_mb(int indx_of_block)
{
    if(!is_sound||(indx_of_block>=no_of_mb))
    {
        throwout(_("illegal request in memblo"));
        return NULL;
    }
    return( *(ppmb + indx_of_block) );
}
/********************************************************************/
bool memblo::reassgn(int size_of_blocks, int number_of_blocks)
{
    int i,j;
    bool snd=true;

    //first free everything
    for(j=0;j<no_of_mb;j++)
    {
        free( *(ppmb+j) );
    }
    ppmb=(char **)realloc(ppmb,number_of_blocks * sizeof( char *));
    if(ppmb==NULL)
       {
          throwout(_("memblo reassignement failed(1)"));
          is_sound=false;
          return false;
       }
    for(i=0;i<no_of_mb;i++)
    {
        *(ppmb+i) = (char *) malloc(blono*sizeof(char));
        if( *(ppmb+i) == NULL)  //error
        {
            throwout(_("memblo assignement failed(2)"));
            for(j=0;j<i;j++)
            {
                free( *(ppmb+j) );
            }
            free(ppmb);
            ppmb=NULL;
            snd=false;
            break;
        }
        if(!snd) break;
    }
    if(!snd) return false;
    is_sound = true;
    return true;
}
/********************************************************************/

/*********** security hilfsfuns ***************/
/**********************************************/
bool arm()
{
    int retval;

    if(wxFileExists(sec_path + _("/paphr_hash.flea")))  //load zugangsdialog
    {
        mylogin dialog(NULL);
        retval=dialog.ShowModal();
        if(retval!=1) { return false;}
        is_armed=true;
        lastarmed=time(NULL);
        return true;
    }
    return false;
}
/**************************************************/
bool unarm()
{
    el_priv d2;
    wxString dummy=_("deadbeef15badf00d");
    //overwrite zugang and current private key
    zugang.wipezero();
    zugang.storhex(&dummy);
    elky.d.wipezero();
    elky.copy_priv_ky(&d2);
    //set indicators
    is_armed=false;
    return true;
}
/****************************************************/
/**************************************************/
bool checkarm()
{
    ulong32 jetzt;

    jetzt=time(NULL);
    if((jetzt - lastarmed) > armed_secs)
    {
        unarm();
        throwout(_("Gesichert wegen langer Inaktivitaet"),5);
        if(!arm())
        {
            to_quit= true; // set flag for closing Program
            return false;
        }
        else
        {
            to_quit=false;
        }
    }
    else if(!is_armed)
        {
            throwout(_("Gesichert wegen langer Inaktivitaet\nTu das nicht wieder! \nAcademic Signature muss am Ende immer korrekt geschlossen werden ;-)"));
            if(!arm())
            {
                to_quit= true; // set flag for closing Program
                return false;
            }
        }
    lastarmed  =  time(NULL);
    return true;
}
/***************************************************************/
//meine anderen Hilfsfuns
/**************************************************************/
int throwout(wxString news)
{
 if(thr_wx) wxMessageBox(news, _("News:"),wxOK|wxCENTRE|wxSTAY_ON_TOP);
 else
 {
     printf("%s",(const char *)(news.mb_str(wxConvLocal)));
 }
  return 1;
}
/********************************************/
/**************************************************************/
int throwout(wxString news,int seconds)
{

  if(thr_wx)
  {
      thrower dialog(NULL,news,seconds);
        return(dialog.ShowModal());
  }
  else
  {
     printf("%s",(const char *)(news.mb_str(wxConvLocal)));
  }
  return 1;
}
/********************************************/

long double factorial(ulong32 n) //Fakultät
{
    ulong32 i;
    long double r=1.0;

    for(i=1;i<n+1;i++)
     r *= (long double)i;

     return r;
}
/**************************************************************/
 inline unsigned char rotbyte(unsigned char byt, unsigned char rotval)
{
  //return ((byt >> rotval&7) | (byt << (8 - rotval&7)));
   unsigned char ch;

   if(rotval==0) return byt;
   ch=byt;
   byt= byt>>rotval;
   byt= byt | (ch<<(8-rotval));
   return byt;
}
/**************************************************/
/**************************************************************/
inline ulong32 rot32(ulong32 val, unsigned char rotval)
{
  //return ((val >> rotval&31) | (val << (32 - rotval&31)));
   ulong32 ch;

   if(rotval==0) return val;
   ch=val;
   val= val>>rotval;
   val= val | (ch<<(32-rotval));
   return val;
}
/**************************************************/
void reverse_wxstring(wxString *strptr)    //dreht die Zeichenfolge um
{
   int i,len;
   char c;

   len=strptr->Len();
   for (i=0;i< len/2;i++)
   {
       c= strptr->GetChar(len-i-1); //buffer
       strptr->SetChar(len-i-1, strptr->GetChar(i));  //swap
       strptr->SetChar(i, c);
   }
   return;
}
/**************************************************//********************************************/
//Routinen der Hilfsklasse rsakey:
rsakey::rsakey() //default constructor
{
    kid=_("default");
    rname= _("Jon Doe");
    type=0;
    blosiz=0;
    e.storlong(0); d.storlong(0); n.storlong(0);
    return;
}
/********************************************/
rsakey::rsakey(wxString id, wxString name, unsigned int blsiz, longnumber *ei, longnumber *ni) //default external key constructor
{
    kid=id;
    rname= name;
    type=1;
    blosiz=blsiz;
    e.copynum(ei); n.copynum(ni);
    d.storlong(0);
    return;
}
/********************************************/
rsakey::rsakey(wxString id, wxString name, unsigned int blsiz, longnumber *ei, longnumber *ni,longnumber *di) //default external key constructor
{
    kid=id;
    rname= name;
    blosiz=blsiz;
    e.copynum(ei); n.copynum(ni);
    d.copynum(di);
    if(blsiz==0)  blosiz = blsiz_check();
    if(selftest()) type=2;
        else type = 0;
    return;
}
/********************************************/
rsakey::rsakey(wxString id, wxString name, unsigned int blsiz, longnumber *ei, longnumber *pp, longnumber *pq,longnumber *di) //external key constructor
{
    kid=id;
    rname= name;
    blosiz=blsiz;
    e.copynum(ei); n.copynum(pp);n.multnum(pq);
    p.copynum(pp); q.copynum(pq);
    d.copynum(di);
    if(blsiz==0)  blosiz = blsiz_check();
    if(selftest_3()) type=3; // p q version vorhanden und funktionell
    else{
       if(selftest()) type=2; //nur n-version vorhanden
       else type = 0;  // kein proivater RSA-key
    }

    return;
}
/********************************************/
bool rsakey::selftest_3()
{
    longnumber test1,test2;
    // kleiner Test
    test1.storlong(123456789);
    test2.copynum(&test1);
    test1.q_pownum(&e, &p, &q); // Ciphertext erzeugen
    test1.q_pownum(&d, &p, &q); // zurückwandeln
    if(test1.compare(&test2)==0) return true; // funktioniert
    else return false; //funktioniert nicht
}
/********************************************/
bool rsakey::selftest()
{
    longnumber test1,test2;
    // kleiner Test
    test1.storlong(123456789);
    test2.copynum(&test1);
    test1.pownum_q(&e, &n); // Ciphertext erzeugen
    test1.pownum_q(&d, &n); // zurückwandeln
    if(test1.compare(&test2)==0) return true; // funktioniert
    else return false; //funktioniert nicht
}
/********************************************/
unsigned int rsakey::blsiz_check() // gibt maximale blocksize zurück
{
    ulong32 laenge;
    unsigned int hs;
    //bytes in n checken
    laenge= n.setsize(true) - 1; // setsize setzt  aktuelle size in bytes und gibt auch size zurück
    if(laenge > 65535) laenge = 65535;
    hs = (unsigned int) laenge;
    if( blosiz> hs) {
        throwout(_("blosiz was defective \n and had to be resized!"));
        blosiz = hs;
        return hs;
    }
    return hs; // gib optimale blosiz zurück, caller decides what to do
}
/********************************************/
rsakey::~rsakey()  // normaler destructor, nothing to do? no mallocs!
{
    return;
}
/********************************************/
void rsakey::copykey(rsakey *well)
{
    kid= well->kid;
    rname= well->rname;
    blosiz=well->blosiz;
    type=well->type;
    e.copynum(&(well->e)); n.copynum(&(well->n));
    p.copynum(&(well->p)); q.copynum(&(well->q));
    d.copynum(&(well->d));
    return;
}
/********************************************/
bool rsakey::exportpublicpart()
{
    wxString exnam,txe,txn;
    FILE *exfilp;

    if(type < 1)
    {
        throwout(_("Key not verfied, cannot export!"));
    }

    exnam= kid;
    exnam += _(".r_exp"); //filenamen setzen
    if((exfilp = fopen((const char*)exnam.mb_str(wxConvLocal), "w"))== NULL)
    {
        throwout(_("File could not be opened for writing 8-O "));
        return false;
    }

    fprintf(exfilp, "id: %s  Name: %s  \n",(const char*)(kid).mb_str(wxConvUTF8),
                  (const char*)(rname).mb_str(wxConvUTF8));
    e.writechar(&txe);
    n.writechar(&txn);

    fprintf(exfilp," e: %s  n: %s ",(const char*)txe.mb_str(wxConvUTF8), (const char*)txn.mb_str(wxConvUTF8));
    fclose(exfilp);
    return true;
}
/********************************************/
// ende meine Hilfsfuns
bool erasieve(int possprim)
{
  int i=2;

  if(possprim==2)return true;
  if(possprim==3)return true;
  if(possprim==4)return false;
  if(possprim==5)return true;
  if(possprim==7)return true;
  if(possprim==11)return true;
  while(i*i <=possprim)
  {
     if(erasieve(i))
     {
        if(possprim % i == 0) return FALSE;
     }
     i++;
  }
  return TRUE;
}
/**************************************************************/
double foxes(double a, double xold)
{
   return ((1-xold)*a*xold);
}
/**************************************************************/
bool dofeistel(int bllen, int keylen, char* ptxtwell,
               char* ctxt, char* key, char flag, int kang_r, int cpa_blen, char* cpa_bl, wxString algo)
{
    unsigned int algonum;

    algonum= get_code_4_algo(algo);
    if(algonum>1000) return false;
    return(dofeistel_n(bllen, keylen, ptxtwell, ctxt, key, flag, kang_r, cpa_blen, cpa_bl, algonum) );
}
/**************************************************************/
bool dofeistel_n(int bllen, int keylen, char* ptxtwell,
               char* ctxt, char* key, char flag, int kang_r, int cpa_blen, char* cpa_bl, unsigned int algoflag)
{
  int hlen,i,j;
  char *ptxt;

  if(bllen<2){throwout(_("block zu klein")); return false;}
  if((bllen&1) == 1)
  {throwout(_("ERROR\nblocklen muss gerade Zahl sein!!!\n schneide ab!!"));
   bllen=bllen-1;}

  ptxt= (char*)malloc(bllen);
  //for(i=0;i<bllen;i++)   //Kopie ziehen
     //*(ptxt+i)= *(ptxtwell+i);
  memcpy(ptxt,ptxtwell,bllen);
  hlen=bllen/2;

  for(j=0;j<flag;j++)   //encipher
  {
    for(i=0;i<hlen;i++)  // Li = R(i-1)
    {
      *(ctxt+i)=*(ptxt+hlen+i);
      *(ctxt+i+hlen)=*(ptxt+hlen+i);//vorbereitg für unten
    }

             // Ri = L(i-1) XOR roundfun(R(i-1),Key)

    if(!roundfun(hlen,keylen,(ctxt+hlen),key,kang_r, cpa_blen, cpa_bl,algoflag))
          {throwout(_("roundfun ERROR!"));free((void *)ptxt); return false;}
    for(i=0;i<hlen;i++)  // Ri = L(i-1) XOR roundfun(R(i-1),Key)
    {
      *(ctxt+hlen+i) ^= *(ptxt+i);
    }
    for(i=0;i<bllen;i++)
    {
      *(ptxt+i)=*(ctxt+i);
    }
  }
  for(j=0;j<-flag;j++)  //decipher
  {
    for(i=0;i<hlen;i++)  // ((Li =)) R(i-1) =Li
    {
      *(ctxt+i+hlen)=*(ptxt+i);
      *(ctxt+i)=*(ptxt+i);//vorbereitg für unten
    }

             // ((Ri =))) L(i-1) XOR roundfun(R(i-1),Key)

    if(!roundfun(hlen,keylen,ctxt,key, kang_r, cpa_blen, cpa_bl, algoflag))
        {throwout(_("roundfun ERROR!"));free((void *)ptxt);return false;}
    for(i=0;i<hlen;i++)  // ((Ri =)) L(i-1) XOR roundfun(R(i-1),Key)
    {
      *(ctxt+i) ^= *(ptxt+hlen+i);
    }
    for(i=0;i<bllen;i++)
    {
      *(ptxt+i)=*(ctxt+i);
    }
  }
  if(ptxt != NULL) free((void *) ptxt);
  return true;
}
/**************************************************************/
/**************************************************************/
float analyzbytejump(int blolen,int inbit, int outbit,
                     int tries, float *inave, float *outave)
{
  return 3;
}
/**************************************************************/
bool comblock(int blolen, char* blad, int rounds)

{
  unsigned int ind, hind,oi;
  unsigned char h1,h2;

  if(ispowof2(blolen)) return(comblock_128(blolen,blad,rounds));

  if(blolen<2)return false;
    h1=(unsigned char) *(blad);  h2=(unsigned char) *(blad+1);
    ind=1;
      for(oi=0;oi<(unsigned int)rounds*(unsigned int)blolen;oi++)
      {
          hind = ((unsigned int) *(blad+((oi+ind)%blolen)))  %blolen; // pointed-to index bestimmen
          //nicht sicher rueckabwickelbare entscheidung
          if( ((*(blad + hind) >> (oi&7))&1) ==1)
          {
              h2+=*(blad+hind);
              *(blad+hind)= rotbyte(*(blad+hind),h1&7);
          }
          else
          {
              h1+= *(blad+hind);
              *(blad+hind) =rotbyte(*(blad+hind),h2&7);
          }
          ind = hind;
      }
  return true;
}
/*********************************************/
bool comblock_128(int blolen, char* bladwell, int rounds)
//quickversion fuer blolen=powof 2

{
  unsigned int ind,oi,hind,mask;
  unsigned char h1,h2;

  if(blolen<2)return false;
  mask = blolen-1;

    h1=(unsigned char) *(bladwell);  h2=(unsigned char) *(bladwell+1);
    ind=1;
      for(oi=0;oi<rounds*(unsigned int)blolen;oi++) //combination lock turning
      {
          hind = ((int) *(bladwell+((oi+ind)&mask)))  &mask; // pointed-to index bestimmen
          //nicht sicher rueckabwickelbare entscheidung
          if( ((*(bladwell + hind) >> (oi&7))&1) ==1)
          {
              h2+=*(bladwell+hind);
              *(bladwell+hind)= rotbyte(*(bladwell+hind),h1&7);
          }
          else
          {
              h1+= *(bladwell+hind);
              *(bladwell+hind) =rotbyte(*(bladwell+hind),h2&7);
          }
          ind = hind;
      }
  return true;
}
/************************************************************************/
bool bytxor_128(int blolen, char* bladwell, char* pertxt, int rounds)
//quickversion fuer blolen=powof 2

{
  unsigned int ind,oi,hind,mask;
  unsigned char *blad;
  int k;

  if(blolen<2)return false;
  mask = blolen-1;

  blad=(unsigned char*)malloc(blolen);

  memcpy(blad,bladwell,blolen);

  for(k=0;k<rounds;k++)
  {
      for(oi=0;oi<(unsigned int)blolen;oi++) //erste schleife systematisch
      {
          hind = (unsigned int)   *(blad +(( *(blad + oi) * *(blad+((oi+1)&mask))) &mask) )&mask; // pointed-to index bestimmen
          if(hind!=oi) *(blad + oi) += rotbyte(*(blad+hind),((unsigned char)(hind+oi)&7));
          *(blad+oi)+=63+(unsigned char)oi;
          if(oi%2 == 0) *(blad+hind) = *(blad+hind) ^255; //jedes zweite mal bei hind invertieren
      }
      ind=0;
      for(oi=0;oi<(unsigned int)blolen;oi++) //zweite schleife wild gesprungwen
      {
          hind = (unsigned int)*(blad+ind)&mask; // pointed-to index bestimmen
          if(hind!= ind) *(blad + ind) += rotbyte(*(blad+hind),(unsigned char)(ind+hind)&7); //wird mit zeigerziel ge xort
          *(blad+ind)+=63+(unsigned char)oi;//nullenkiller
          if(oi%2 == 1) *(blad+hind) = *(blad+hind) ^255; //jedes zweite mal bei hind invertieren
          ind = (ind +hind+oi)&mask; //neuer index wo# s hinzeigt plus 1 modulo blono
     }
  }
  memcpy(pertxt,blad,blolen);
  if(blad != NULL) free( blad);
  return true;
}
/*********************************************//*********************************************/
bool bytxor(int blolen, char* bladwell, char* pertxt, int rounds)

//neue Basisvariante ohne Combination Lock

{
  int ind,oi,k,hind;
  unsigned char *blad;

  if(ispowof2(blolen)) return(bytxor_128(blolen,bladwell,pertxt, rounds));

  if(blolen<2)return false;

  blad=(unsigned char*)malloc(blolen);

  memcpy(blad,bladwell,blolen);

  for(k=0;k<rounds;k++)
  {
      for(oi=0;oi<blolen;oi++) //erste schleife systematisch
      {
          hind = (ulong32)   *(blad +( *(blad + oi) * *(blad+(oi+1)%blolen)) %blolen )%blolen; // pointed-to index bestimmen
          if(hind!=oi) *(blad + oi) += rotbyte(*(blad+hind),((unsigned char)(hind+oi)&7));
          *(blad+oi)+=63+(unsigned char)oi;
          if(oi%2 == 0) *(blad+hind) = *(blad+hind) ^255; //jedes zweite mal bei hind invertieren
      }
      ind=0;
      for(oi=0;oi<blolen;oi++) //zweite schleife wild gesprungwen
      {
          hind = (ulong32)*(blad+ind)%blolen; // pointed-to index bestimmen
          if(hind!= ind) *(blad + ind) += rotbyte(*(blad+hind),(unsigned char)(ind+hind)&7); //wird mit zeigerziel ge xort
          *(blad+ind)+=63+(unsigned char)oi;//nullenkiller
          if(oi%2 == 1) *(blad+hind) = *(blad+hind) ^255; //jedes zweite mal bei hind invertieren
          ind = (ind +hind+oi)%blolen; //neuer index wo's hinzeigt plus 1 modulo blono
     }
  }
  memcpy(pertxt,blad,blolen);
  if(blad != NULL) free( blad);
  return true;
}
/*********************************************/
/************************************************************************/
bool bytadd_128(int blolen, char* bladwell, char* pertxt, int rounds)
//quickversion fuer blolen=powof 2

{
  unsigned int ind,oi,hind,mask,magic=0;
  unsigned char *blad;
  int k;

  if(blolen<2)return false;
  mask = blolen-1;

  blad=(unsigned char*)malloc(blolen);

  memcpy(blad,bladwell,blolen);

  for(k=0;k<rounds;k++)
  {
      for(oi=0;oi<(unsigned int)blolen;oi++) //erste schleife systematisch
      {
          hind = (unsigned int)   *(blad +(((unsigned int) *(blad + oi))&mask)  );
          hind *= *(blad+((oi+1)&mask)); // pointed-to index bestimmen
          hind &= mask;
          if(hind!=oi) *(blad + oi) += rotbyte(*(blad+hind),((unsigned char)(hind+oi)&7));
          *(blad+oi)+=63+(unsigned char)oi;
          if(!(oi&2)) *(blad+hind) = *(blad+hind) ^255; //jedes zweite mal bei hind invertieren
      }
      ind=blolen/2;
      for(oi=0;oi<(unsigned int)blolen;oi++) //zweite schleife wild gesprungwen
      {
          hind = (unsigned int)*(blad+ind)&mask; // pointed-to index bestimmen
          hind +=magic;
          magic += *(blad + oi);
          hind &=mask;
          if(hind!= ind) *(blad + ind) += rotbyte(*(blad+hind),(unsigned char)(ind+hind)&7); //wird mit zeigerziel ge xort
          *(blad+ind)+=63+(unsigned char)oi;//nullenkiller
          if(oi&2) *(blad+hind) = *(blad+hind) ^255; //jedes zweite mal bei hind invertieren
          ind = (ind +hind+oi)&mask; //neuer index wo# s hinzeigt plus 1 modulo blono
     }
  }
  memcpy(pertxt,blad,blolen);
  if(blad != NULL) free( blad);
  return true;
}
/*********************************************//*********************************************/
bool bytadd(int blolen, char* bladwell, char* pertxt, int rounds)

//neue Basisvariante ohne Combination Lock

{
  unsigned int ind,oi,k,hind;
  unsigned char *blad;
  unsigned int magic=0;

  if(ispowof2(blolen)) return(bytadd_128(blolen,bladwell,pertxt, rounds));

  if(blolen<2)return false;

  blad=(unsigned char*)malloc(blolen);

  memcpy(blad,bladwell,blolen);

  for(k=0;k<(unsigned int)rounds;k++)
  {
      for(oi=0;oi<(unsigned int)blolen;oi++) //erste schleife systematisch
      {
          hind = (unsigned int)   *(blad +(( *(blad + oi))%blolen));
          hind *= *(blad+((oi+1)%blolen)); // pointed-to index bestimmen
          hind %= blolen;
          if(hind!=oi) *(blad + oi) += rotbyte(*(blad+hind),((unsigned char)(hind+oi)&7));
          *(blad+oi)+=63+(unsigned char)oi;
          if(!(oi&2)) *(blad+hind) = *(blad+hind) ^255; //jedes zweite mal bei hind invertieren
      }
      ind=blolen/2;
      for(oi=0;oi<(unsigned int)blolen;oi++) //zweite schleife wild gesprungwen
      {
          hind = (unsigned int)*(blad+ind)%blolen; // pointed-to index bestimmen
          hind +=magic;

          magic += *(blad + oi);
          hind %= blolen;
          if(hind!= ind) *(blad + ind) += rotbyte(*(blad+hind),(unsigned char)(ind+hind)&7); //wird mit zeigerziel ge xort
          *(blad+ind)+=63+(unsigned char)oi;//nullenkiller
          if(oi&2) *(blad+hind) = *(blad+hind) ^255; //jedes zweite mal bei hind invertieren
          ind = (ind +hind+oi)%blolen; //neuer index wo# s hinzeigt plus 1 modulo blono
     }
  }
  memcpy(pertxt,blad,blolen);
  if(blad != NULL) free( blad);
  return true;
}
/*************************************/
bool dbytadd_a_l(ulong32 blolen, char* bladwell,  char* pertxt, int rounds, int spice, double fr)

//for powers of two
//spice is individualizer(def 1 set in header),fr is flea loop multiplier(def 3 set in header)

{
  ulong32 ind,oi,k,hind,thresh3;
  unsigned char *blad;
  ulong32 magic=0;

  if(blolen<2)return false;
  if(fr<0||fr>10)fr=3; //default is 3 anyways

  //if(ispowof2(blolen)) return dbytadd_a_128( blolen, bladwell, pertxt, rounds, spice, fr);

  blad=(unsigned char*)malloc(blolen);

  memcpy(blad,bladwell,blolen);

  for(k=0;k<(unsigned int)rounds;k++)
  {
      for(oi=0;oi<blolen;oi++) //systematically working up
      {
          hind = (ulong32)   *(blad +(( *(blad + oi))%blolen));  //hind ist ein byte
          hind+=(unsigned int)spice; //allow for differently spiced variants
          hind *= *(blad+((oi+1)%blolen)); // determine target index, hind is jetzt 2 byte groß
          hind++; //remove even-odd bias
          hind *= *(blad+((oi+3)%blolen)); // determine target index, hind is jetzt 3 byte groß
          hind++;
          hind *= *(blad+((oi+7)%blolen)); // determine target index, hind is jetzt 4 byte groß
          hind++;
          hind %= blolen; //points to anywhere in block
          if(hind!=oi) *(blad + oi) += rotbyte(*(blad+hind),((unsigned char)(hind+oi)&7));
          *(blad+oi)+=(unsigned char)spice+(unsigned char)oi;
          if(!(oi&2)) *(blad+hind) = *(blad+hind) ^255; //entropizer step
      }
      for(oi=blolen-1;oi>0;oi--) //systematically working down
      {
          hind = (unsigned int)   *(blad +(( *(blad + oi))%blolen));
          hind+=(unsigned int)spice; //allow for differently spiced variants
          hind *= *(blad+((oi+1)%blolen)); // determine target index
          hind++;
          hind *= *(blad+((oi+2)%blolen)); // determine target index, hind is jetzt 3 byte groß
          hind++;
          hind *= *(blad+((oi+6)%blolen)); // determine target index, hind is jetzt 4 byte groß
          hind++;
          hind %= blolen;
          if(hind!=oi) *(blad + oi) += rotbyte(*(blad+hind),((unsigned char)(hind+oi)&7));
          *(blad+oi)+=(unsigned char)spice+(unsigned char)oi;
          if(!(oi&2)) *(blad+hind) = *(blad+hind) ^255; //entropizer step
      }
      ind=blolen/2;
      thresh3=((ulong32)(fr*(double)blolen));
      for(oi=0;oi<thresh3;oi++) //second loop statistically jumping
      {
          hind = (ulong32)*(blad+ind); // determine target index, 1 byte long
          hind +=magic; //might be a little more than 1 byte now
          magic += *(blad + oi%blolen) +(unsigned int)spice;
          hind *= (ulong32)*(blad+(ind+2)%blolen); //now more than 2 bytes
          hind++;
          hind *= (ulong32)*(blad+(ind+5)%blolen); //now more than 3 bytes
          hind++;
          hind *= (ulong32)*(blad+(ind+7)%blolen); //now more than 4 bytes, possibly reentrant
          hind++;
          hind %= blolen;
          if(hind!= ind) *(blad + ind) += rotbyte(*(blad+hind),(unsigned char)(ind+hind)&7); //add rotated target byte
          *(blad+ind)+=spice+(unsigned char)oi;//kill zeros
          if(oi&2) *(blad+hind) = *(blad+hind) ^255; //entropizer
          ind = (ind +hind+oi+spice)%blolen; //reset entry index, expression possibly reentrant
     }
  }
  memcpy(pertxt,blad,blolen);
  if(blad != NULL) free( blad);
  return true;
}
/*************************************/
bool dbytadd_a(int blolen, char* bladwell,  char* pertxt, int rounds, int spice, double fr)

//for powers of two
//spice is individualizer(def 1 set in header),fr is flea loop multiplier(def 3 set in header)

{
  unsigned int ind,oi,k,hind,thresh3;
  unsigned char *blad;
  unsigned int magic=0;

  if(blolen<2)return false;
  if(fr<0||fr>10)fr=3; //default is 3 anyways

  if(ispowof2(blolen)) return dbytadd_a_128( blolen, bladwell, pertxt, rounds, spice, fr);

  blad=(unsigned char*)malloc(blolen);

  memcpy(blad,bladwell,blolen);

  for(k=0;k<(unsigned int)rounds;k++)
  {
      for(oi=0;oi<(unsigned int)blolen;oi++) //systematically working up
      {
          hind = (unsigned int)   *(blad +(( *(blad + oi))%blolen));
          hind+=(unsigned int)spice; //allow for differently spiced variants
          hind *= *(blad+((oi+1)%blolen)); // determine target index
          hind %= blolen;
          if(hind!=oi) *(blad + oi) += rotbyte(*(blad+hind),((unsigned char)(hind+oi)&7));
          *(blad+oi)+=(unsigned char)spice+(unsigned char)oi;
          if(!(oi&2)) *(blad+hind) = *(blad+hind) ^255; //entropizer step
      }
      for(oi=(unsigned int)blolen-1;oi>0;oi--) //systematically working down
      {
          hind = (unsigned int)   *(blad +(( *(blad + oi))%blolen));
          hind+=(unsigned int)spice; //allow for differently spiced variants
          hind *= *(blad+((oi+1)%blolen)); // determine target index
          hind %= blolen;
          if(hind!=oi) *(blad + oi) += rotbyte(*(blad+hind),((unsigned char)(hind+oi)&7));
          *(blad+oi)+=(unsigned char)spice+(unsigned char)oi;
          if(!(oi&2)) *(blad+hind) = *(blad+hind) ^255; //entropizer step
      }
      ind=blolen/2;
      thresh3=((unsigned int)(fr*(double)blolen));
      for(oi=0;oi<thresh3;oi++) //second loop statistically jumping
      {
          hind = (unsigned int)*(blad+ind)%blolen; // determine target index
          hind +=magic;

          magic += *(blad + oi%blolen) +(unsigned int)spice;
          hind %= blolen;
          if(hind!= ind) *(blad + ind) += rotbyte(*(blad+hind),(unsigned char)(ind+hind)&7); //add rotated target byte
          *(blad+ind)+=spice+(unsigned char)oi;//kill zeros
          if(oi&2) *(blad+hind) = *(blad+hind) ^255; //entropizer
          ind = (ind +hind+oi+spice)%blolen; //reset entry index
     }
  }
  memcpy(pertxt,blad,blolen);
  if(blad != NULL) free( blad);
  return true;
}

/*************************************/
bool dbytadd_ax(int blolen, char* bladwell,  char* pertxt, int rounds, int spice, double fr)

//for powers of two
//spice is individualizer(def 1 set in header),fr is flea loop multiplier(def 3 set in header)

{
  unsigned int oi,k,thresh3,blolenx,modlen;
  ulong32 hind;
  unsigned char *blad;
  ulong32 magic=0;
  unsigned short shind,ind;

  if(blolen<4)return false;
  if(fr<0||fr>10)fr=3; //default is 3 anyways
  if(blolen&3) blolenx=(unsigned int) (blolen - (blolen&3) + 4); //using access as integer requires integer frame
  else blolenx=blolen;
  modlen=blolenx-4;
  blad=(unsigned char*)malloc(blolenx);
  memcpy(blad,bladwell,blolen);
  for(oi=(unsigned int)blolen;oi<blolenx;oi++)  //set possible additional bytes to zero
  {
      *(blad+oi)=0;
  }

  for(k=0;k<(unsigned int)rounds;k++)
  {
      for(oi=0;oi<(unsigned int)blolen-3;oi+=4) //systematically working up upper -3
      {
          hind =  *((ulong32*)  (blad +( *((unsigned short*)(blad + oi))%modlen) ));
          hind+=(ulong32)spice; //allow for differently spiced variants
          hind *= *(blad+(((unsigned short)oi+1)%modlen)); // determine target index
          shind = (unsigned short) hind;
          shind %= modlen;
          if(shind!=(unsigned short)oi) *((ulong32*)(blad + oi)) += rot32(*((ulong32*)(blad+shind)),((unsigned char)(shind+oi)&31));
          *((ulong32*)(blad+oi))+=(ulong32)spice+oi+hind;
          if(!(oi&2)) *((ulong32*)(blad+shind)) = *((ulong32*)(blad+shind)) ^0xffffffff; //entropizer step
      }
      //do highest block
      {
          oi=blolen-4;
          hind =  *((ulong32*)  (blad +( *((unsigned short*)(blad + oi))%modlen) ));
          hind+=(ulong32)spice; //allow for differently spiced variants
          hind *= *(blad+(((unsigned short)oi+1)%modlen)); // determine target index
          shind=(unsigned short) hind;
          shind %= modlen;
          if(shind!=(unsigned short)oi) *((ulong32*)(blad + oi)) += rot32(*((ulong32*)(blad+shind)),((unsigned char)(hind+oi)&31));
          *((ulong32*)(blad+oi))+=(ulong32)spice+oi+hind;
          if(!(oi&2)) *((ulong32*)(blad+shind)) = *((ulong32*)(blad+shind)) ^0xffffffff; //entropizer step
      }
      for(oi=(unsigned int)blolen-4;oi>3;oi-=4) //systematically working down
      {
          hind = *((ulong32*)   (blad +( *((unsigned short*) (blad + oi))%modlen)));
          hind+=(ulong32)spice; //allow for differently spiced variants
          hind *= *(blad+(((unsigned short)oi+1)%modlen)); // determine target index
          shind=(unsigned short) hind;
          shind %= modlen;
          if(shind!=oi) *((ulong32*)(blad + oi)) += rot32(*((ulong32*)(blad+shind)),((unsigned char)(hind+oi)&31));
          *((ulong32*)(blad+oi))+=(ulong32)spice+(ulong32)oi+hind;
          if(!(oi&2)) *((ulong32*)(blad+shind)) = *((ulong32*)(blad+shind)) ^0xffffffff; //entropizer step
      }
      //do lowest block
      {
          hind = *((ulong32*)   (blad +( *((unsigned short*) (blad))%modlen)));
          hind+=(ulong32)spice; //allow for differently spiced variants
          hind *= *(blad+1); // determine target index
          shind=(unsigned short) hind;
          shind %= modlen;
          if(shind!=0) *((ulong32*)(blad)) += rot32(*((ulong32*)(blad+shind)),((unsigned char)(hind)&31));
          *((ulong32*)(blad))+=(ulong32)spice+hind;
      }
      ind=modlen/2;
      thresh3=((unsigned int)(fr*(double)blolen))/2;
      for(oi=0;oi<thresh3;oi++) //third loop statistically jumping 4byte chunks
      {
          hind =  *((ulong32*)  (blad +( *((unsigned short*)(blad + ind))%modlen) )); // determine target index
          hind +=magic;
          shind=(unsigned short) hind;
          shind %= modlen;
          magic += *((ulong32*)(blad + oi%modlen)) +(unsigned int)spice;
          if(shind!= ind) *((ulong32*)(blad + ind)) += rot32(*((ulong32*)(blad+shind)),((unsigned char)(hind+oi)&31)); //add rotated target byte
          *((ulong32*)(blad+ind)) += (ulong32)spice+(ulong32)oi+hind;//kill zeros
          if(oi&3) *((ulong32*)(blad+shind)) = *((ulong32*)(blad+shind)) ^0xffffffff; //entropizer step
          ind = (ind +shind+(unsigned short)oi+(unsigned short)spice)%modlen; //reset entry index
     }
      for(oi=0;oi<thresh3;oi++) //fourth loop statistically jumping one byte version
      {
          shind = (unsigned short)*(blad+ind)%blolen; // determine target index
          shind +=(unsigned short)magic;

          magic += *(blad + oi%blolen) +(unsigned int)spice;
          shind %= blolen;
          if(shind!= ind) *(blad + ind) += rotbyte(*(blad+shind),(unsigned char)(ind+shind)&7); //add rotated target byte
          *(blad+ind)+=spice+(unsigned char)oi;//kill zeros
          if(oi&2) *(blad+shind) = *(blad+shind) ^0xff; //entropizer
          ind = (ind +shind+oi+spice)%blolen; //reset entry index
     }
  }
  memcpy(pertxt,blad,blolen);
  if(blad != NULL) free( blad);
  return true;
}
/*********************************************/
/*************************************/
bool dbytadd_ay(int blolen, char* bladwell,  char* pertxt, int rounds, int spice, double fr)

//for powers of two
//spice is individualizer(def 1 set in header),fr is flea loop multiplier(def 3 set in header)

{
  unsigned int oi,k,thresh3,blolenx,modlen;
  ulong32 hind;
  unsigned short ind, shind;
  unsigned char *blad;
  ulong32 magic=0;

  if(blolen<4)return false;
  if(fr<0||fr>10)fr=3; //default is 3 anyways
  //get number in 4 step
  if(blolen&3) blolenx=(unsigned int) (blolen - (blolen&3) + 4);
  else blolenx=blolen;
  modlen=blolenx-4;
  blad=(unsigned char*)malloc(blolenx);
  memcpy(blad,bladwell,blolen);
  for(oi=(unsigned int)blolen;oi<blolenx;oi++)  //set possible additional bytes to zero
  {
      *(blad+oi)=0;
  }

  //xor high32bit with low
  *((ulong32*)(blad+modlen)) ^= *((ulong32*)(blad));

  for(k=0;k<(unsigned int)rounds;k++)
  {
      for(oi=(unsigned int)blolen-4;oi>3;oi-=4) //systematically working down
      {
          hind = *((ulong32*)   (blad +( *((unsigned short*) (blad + oi))%modlen)));
          hind+=(ulong32)spice; //allow for differently spiced variants
          hind *= *(blad+(((unsigned short)oi+1)%modlen)); // determine target index
          shind=(unsigned short) hind;
          shind %= modlen;
          if(shind!=oi) *((ulong32*)(blad + oi)) += rot32(*((ulong32*)(blad+shind)),((unsigned char)(hind+oi)&31));
          *((ulong32*)(blad+oi))+=(ulong32)spice+(ulong32)oi+hind;
          if(!(oi&2)) *((ulong32*)(blad+shind)) = *((ulong32*)(blad+shind)) ^0xffffffff; //entropizer step
      }
      //do lowest block
      {
          hind = *((ulong32*)   (blad +( *((unsigned short*) (blad))%modlen)));
          hind+=(ulong32)spice; //allow for differently spiced variants
          hind *= *(blad+1); // determine target index
          shind=(unsigned short) hind;
          shind %= modlen;
          if(shind!=0) *((ulong32*)(blad)) += rot32(*((ulong32*)(blad+shind)),((unsigned char)(hind)&31));
          *((ulong32*)(blad))+=(ulong32)spice+hind;
      }
      for(oi=0;oi<(unsigned int)blolen-3;oi+=4) //systematically working up upper -3
      {
          hind =  *((ulong32*)  (blad +( *((unsigned short*)(blad + oi))%modlen) ));
          hind+=(ulong32)spice; //allow for differently spiced variants
          hind *= *(blad+(((unsigned short)oi+1)%modlen)); // determine target index
          shind = (unsigned short) hind;
          shind %= modlen;
          if(shind!=(unsigned short)oi) *((ulong32*)(blad + oi)) += rot32(*((ulong32*)(blad+shind)),((unsigned char)(shind+oi)&31));
          *((ulong32*)(blad+oi))+=(ulong32)spice+oi+hind;
          if(!(oi&2)) *((ulong32*)(blad+shind)) = *((ulong32*)(blad+shind)) ^0xffffffff; //entropizer step
      }
      //do highest block
      {
          oi=blolen-4;
          hind =  *((ulong32*)  (blad +( *((unsigned short*)(blad + oi))%modlen) ));
          hind+=(ulong32)spice; //allow for differently spiced variants
          hind *= *(blad+(((unsigned short)oi+1)%modlen)); // determine target index
          shind=(unsigned short) hind;
          shind %= modlen;
          if(shind!=(unsigned short)oi) *((ulong32*)(blad + oi)) += rot32(*((ulong32*)(blad+shind)),((unsigned char)(hind+oi)&31));
          *((ulong32*)(blad+oi))+=(ulong32)spice+oi+hind;
          if(!(oi&2)) *((ulong32*)(blad+shind)) = *((ulong32*)(blad+shind)) ^0xffffffff; //entropizer step
      }
      ind=modlen/2;
      thresh3=((unsigned int)(fr*(double)blolen))/2;
      for(oi=0;oi<thresh3;oi++) //third loop statistically jumping 4byte chunks
      {
          hind =  *((ulong32*)  (blad +( *((unsigned short*)(blad + ind))%modlen) )); // determine target index
          hind +=magic;
          shind=(unsigned short) hind;
          shind %= modlen;
          magic += *((ulong32*)(blad + oi%modlen)) +(unsigned int)spice;
          if(shind!= ind) *((ulong32*)(blad + ind)) += rot32(*((ulong32*)(blad+shind)),((unsigned char)(hind+oi)&31)); //add rotated target byte
          *((ulong32*)(blad+ind)) += (ulong32)spice+(ulong32)oi+hind;//kill zeros
          if(oi&3) *((ulong32*)(blad+shind)) = *((ulong32*)(blad+shind)) ^0xffffffff; //entropizer step
          ind = (ind +shind+(unsigned short)oi+(unsigned short)spice)%modlen; //reset entry index
     }
      for(oi=0;oi<thresh3;oi++) //fourth loop statistically jumping one byte version
      {
          shind = (unsigned short)*(blad+ind)%blolen; // determine target index
          shind +=(unsigned short)magic;

          magic += *(blad + oi%blolen) +(unsigned int)spice;
          shind %= blolen;
          if(shind!= ind) *(blad + ind) += rotbyte(*(blad+shind),(unsigned char)(ind+shind)&7); //add rotated target byte
          *(blad+ind)+=spice+(unsigned char)oi;//kill zeros
          if(oi&2) *(blad+shind) = *(blad+shind) ^0xff; //entropizer
          ind = (ind +shind+oi+spice)%blolen; //reset entry index
     }
  }
  memcpy(pertxt,blad,blolen);
  if(blad != NULL) free( blad);
  return true;
}
/*********************************************/

/*********************************************/
/*********************************************//*********************************************/
bool bytadd_a(int blolen, char* bladwell,  char* pertxt, int rounds, int spice, char fr)

//for powers of two
//spice is individualizer(def 1 set in header),fr is flea loop multiplier(def 3 set in header)

{
  unsigned int ind,oi,k,hind;
  unsigned char *blad;
  unsigned int magic=0;

  if(blolen<2)return false;
  if(fr<0||fr>10)fr=3; //default is 3 anyways

  if(ispowof2(blolen)) return bytadd_a_128( blolen, bladwell, pertxt, rounds, spice, fr);

  blad=(unsigned char*)malloc(blolen);

  memcpy(blad,bladwell,blolen);

  for(k=0;k<(unsigned int)rounds;k++)
  {
      for(oi=0;oi<(unsigned int)blolen;oi++) //systematically working up
      {
          hind = (unsigned int)   *(blad +(( *(blad + oi))%blolen));
          hind+=(unsigned int)spice; //allow for differently spiced variants
          hind *= *(blad+((oi+1)%blolen)); // determine target index
          hind %= blolen;
          if(hind!=oi) *(blad + oi) += rotbyte(*(blad+hind),((unsigned char)(hind+oi)&7));
          *(blad+oi)+=(unsigned char)spice+(unsigned char)oi;
          if(!(oi&2)) *(blad+hind) = *(blad+hind) ^255; //entropizer step
      }
      for(oi=(unsigned int)blolen-1;oi>0;oi--) //systematically working down
      {
          hind = (unsigned int)   *(blad +(( *(blad + oi))%blolen));
          hind+=(unsigned int)spice; //allow for differently spiced variants
          hind *= *(blad+((oi+1)%blolen)); // determine target index
          hind %= blolen;
          if(hind!=oi) *(blad + oi) += rotbyte(*(blad+hind),((unsigned char)(hind+oi)&7));
          *(blad+oi)+=(unsigned char)spice+(unsigned char)oi;
          if(!(oi&2)) *(blad+hind) = *(blad+hind) ^255; //entropizer step
      }
      ind=blolen/2;
      for(oi=0;oi<fr*(unsigned int)blolen;oi++) //second loop statistically jumping
      {
          hind = (unsigned int)*(blad+ind)%blolen; // determine target index
          hind +=magic;

          magic += *(blad + oi%blolen) +(unsigned int)spice;
          hind %= blolen;
          if(hind!= ind) *(blad + ind) += rotbyte(*(blad+hind),(unsigned char)(ind+hind)&7); //add rotated target byte
          *(blad+ind)+=spice+(unsigned char)oi;//kill zeros
          if(oi&2) *(blad+hind) = *(blad+hind) ^255; //entropizer
          ind = (ind +hind+oi+spice)%blolen; //reset entry index
     }
  }
  memcpy(pertxt,blad,blolen);
  if(blad != NULL) free( blad);
  return true;
}
/**********************************************************************/
bool dbytadd_a_128(int blolen, char* bladwell, char* pertxt, int rounds, int spice, double fr)

//for powers of two
//spice is individualizer(def 1 set in header),fr is flea loop multiplier(def 3 set in header)

{
  unsigned int ind,oi,k,hind,thresh3;
  unsigned char *blad;
  unsigned int magic=0;
  unsigned int mask;
  unsigned char cmask;


  //seting masks, assuming blolen is powof 2
  mask=(unsigned int) blolen - 1;
  if (blolen> 128) cmask=255;
    else cmask= (unsigned char)blolen -1;

  blad=(unsigned char*)malloc(blolen);
  memcpy(blad,bladwell,blolen);

  for(k=0;k<(unsigned int)rounds;k++)
  {
      for(oi=0;oi<(unsigned int)blolen;oi++) //systematically working up
      {
          hind = (unsigned int)   *(blad +(( *(blad + oi))&cmask));
          hind+=(unsigned int)spice; //allow for differently spiced variants
          hind *= *(blad+((oi+1)&mask)); // determine target index
          hind = hind&mask;
          if(hind!=oi) *(blad + oi) += rotbyte(*(blad+hind),((unsigned char)(hind+oi)&7));
          *(blad+oi)+=(unsigned char)spice+(unsigned char)oi;
          if(!(oi&2)) *(blad+hind) = *(blad+hind) ^255; //entropizer step
      }
      for(oi=(unsigned int)blolen-1;oi>0;oi--) //systematically working down
      {
          hind = (unsigned int)   *(blad +(( *(blad + oi))&cmask));
          hind+=(unsigned int)spice; //allow for differently spiced variants
          hind *= *(blad+((oi+1)&mask)); // determine target index
          hind = hind&mask;
          if(hind!=oi) *(blad + oi) += rotbyte(*(blad+hind),((unsigned char)(hind+oi)&7));
          *(blad+oi)+=(unsigned char)spice+(unsigned char)oi;
          if(!(oi&2)) *(blad+hind) = *(blad+hind) ^255; //entropizer step
      }
      ind=blolen/2;
      thresh3=(unsigned int)(fr*(double)blolen);
      for(oi=0;oi<thresh3;oi++) //second loop statistically jumping
      {
          hind = (unsigned int)*(blad+ind)&cmask; // determine target index
          hind +=magic;

          magic += *(blad + (oi&mask)) +(unsigned int)spice;
          hind = hind&mask;
          if(hind!= ind) *(blad + ind) += rotbyte(*(blad+hind),(unsigned char)(ind+hind)&7); //add rotated target byte
          *(blad+ind)+=spice+(unsigned char)oi;//kill zeros
          if(oi&2) *(blad+hind) = *(blad+hind) ^255; //entropizer
          ind = (ind +hind+oi+spice)&mask; //reset entry index
     }
  }
  memcpy(pertxt,blad,blolen);
  if(blad != NULL) free( blad);
  return true;
}
/*************************************/
bool bytadd_a_128(int blolen, char* bladwell, char* pertxt, int rounds, int spice, char fr)

//for powers of two
//spice is individualizer(def 1 set in header),fr is flea loop multiplier(def 3 set in header)

{
  unsigned int ind,oi,k,hind;
  unsigned char *blad;
  unsigned int magic=0;
  unsigned int mask;
  unsigned char cmask;


  //seting masks, assuming blolen is powof 2
  mask=(unsigned int) blolen - 1;
  if (blolen> 128) cmask=255;
    else cmask= (unsigned char)blolen -1;

  blad=(unsigned char*)malloc(blolen);
  memcpy(blad,bladwell,blolen);

  for(k=0;k<(unsigned int)rounds;k++)
  {
      for(oi=0;oi<(unsigned int)blolen;oi++) //systematically working up
      {
          hind = (unsigned int)   *(blad +(( *(blad + oi))&cmask));
          hind+=(unsigned int)spice; //allow for differently spiced variants
          hind *= *(blad+((oi+1)&mask)); // determine target index
          hind = hind&mask;
          if(hind!=oi) *(blad + oi) += rotbyte(*(blad+hind),((unsigned char)(hind+oi)&7));
          *(blad+oi)+=(unsigned char)spice+(unsigned char)oi;
          if(!(oi&2)) *(blad+hind) = *(blad+hind) ^255; //entropizer step
      }
      for(oi=(unsigned int)blolen-1;oi>0;oi--) //systematically working down
      {
          hind = (unsigned int)   *(blad +(( *(blad + oi))&cmask));
          hind+=(unsigned int)spice; //allow for differently spiced variants
          hind *= *(blad+((oi+1)&mask)); // determine target index
          hind = hind&mask;
          if(hind!=oi) *(blad + oi) += rotbyte(*(blad+hind),((unsigned char)(hind+oi)&7));
          *(blad+oi)+=(unsigned char)spice+(unsigned char)oi;
          if(!(oi&2)) *(blad+hind) = *(blad+hind) ^255; //entropizer step
      }
      ind=blolen/2;
      for(oi=0;oi<fr*(unsigned int)blolen;oi++) //second loop statistically jumping
      {
          hind = (unsigned int)*(blad+ind)&cmask; // determine target index
          hind +=magic;

          magic += *(blad + (oi&mask)) +(unsigned int)spice;
          hind = hind&mask;
          if(hind!= ind) *(blad + ind) += rotbyte(*(blad+hind),(unsigned char)(ind+hind)&7); //add rotated target byte
          *(blad+ind)+=spice+(unsigned char)oi;//kill zeros
          if(oi&2) *(blad+hind) = *(blad+hind) ^255; //entropizer
          ind = (ind +hind+oi+spice)&mask; //reset entry index
     }
  }
  memcpy(pertxt,blad,blolen);
  if(blad != NULL) free( blad);
  return true;
}
/*********************************************/
bool bytxor_x1_128(int blolen, char* bladwell, char* pertxt, int rounds)
//quickversion fuer blolen=powof 2

{
  unsigned int ind,oi,hind,mask;
  unsigned char *blad,h1,h2;
  int k;

  if(blolen<2)return false;
  mask = blolen-1;

  blad=(unsigned char*)malloc(blolen);

  memcpy(blad,bladwell,blolen);

  for(k=0;k<rounds;k++)
  {
      for(oi=0;oi<(unsigned int)blolen;oi++) //erste schleife systematisch
      {
          hind = (unsigned int)   *(blad +(( *(blad + oi) * *(blad+((oi+1)&mask))) &mask ))&mask; // pointed-to index bestimmen
          if(hind!=oi) *(blad + oi) += rotbyte(*(blad+hind),((unsigned char)(hind+oi)&7));
          //*(blad +oi)=rotbyte(*(blad+oi),(unsigned char)hind&7);
          *(blad+oi)+=63+(unsigned char)oi;
          if(oi%2 == 0) *(blad+hind) = *(blad+hind) ^255; //jedes zweite mal bei hind invertieren
      }
      ind=0;
      for(oi=0;oi<(unsigned int)blolen;oi++) //zweite schleife wild gesprungwen
      {
          hind = (ulong32)*(blad+ind)&mask; // pointed-to index bestimmen
         if(hind!= ind) *(blad + ind) += rotbyte(*(blad+hind),(unsigned char)(ind+hind)&7); //wird mit zeigerziel ge xort
         // *(blad +ind)=rotbyte(*(blad+ind),(unsigned char)oi&7);
          *(blad+ind)+=63+(unsigned char)oi;//nullenkiller
          if(oi%2 == 1) *(blad+hind) = *(blad+hind) ^255; //jedes zweite mal bei hind invertieren
          ind = (ind +hind+oi)&mask; //neuer index wo# s hinzeigt plus 1 modulo blono
     }
      //now statistically closing the lid
      h1=(unsigned char) ind;  h2=(unsigned char) (hind+ind);
      for(oi=0;oi<2*(unsigned int)blolen;oi++) //dritte schleife, retrace buster
      {
          hind = ((int) *(blad+((oi+ind)&mask)))  &mask; // pointed-to index bestimmen
          //nicht sicher rueckabwickelbare entscheidung
          if( ((*(blad + hind) >> (oi&7))&1) ==1)
          {
              //der rotator
              h2+=*(blad+hind);
              *(blad+hind)= rotbyte(*(blad+hind),h1&7);
          }
          else
          {
              // der xor step
              h1+= *(blad+hind);
              *(blad+hind) =rotbyte(*(blad+hind),h2&7);
          }
          ind = hind;
      }
  }
  memcpy(pertxt,blad,blolen);
  if(blad != NULL) free( blad);
  return true;
}
/*********************************************/
/*********************************************/
/*********************************************/
bool bytxor_x1(int blolen, char* bladwell, char* pertxt, int rounds)

//Funktion setzt einen blolen großen Block in
//einen blolengroßen Permutationsarray um
//1.wohin zeigt byte-wohin zeigt das wieder+ index, original wird mit dem gexort
//2. wohin zeigt byte, wird mit dem gexort, wohin zeigt das, wird mit dem gexort etc blolen mal
// bei zwei heisst zeigt: alter index + neuer + index modulo blolen
{
  int ind,oi,k,hind;
  unsigned char *blad,h1,h2;

  if(ispowof2(blolen)) return(bytxor_x1_128(blolen,bladwell,pertxt, rounds));

  if(blolen<2)return false;

  blad=(unsigned char*)malloc(blolen);

  memcpy(blad,bladwell,blolen);

  for(k=0;k<rounds;k++)
  {
      for(oi=0;oi<blolen;oi++) //erste schleife systematisch
      {
          hind = (ulong32)   *(blad +( *(blad + oi) * *(blad+(oi+1)%blolen)) %blolen )%blolen; // pointed-to index bestimmen
          if(hind!=oi) *(blad + oi) += rotbyte(*(blad+hind),((unsigned char)(hind+oi)&7));
          //*(blad +oi)=rotbyte(*(blad+oi),(unsigned char)hind&7);
          *(blad+oi)+=63+(unsigned char)oi;
          if(oi%2 == 0) *(blad+hind) = *(blad+hind) ^255; //jedes zweite mal bei hind invertieren
      }
      ind=0;
      for(oi=0;oi<blolen;oi++) //zweite schleife wild gesprungwen
      {
          hind = (ulong32)*(blad+ind)%blolen; // pointed-to index bestimmen
         if(hind!= ind) *(blad + ind) += rotbyte(*(blad+hind),(unsigned char)(ind+hind)&7); //wird mit zeigerziel ge xort
         // *(blad +ind)=rotbyte(*(blad+ind),(unsigned char)oi&7);
          *(blad+ind)+=63+(unsigned char)oi;//nullenkiller
          if(oi%2 == 1) *(blad+hind) = *(blad+hind) ^255; //jedes zweite mal bei hind invertieren
          ind = (ind +hind+oi)%blolen; //neuer index wo# s hinzeigt plus 1 modulo blono
     }
      //now statistically closing the lid
      h1=(unsigned char) ind;  h2=(unsigned char) (hind+ind);
      for(oi=0;oi<2*blolen;oi++) //dritte schleife, retrace buster
      {
          hind = ((int) *(blad+((oi+ind)%blolen)))  %blolen; // pointed-to index bestimmen
          //nicht sicher rueckabwickelbare entscheidung
          if( ((*(blad + hind) >> (oi&7))&1) ==1)
          {
              //der rotator
              h2+=*(blad+hind);
              *(blad+hind)= rotbyte(*(blad+hind),h1&7);
          }
          else
          {
              // der xor step
              h1+= *(blad+hind);
              *(blad+hind) =rotbyte(*(blad+hind),h2&7);
          }
          ind = hind;
      }
  }
  memcpy(pertxt,blad,blolen);
  if(blad != NULL) free( blad);
  return true;
}
/*********************************************/
bool bytxor_x2(int blolen, char* bladwell, char* pertxt, int keylen, char* keytxt, int rounds)

//Funktion setzt einen blolen großen Block in
//einen blolengroßen Permutationsarray um
//1.wohin zeigt byte-wohin zeigt das wieder+ index, original wird mit dem gexort
//2. wohin zeigt byte, wird mit dem gexort, wohin zeigt das, wird mit dem gexort etc blolen mal
// bei zwei heisst zeigt: alter index + neuer + index modulo blolen
{
  int totlen;
  unsigned char *blad;
  bool retval;


  if(blolen<2)return false;
  totlen=blolen+keylen;
  blad=(unsigned char*)malloc(totlen);

  memcpy(blad,bladwell,blolen);
  memcpy(blad+blolen,keytxt,keylen);
  retval= bytxor_x1(totlen,(char *) blad, (char *) blad, rounds);

  memcpy(pertxt,blad+keylen/2,blolen);
  if(blad != NULL) free((char *) blad);
  return true;
}
/**************************************************************//**************************************************************/
bool kbytmixx(int blolen, char* bladwell, char* pertxt, int keylen, char* keytxt, int rounds)
// Funktion mixert und permutiert den block massiv durch mit keyblock

{
   int i;
   char *kbuff;

   //key muss evtl auf blolen gehashed werden
   if(keylen>= blolen) kbuff=bytmixxmalloc(keylen, keytxt, 1, blolen);
   //oder wie gehabt reentrant auffüllen
   else
   {
         kbuff= (char *) malloc(blolen);
   //keyblock kbuff mit key auffüllen
         memcpy(kbuff,keytxt,keylen);
         for(i= keylen;i<blolen;i++)
                *(kbuff+i)=(*(kbuff+(i%keylen)))^((char) i) ;//wird mit xortem mit indx aufgefüllt
   }

   for(i=0;i<blolen;i++)
   {
       *(kbuff+i)= *(bladwell + i) ^ *(kbuff+i); //kbuff wird mit bladwell gexort
   }


   if(!bytmixx_d(blolen, kbuff, pertxt, rounds))
     {
            throwout(_("warnung, Fehler in kbytmixx"));
            free(kbuff);
            return false;
     }
   free(kbuff);
   return true;
}
/*********************************************/
bool k_develop_f4(int blolen, char* bladwell, int cpa_blen, char* cpa_bl, int keylen, char* keytxt, int rounds, int algonum , memblo *mb)
// blolen lange bloecke für bladwell, cpa_blen fuer cpa_bl
//hier wird der keyblock auch im Algorithmus verwendet
//cpa_blocker and (derived?) key were precalculated
{
   unsigned int totlen,bufflen;
   char *devbuff;
   int i;
   bool okflag=true;

   //check validity of algonum
   if(algonum>16 || algonum<0){
     throwout(_("unbekannter Algorithmus in k_develop_f4")); return false;}

   totlen= blolen+keylen; //total blocklength
   bufflen= totlen + (totlen&3);
   devbuff= (char *) malloc(bufflen); //allocate working block, make sure to allocate space for an integer from last byte
   //zero excess bytes in advance
    *((unsigned int *) (devbuff+bufflen -4))=0;
   //set first part of working block from xored bladwell and cpa_bl
   if((cpa_blen>0)&&(cpa_bl != NULL))
   {
      for(i=0;i<blolen;i++)
         *(devbuff+i)= *(bladwell+i) ^ *(cpa_bl + (i%cpa_blen));
   }
   else
   {
      memcpy(devbuff,bladwell,blolen);
   }
   //fill (possibly derived) key into rest of working block
   memcpy(devbuff+blolen,keytxt,keylen);
   //oneway-develop devbuff for rounds times
   switch( algonum)
   {
     case 0: //for Fleas_ 4
       if(!develop_e(devbuff, blolen, rounds)) okflag = false;
       break;

     case 1:   //for Fleas_5
       if(!develop_x(devbuff, blolen, rounds, 5)) okflag = false;
       break;

     case 2:  //for x2
       if(!develop_x1(devbuff, blolen, rounds,2)) okflag = false;
       break;

     case 3:  //for x5
       if(!develop_x1(devbuff, blolen, rounds, 5)) okflag = false;
       break;

     case 4: //for o2
       if(ispowof2(totlen)) //wenn totlen eine zweierpotenz
       {
           if(!develop_o(devbuff, totlen, rounds,2,0)) okflag = false;
       }
       else if(ispowof2(blolen)&&blolen>keylen)
       {
         if(!develop_o(devbuff, blolen, rounds,2,0)) okflag = false;
         if(!develop_o(devbuff+keylen, blolen, rounds,2,0)) okflag = false;
       }
       else
       {
         if(!develop_o(devbuff, totlen, rounds,2,0)) okflag = false;
       }
       break;

     case 5: //for o5
       if(ispowof2(totlen)) //wenn totlen eine zweierpotenz
       {
           if(!develop_o(devbuff, totlen, rounds,5,2)) okflag = false;
       }
       else if(ispowof2(blolen)&&blolen>keylen)
       {
         if(!develop_o(devbuff, blolen, rounds,5,2)) okflag = false;
         if(!develop_o(devbuff+keylen, blolen, rounds,5,2)) okflag = false;
       }
       else
       {
         if(!develop_o(devbuff, totlen, rounds,5,2)) okflag = false;
       }
       break;

     case 6:  //for develop_l
       if(ispowof2(totlen)) //wenn totlen eine zweierpotenz
       {
           if(!develop_l(devbuff, totlen, rounds, mb)) okflag = false;
       }
       else if(ispowof2(blolen)&&blolen>keylen)
       {
         if(!develop_l(devbuff, blolen, rounds,mb)) okflag = false;
         if(!develop_l(devbuff+keylen, blolen, rounds,mb)) okflag = false;
       }
       else
       {
         if(!develop_l(devbuff, totlen, rounds,mb)) okflag = false;
       }
       break;

     case 7:  //für Fleas_ls
//       if(ispowof2(totlen)) //wenn totlen eine zweierpotenz
//       {
           if(!develop_ls(devbuff, totlen, rounds,2,1)) okflag = false;
//       }
//       else
  //     {
//         if(!develop_ls(devbuff, totlen, rounds,2,1)) okflag = false;
//       }
       break;

     case 8:  //für Hash: Fleas_lx3 und cipher l3
           if(!develop_ls(devbuff, totlen, rounds,3,1)) okflag = false;
       break;

     case 9:
     case 12:  //für Hash und cipher: Fleas_lc
           if(!develop_ln(devbuff, totlen, rounds,3,1)) okflag = false;
     break;

     case 10:  //für Hash und cipher: Fleas_ld jetzt multicore, 4 paths
           if(!develop_ln2_mt(devbuff, totlen, rounds,4,1)) okflag = false; //in CBC mode multipath
     break;
     case 13:  //Fleas_ld in CTR
           if(!develop_ln2(devbuff, totlen, rounds,4,1)) okflag = false; //in CTR mode no multipath
     break;

     case 14:    // fuer Hash Fleas_b
           if(!develop_ln2_mt(devbuff, totlen, rounds,2,1)) okflag = false;  //in CBC multithread
     break;
     case 11:  //fuer Hash und cipher: Fleas_lb jetzt multicore, 2 paths
           if(!develop_ln2(devbuff, totlen, rounds,2,1)) okflag = false;  //in CNT mode monothread
     break;

     case 15:   //for flight
           if(!develop_flight(devbuff, totlen)) okflag = false;
     break;
     case 16:   //for flight
           if(!develop_flightx(devbuff, totlen)) okflag = false;
     break;

     default:
        throwout(_("uUnbek. Algorithmus in k_develop_f4"));
        okflag=false;
   }
   if(!okflag) throwout(_("Fehler bei oneway-fn in k_develop_f4"));
   // overwrite bladwell with result from devbuff-center
   memcpy(bladwell,devbuff+keylen/2,blolen);
   free(devbuff); // free working block
   return okflag;
}
/*********************************************/
bool k_develop_b_f3(int blolen, char* bladwell, int keylen, char* keytxt, int rounds, unsigned int cagesize,int cpa_blen, char* cpa_bl)
// Funktion mixert und permutiert den block massiv durch mit keyblock
//jetzt wird der keyblock auch im Algorithmus verwendet, um stärker zu verschlüsseln

 {
   int i;
   unsigned int totlen;
   char *kbuff;

   totlen= blolen+keylen;
   if(totlen<cagesize) totlen=cagesize;
   kbuff= (char *) malloc(totlen);
   //cbuff2= (char *) malloc(totlen);
      //blockbegin kbuff plainly fill with argument if no chosen plaintaxt attack blocker present
    if((cpa_blen==0)||(cpa_bl==NULL)){
      for(i=0;i<blolen;i++)
         *(kbuff+i)= *(bladwell+i);
    }
    //if cpa-blocker present -> bytewise add with blocker
    else{
         for(i=0;i<blolen;i++)
         *(kbuff+i)= *(bladwell+i) + *(cpa_bl + (i%cpa_blen));
    }
    //fill in key
    for(i=blolen;i<blolen+keylen;i++)
       *(kbuff+i)= *(keytxt+i-blolen);
    //fill rest of cage with dummy if necessary
    for(i=(blolen+keylen);i<(int)totlen;i++)
       *(kbuff+i)= (char)i;
    if(!develop_b(kbuff, totlen, rounds))
     {
            throwout(_("warnung, Fehler in develop_b"));
            free(kbuff);
            return false;
     }
   memcpy(bladwell,kbuff+keylen/2,blolen);
   free(kbuff);
   return true;
}
/**************************************************************/
bool kbytmixx2(int blolen, char* bladwell, char* pertxt, int keylen, char* keytxt, int rounds, unsigned int cagesize,int cpa_blen, char* cpa_bl)
// Funktion mixert und permutiert den block massiv durch mit keyblock
//jetzt wird der keyblock auch im Algorithmus verwendet, um stärker zu verschlüsseln

 {
   int i;
   unsigned int totlen;
   char *kbuff,*cbuff2;

   totlen= blolen+keylen;
   if(totlen<cagesize) totlen=cagesize;
   kbuff= (char *) malloc(totlen);
   cbuff2= (char *) malloc(totlen);
      //blockbegin kbuff plainly fill with argument if no chosen plaintaxt attack blocker present
    if((cpa_blen==0)||(cpa_bl==NULL)){
      for(i=0;i<blolen;i++)
         *(kbuff+i)= *(bladwell+i);
    }
    //if cpa-blocker present -> bytewise add with blocker
    else{
         for(i=0;i<blolen;i++)
         *(kbuff+i)= *(bladwell+i) + *(cpa_bl + (i%cpa_blen));
    }
    //fill in key
    for(i=blolen;i<blolen+keylen;i++)
       *(kbuff+i)= *(keytxt+i-blolen);
    //fill rest of cage with dummy if necessary
    for(i=(blolen+keylen);i<(int)totlen;i++)
       *(kbuff+i)= (char)i;

/*
   for(i=0;i<rounds;i++)
   {

     bytinteg(totlen, kbuff, 127, 1); //integration zur "Entcodierung" von Texten
     if( !bytperm_x1(totlen, kbuff, cbuff2, 1))  {free(kbuff); return false;}
     bytinteg(totlen, cbuff2, 127, 1); //integration kostet nix und dispergiert
     memcpy(kbuff,cbuff2,totlen);
     if( !bytxor_x1(totlen, kbuff, cbuff2, 1)) {free(kbuff); return false;}
     //permutiertes und umgewidmetes liegt jetzt in cbuff2
     memcpy(kbuff,cbuff2,totlen);
   }*/
      if(!bytmixx_d(totlen, kbuff, cbuff2, rounds))
     {
            throwout(_("warnung, Fehler in kbytmixx"));
            free(kbuff);
            return false;
     }


   memcpy(pertxt,cbuff2+keylen/2,blolen);
   free(kbuff); free(cbuff2);
   return true;
}
/*********************************************/
bool bytperm_x2(int blolen, char* bladwell, char* pertxt, int keylen, char* keytxt, int rounds)

//Funktion setzt einen blolen großen Block in
//einen blolengroßen Permutationsarray um
//1.wohin zeigt byte-wohin zeigt das wieder+ index, original wird mit dem vertauscht
//2. wohin zeigt byte, wird mit dem gexort, wohin zeigt das, wird mit dem gexort etc blolen mal
// bei zwei heisst zeigt: alter index + neuer + index modulo blolen

{
  int totlen;
  unsigned char *blad;
  bool retval;


  if(blolen<2)return true;
  totlen=blolen+keylen;
  blad=(unsigned char*)malloc(totlen);

  memcpy(blad,bladwell,blolen);
  memcpy(blad+blolen,keytxt,keylen);


  retval=bytperm_x1(totlen,(char *)blad,(char *)blad,rounds);

  memcpy(pertxt,blad+keylen/2,blolen);
  if(blad != NULL) free((char *) blad);
  return retval;
}
/**************************************************************/
bool bytmult(int blolen, char* bladwell, char* pertxt, int rounds)
{
  unsigned int ind,hind,ihl,oi,magic;
  int k;
  unsigned char *blad,*zsto;


  if(ispowof2(blolen))
  {
      return bytmult_128(blolen,bladwell,pertxt,rounds);
  }
  if(blolen<2)return false;
  blad=(unsigned char*)malloc(blolen);
  memcpy(blad,bladwell,blolen);
  magic=(unsigned int) (blolen/2);
  for(k=0;k<rounds;k++)
  {
      for(oi=0;oi<(unsigned int)blolen;oi++) //first loop systematically
      {
		  // determine pointed-to pointed to index
          magic += (ulong32)   *(blad + (*(blad+oi)%blolen));
          hind = (oi + magic)%blolen; //add with oi for homogeneous index migration
          if(magic&4) { *(blad+oi) ^= 255;} //invert every now and then
          ihl= oi+ ((unsigned int) *(blad+hind)) * ((unsigned int) *(blad+oi)); //bitdiffusion by multiplication
         *(blad+oi)=  ((unsigned char)ihl)+ (unsigned char)(ihl>>8); //press into one byte
          if(hind&1) { *(blad+oi) ^= 255;} //entropy enhancer
          *(blad+hind)= rotbyte(*(blad+hind),((unsigned char)(oi))&7);  //rotator
      }
      ind = blolen/2; //initially set to middle of block
      zsto=(unsigned char *) malloc(blolen); //allocate space vor intermediate storage
      memcpy(zsto,blad,blolen);//fill with result of first loop
      for(oi=0;oi<(unsigned int)blolen;oi++) //second loop, jump self referentially
      {
          magic +=  *(blad + *(blad+((oi+ind)%blolen)) %blolen) ;
          hind = (oi + magic)%blolen; //add with oi for homogeneous index migration
          if(magic&2) { *(blad+(oi%blolen)) ^= 255;} //invert passive block every now and then
          ihl=(unsigned int)oi +((unsigned int) *(blad+hind)) * ((unsigned int) *(blad+ind)); //diffusion by multiplication
          *(zsto+ind)=  ((unsigned char)ihl)+ (unsigned char)(ihl>>8); //press into one byte
          if(oi&1) { *(zsto+ind) ^= 255;} //entropy enhancer
          *(zsto+ind)= rotbyte(*(zsto+ind),((unsigned char)(ind))&7); //rotator
          ind = hind; //use last index as new reference index
      }
      memcpy(blad,zsto,blolen); //overwrite first loop result with second loop result
      free(zsto); // free intermediate storage
  }
  memcpy(pertxt,blad,blolen);// copy result to output block
  if(blad != NULL) free( blad);
  return true;
}
/**************************************************************/
bool bytmult_a_128(int blolen, char* bladwell, char* pertxt, int rounds, int spice,char fr)
{
  unsigned int ind,hind,ihl,oi,magic;
  int k;
  unsigned char *blad,*zsto;
  unsigned int mask;
  unsigned char cmask;


  //seting masks, assuming blolen is powof 2
  mask=(unsigned int) blolen - 1;
  if (blolen> 128) cmask=255;
    else cmask= (unsigned char)blolen -1;


  if(blolen<2)return false;
  blad=(unsigned char*)malloc(blolen);
  memcpy(blad,bladwell,blolen);
  magic=(unsigned int) (blolen/2);
  for(k=0;k<rounds;k++)
  {
      for(oi=0;oi<(unsigned int)blolen;oi++) //first loop systematically up
      {
		  // determine pointed-to pointed to index
          magic += (ulong32)   *(blad + (*(blad+oi)&cmask));
          hind = (oi + magic+spice)&mask; //add with oi for homogeneous index migration
          if(magic&4) { *(blad+oi) ^= 255;} //invert every now and then
          ihl=oi+((unsigned int) *(blad+hind)) * ((unsigned int) *(blad+(oi&mask))); //bitdiffusion by multiplication
          //hier noch ein entropizer für ihl rein
          ihl += (oi + spice +magic)%257;
         *(blad+oi)^=  ((unsigned char)ihl); //+ (unsigned char)(ihl>>8); //press into one byte
          if(hind&1) { *(blad+oi) ^= 255;} //entropy enhancer
          *(blad+hind)= rotbyte(*(blad+hind),((unsigned char)(oi))&7);  //rotator
      }
      for(oi=(unsigned int)blolen-1;oi>0;oi--) //second loop systematically down
      {
		  // determine pointed-to pointed to index
          magic += (ulong32)   *(blad + (*(blad+oi)&cmask));
          hind = (oi + magic+spice)&mask; //add with oi for homogeneous index migration
          if(magic&4) { *(blad+oi) ^= 255;} //invert every now and then
          ihl=oi+((unsigned int) *(blad+hind)) * ((unsigned int) *(blad+(oi&mask))); //bitdiffusion by multiplication
          //hier noch ein entropizer für ihl rein
          ihl += (oi + spice +magic)%257;
         *(blad+oi)^=  ((unsigned char)ihl); //+ (unsigned char)(ihl>>8); //press into one byte
          if(hind&1) { *(blad+oi) ^= 255;} //entropy enhancer
          *(blad+hind)= rotbyte(*(blad+hind),((unsigned char)(oi))&7);  //rotator
      }
      ind = blolen/2; //initially set to middle of block
      zsto=(unsigned char *) malloc(blolen); //allocate space vor intermediate storage
      memcpy(zsto,blad,blolen);//fill with result of first loop
      for(oi=0;oi<(unsigned int)blolen*fr;oi++) //second loop, jump self referentially
      {
          magic += *(blad + (*(blad+((oi+ind)&mask))&cmask)) ;
          hind = (oi + magic+spice)&mask; //add with oi for homogeneous index migration
          if(magic&2) { *(blad+(oi&mask)) ^= 255;} //invert passive block every now and then
          ihl=oi +(((unsigned int) *(blad+hind))) * ((unsigned int) *(blad+ind)); //diffusion by multiplication
          //hier noch ein entropizer fuer ihl rein
          ihl += (oi + spice +magic)%257;
          *(zsto+ind)^=  ((unsigned char)ihl); //+ (unsigned char)(ihl>>8); //press into one byte
          if(oi&1) { *(zsto+ind) ^= 255;} //entropy enhancer
          *(zsto+ind)= rotbyte(*(zsto+ind),((unsigned char)(ind))&7); //rotator
          ind = hind; //use last index as new reference index
      }
      memcpy(blad,zsto,blolen); //overwrite first loop result with second loop result
      free(zsto); // free intermediate storage
  }
  memcpy(pertxt,blad,blolen);// copy result to output block
  if(blad != NULL) free( blad);
  return true;
}
/**************************************************************/
bool bbytmult_a_128(int blolen, char* bladwell, char* pertxt, int rounds, int spice,double fr)
{
  unsigned int ind,hind,ihl,oi,magic,thresh3;
  int k;
  unsigned char *blad,*zsto;
  unsigned int mask;
  unsigned char cmask;


  //seting masks, assuming blolen is powof 2
  mask=(unsigned int) blolen - 1;
  if (blolen> 128) cmask=255;
    else cmask= (unsigned char)blolen -1;


  if(blolen<2)return false;
  blad=(unsigned char*)malloc(blolen);
  memcpy(blad,bladwell,blolen);
  magic=(unsigned int) (blolen/2);
  for(k=0;k<rounds;k++)
  {
      for(oi=(unsigned int)blolen-1;oi>0;oi--) //second loop systematically down
      {
		  // determine pointed-to pointed to index
          magic += (ulong32)   *(blad + (*(blad+oi)&cmask));
          hind = (oi + magic+spice)&mask; //add with oi for homogeneous index migration
          if(magic&4) { *(blad+oi) ^= 255;} //invert every now and then
          ihl=oi+((unsigned int) *(blad+hind)) * ((unsigned int) *(blad+(oi&mask))); //bitdiffusion by multiplication
          //hier noch ein entropizer für ihl rein
          ihl += (oi + spice +magic)%257;
         *(blad+oi)^=  ((unsigned char)ihl); //+ (unsigned char)(ihl>>8); //press into one byte
          if(hind&1) { *(blad+oi) ^= 255;} //entropy enhancer
          *(blad+hind)= rotbyte(*(blad+hind),((unsigned char)(oi))&7);  //rotator
      }
      for(oi=0;oi<(unsigned int)blolen;oi++) //first loop systematically up
      {
		  // determine pointed-to pointed to index
          magic += (ulong32)   *(blad + (*(blad+oi)&cmask));
          hind = (oi + magic+spice)&mask; //add with oi for homogeneous index migration
          if(magic&4) { *(blad+oi) ^= 255;} //invert every now and then
          ihl=oi+((unsigned int) *(blad+hind)) * ((unsigned int) *(blad+(oi&mask))); //bitdiffusion by multiplication
          //hier noch ein entropizer für ihl rein
          ihl += (oi + spice +magic)%257;
         *(blad+oi)^=  ((unsigned char)ihl); //+ (unsigned char)(ihl>>8); //press into one byte
          if(hind&1) { *(blad+oi) ^= 255;} //entropy enhancer
          *(blad+hind)= rotbyte(*(blad+hind),((unsigned char)(oi))&7);  //rotator
      }
      ind = blolen/2; //initially set to middle of block
      zsto=(unsigned char *) malloc(blolen); //allocate space vor intermediate storage
      memcpy(zsto,blad,blolen);//fill with result of first loop
      thresh3=(unsigned int)((double)blolen*fr);
      for(oi=0;oi<thresh3;oi++) //second loop, jump self referentially
      {
          magic += *(blad + (*(blad+((oi+ind)&mask))&cmask)) ;
          hind = (oi + magic+spice)&mask; //add with oi for homogeneous index migration
          if(magic&2) { *(blad+(oi&mask)) ^= 255;} //invert passive block every now and then
          ihl=oi +(((unsigned int) *(blad+hind))) * ((unsigned int) *(blad+ind)); //diffusion by multiplication
          //hier noch ein entropizer fuer ihl rein
          ihl += (oi + spice +magic)%257;
          *(zsto+ind)^=  ((unsigned char)ihl); //+ (unsigned char)(ihl>>8); //press into one byte
          if(oi&1) { *(zsto+ind) ^= 255;} //entropy enhancer
          *(zsto+ind)= rotbyte(*(zsto+ind),((unsigned char)(ind))&7); //rotator
          ind = hind; //use last index as new reference index
      }
      memcpy(blad,zsto,blolen); //overwrite first loop result with second loop result
      free(zsto); // free intermediate storage
  }
  memcpy(pertxt,blad,blolen);// copy result to output block
  if(blad != NULL) free( blad);
  return true;
}
/**************************************************************/
/**************************************************************/
bool bytmult_a(int blolen, char* bladwell, char* pertxt, int rounds, int spice,char fr)
{
  unsigned int ind,hind,ihl,oi,magic;
  int k;
  unsigned char *blad,*zsto;

  if(blolen<2)return false;
  if(fr<0||fr>10)fr=3; //default is 3 anyways

  if(ispowof2(blolen)) return bytmult_a_128( blolen, bladwell, pertxt, rounds, spice, fr);

  blad=(unsigned char*)malloc(blolen);
  memcpy(blad,bladwell,blolen);
  magic=(unsigned int) (blolen/2);
  for(k=0;k<rounds;k++)
  {
      for(oi=0;oi<(unsigned int)blolen;oi++) //first loop systematically up
      {
		  // determine pointed-to pointed to index
          magic += (ulong32)   *(blad + (*(blad+oi)%blolen));
          hind = (oi + magic+spice)%blolen; //add with oi for homogeneous index migration
          if(magic&4) { *(blad+oi) ^= 255;} //invert every now and then
          ihl=oi+((unsigned int) *(blad+hind)) * ((unsigned int) *(blad+oi%blolen)); //bitdiffusion by multiplication
          //hier noch ein entropizer für ihl rein
          ihl += (oi + spice +magic)%257;
         *(blad+oi)^=  ((unsigned char)ihl); //+ (unsigned char)(ihl>>8); //press into one byte
          if(hind&1) { *(blad+oi) ^= 255;} //entropy enhancer
          *(blad+hind)= rotbyte(*(blad+hind),((unsigned char)(oi))&7);  //rotator
      }
      for(oi=(unsigned int)blolen-1;oi>0;oi--) //second loop systematically down
      {
		  // determine pointed-to pointed to index
          magic += (ulong32)   *(blad + (*(blad+oi)%blolen));
          hind = (oi + magic+spice)%blolen; //add with oi for homogeneous index migration
          if(magic&4) { *(blad+oi) ^= 255;} //invert every now and then
          ihl=oi+((unsigned int) *(blad+hind)) * ((unsigned int) *(blad+oi%blolen)); //bitdiffusion by multiplication
          //hier noch ein entropizer für ihl rein
          ihl += (oi + spice +magic)%257;
         *(blad+oi)^=  ((unsigned char)ihl); //+ (unsigned char)(ihl>>8); //press into one byte
          if(hind&1) { *(blad+oi) ^= 255;} //entropy enhancer
          *(blad+hind)= rotbyte(*(blad+hind),((unsigned char)(oi))&7);  //rotator
      }
      ind = blolen/2; //initially set to middle of block
      zsto=(unsigned char *) malloc(blolen); //allocate space vor intermediate storage
      memcpy(zsto,blad,blolen);//fill with result of first loop
      for(oi=0;oi<(unsigned int)blolen*fr;oi++) //second loop, jump self referentially
      {
          magic += *(blad + *(blad+((oi+ind)%blolen)) %blolen) ;
          hind = (oi + magic+spice)%blolen; //add with oi for homogeneous index migration
          if(magic&2) { *(blad+(oi%blolen)) ^= 255;} //invert passive block every now and then
          ihl=oi +(((unsigned int) *(blad+hind))) * ((unsigned int) *(blad+ind)); //diffusion by multiplication
          //hier noch ein entropizer fuer ihl rein
          ihl += (oi + spice +magic)%257;
          *(zsto+ind)^=  ((unsigned char)ihl); //+ (unsigned char)(ihl>>8); //press into one byte
          if(oi&1) { *(zsto+ind) ^= 255;} //entropy enhancer
          *(zsto+ind)= rotbyte(*(zsto+ind),((unsigned char)(ind))&7); //rotator
          ind = hind; //use last index as new reference index
      }
      memcpy(blad,zsto,blolen); //overwrite first loop result with second loop result
      free(zsto); // free intermediate storage
  }
  memcpy(pertxt,blad,blolen);// copy result to output block
  if(blad != NULL) free( blad);
  return true;
}
/**************************************************************/
bool bbytmult_ax(int blolen, char* bladwell, char* pertxt, int rounds, int spice,double fr)
{
  unsigned int oi,thresh3,blolenx,modlen;
  int k;
  ulong32 hind,ind,ihl,magic=0;
  unsigned char *blad,*zsto;

  if(blolen<4)return false;
  if(fr<0||fr>10)fr=3; //default is 3 anyways
  if(blolen&3) blolenx=(unsigned int) (blolen - (blolen&3) + 4);
  modlen=blolenx-4;
  blad=(unsigned char*)malloc(blolenx);

  memcpy(blad,bladwell,blolenx); //bladwell is supposed to be zero patched and mod4 allocated
    //xor high32bit with low
  *((ulong32*)(blad+modlen)) ^= *((ulong32*)(blad));

  magic=(ulong32) (blolen/2);
  for(k=0;k<rounds;k++)
  {
      for(oi=(unsigned int)blolen-4;oi>3;oi-=4) //first loop systematically down
      {
		  // determine pointed-to pointed to index
          magic +=  *((ulong32*)(blad + (*((ulong32*)(blad+oi))%modlen)));
          hind = (oi + magic+spice)%modlen; //add with oi for homogeneous index migration
          if(magic&4) { *((ulong32*)(blad+oi)) ^= 0xffffffff;} //invert every now and then
          ihl=oi+ (*((ulong32*)(blad+hind))) * ( *((ulong32*)(blad+oi%modlen))); //bitdiffusion by multiplication
          //hier noch ein entropizer für ihl rein
          ihl += (oi + spice +magic);
         *((ulong32*)(blad+oi))^=  ihl; //xor with ihl
          if(hind&1) { *(blad+oi) ^= 255;} //entropy enhancer
          *((ulong32*)(blad+hind))= rot32(*((ulong32*)(blad+hind)),((unsigned char)(oi))&31);  //rotator
          if(oi<4) //just for debugging
          {
              continue;
          }
      }
      if(oi>0)  //didnt hit 0 exactly
      {
		  // determine pointed-to pointed to index
          magic +=  *((ulong32*)(blad + (*((ulong32*)(blad))%modlen)));
          hind = (magic+spice)%modlen; //add with oi for homogeneous index migration
          if(magic&4) { *((ulong32*)(blad)) ^= 0xffffffff;} //invert every now and then
          ihl=(*((ulong32*)(blad+hind))) * ( *((ulong32*)(blad))); //bitdiffusion by multiplication
          //hier noch ein entropizer für ihl rein
          ihl += ( spice +magic);
         *((ulong32*)(blad))^=  ihl; //xor with ihl
          if(hind&1) { *(blad) ^= 255;} //entropy enhancer
      }

      for(oi=0;oi<(unsigned int)blolen-3;oi+=4) //second loop systematically up
      {
		  // determine pointed-to pointed to index
          magic +=  *((ulong32*)(blad + (*((ulong32*)(blad+oi))%modlen)));
          hind = (oi + magic+spice)%modlen; //add with oi for homogeneous index migration
          if(magic&4) { *((ulong32*)(blad+oi)) ^= 0xffffffff;} //invert every now and then
          ihl=oi+( *((ulong32*)(blad+hind))) * ( *((ulong32*)(blad+oi%modlen))); //bitdiffusion by multiplication
          //hier noch ein entropizer für ihl rein
          ihl += (oi + spice +magic);
         *((ulong32*)(blad+oi))^=  ihl; //xor with ihl
          if(hind&1) { *(blad+oi) ^= 255;} //entropy enhancer
          *((ulong32*)(blad+hind))= rot32(*((ulong32*)(blad+hind)),((unsigned char)(oi))&31);  //rotator
      }
      if(oi!=(unsigned int)blolen-3)
      {
          oi=blolen-3;
		  // determine pointed-to pointed to index
          magic +=  *((ulong32*)(blad + (*((ulong32*)(blad+oi))%modlen)));
          hind = (oi + magic+spice)%modlen; //add with oi for homogeneous index migration
          if(magic&4) { *((ulong32*)(blad+oi)) ^= 0xffffffff;} //invert every now and then
          ihl=oi+( *((ulong32*)(blad+hind))) * ( *((ulong32*)(blad+oi%modlen))); //bitdiffusion by multiplication
          //hier noch ein entropizer für ihl rein
          ihl += (oi + spice +magic);
         *((ulong32*)(blad+oi))^=  ihl; //xor with ihl
          if(hind&1) { *(blad+oi) ^= 255;} //entropy enhancer
          *((ulong32*)(blad+hind))= rot32(*((ulong32*)(blad+hind)),((unsigned char)(oi))&31);  //rotator
      }


      ind = blolen/2; //initially set to middle of block
      zsto=(unsigned char *) malloc(blolenx); //allocate space vor intermediate storage
      memcpy(zsto,blad,blolenx);//fill with result of first loop
      thresh3=(unsigned int)((double)blolen*fr)/2;
      for(oi=0;oi<thresh3;oi++) //third loop, jump self referentially in 4byte clusters
      {
          magic += *((ulong32*)(blad + *((ulong32*)(blad+((oi+ind)%modlen))) %modlen)) ;
          hind = (oi + magic+spice)%modlen; //add with oi for homogeneous index migration
          if(magic&2) { *((ulong32*)(blad+oi%modlen)) ^= 0xffffffff;} //invert passive block every now and then
          ihl=oi + *((ulong32*)(blad+hind)) *  *((ulong32*)(blad+ind)); //diffusion by multiplication
          //hier noch ein entropizer fuer ihl rein
          ihl += oi + spice +magic;
          *((ulong32*)(zsto+ind))^=  ihl;
          if(oi&1) { *(zsto+ind) ^= 255;} //entropy enhancer
          *((ulong32*)(zsto+ind))= rot32( *((ulong32*)(zsto+ind)),((unsigned char)(ind))&31); //rotator
          ind = hind; //use last index as new reference index
      }
      for(oi=0;oi<thresh3;oi++) //fourth loop, jump self referentially bytewise
      {
          magic += *(blad + *(blad+((oi+ind)%blolen)) %blolen) ;
          hind = (oi + magic+spice)%blolen; //add with oi for homogeneous index migration
          if(magic&2) { *(blad+(oi%blolen)) ^= 255;} //invert passive block every now and then
          ihl=oi +(((unsigned int) *(blad+hind))) * ((unsigned int) *(blad+ind)); //diffusion by multiplication
          //hier noch ein entropizer fuer ihl rein
          ihl += (oi + spice +magic)%257;
          *(zsto+ind)^=  ((unsigned char)ihl); //+ (unsigned char)(ihl>>8); //press into one byte
          if(oi&1) { *(zsto+ind) ^= 255;} //entropy enhancer
          *(zsto+ind)= rotbyte(*(zsto+ind),((unsigned char)(ind))&7); //rotator
          ind = hind; //use last index as new reference index
      }
      memcpy(blad,zsto,blolen); //overwrite first loop result with second loop result
      free(zsto); // free intermediate storage
  }
  memcpy(pertxt,blad,blolen);// copy result to output block
  if(blad != NULL) free( blad);
  return true;
}
/**************************************************************/
bool bbytmult_a(int blolen, char* bladwell, char* pertxt, int rounds, int spice,double fr)
{
  unsigned int ind,hind,ihl,oi,magic,thresh3;
  int k;
  unsigned char *blad,*zsto;

  if(blolen<2)return false;
  if(fr<0||fr>10)fr=3; //default is 3 anyways

  if(ispowof2(blolen)) return bbytmult_a_128( blolen, bladwell, pertxt, rounds, spice, fr);

  blad=(unsigned char*)malloc(blolen);
  memcpy(blad,bladwell,blolen);
  magic=(unsigned int) (blolen/2);
  for(k=0;k<rounds;k++)
  {
      for(oi=(unsigned int)blolen-1;oi>0;oi--) //second loop systematically down
      {
		  // determine pointed-to pointed to index
          magic += (ulong32)   *(blad + (*(blad+oi)%blolen));
          hind = (oi + magic+spice)%blolen; //add with oi for homogeneous index migration
          if(magic&4) { *(blad+oi) ^= 255;} //invert every now and then
          ihl=oi+((unsigned int) *(blad+hind)) * ((unsigned int) *(blad+oi%blolen)); //bitdiffusion by multiplication
          //hier noch ein entropizer für ihl rein
          ihl += (oi + spice +magic)%257;
         *(blad+oi)^=  ((unsigned char)ihl); //+ (unsigned char)(ihl>>8); //press into one byte
          if(hind&1) { *(blad+oi) ^= 255;} //entropy enhancer
          *(blad+hind)= rotbyte(*(blad+hind),((unsigned char)(oi))&7);  //rotator
      }
      for(oi=0;oi<(unsigned int)blolen;oi++) //first loop systematically up
      {
		  // determine pointed-to pointed to index
          magic += (ulong32)   *(blad + (*(blad+oi)%blolen));
          hind = (oi + magic+spice)%blolen; //add with oi for homogeneous index migration
          if(magic&4) { *(blad+oi) ^= 255;} //invert every now and then
          ihl=oi+((unsigned int) *(blad+hind)) * ((unsigned int) *(blad+oi%blolen)); //bitdiffusion by multiplication
          //hier noch ein entropizer für ihl rein
          ihl += (oi + spice +magic)%257;
         *(blad+oi)^=  ((unsigned char)ihl); //+ (unsigned char)(ihl>>8); //press into one byte
          if(hind&1) { *(blad+oi) ^= 255;} //entropy enhancer
          *(blad+hind)= rotbyte(*(blad+hind),((unsigned char)(oi))&7);  //rotator
      }
      ind = blolen/2; //initially set to middle of block
      zsto=(unsigned char *) malloc(blolen); //allocate space vor intermediate storage
      memcpy(zsto,blad,blolen);//fill with result of first loop
      thresh3=(unsigned int)((double)blolen*fr);
      for(oi=0;oi<thresh3;oi++) //second loop, jump self referentially
      {
          magic += *(blad + *(blad+((oi+ind)%blolen)) %blolen) ;
          hind = (oi + magic+spice)%blolen; //add with oi for homogeneous index migration
          if(magic&2) { *(blad+(oi%blolen)) ^= 255;} //invert passive block every now and then
          ihl=oi +(((unsigned int) *(blad+hind))) * ((unsigned int) *(blad+ind)); //diffusion by multiplication
          //hier noch ein entropizer fuer ihl rein
          ihl += (oi + spice +magic)%257;
          *(zsto+ind)^=  ((unsigned char)ihl); //+ (unsigned char)(ihl>>8); //press into one byte
          if(oi&1) { *(zsto+ind) ^= 255;} //entropy enhancer
          *(zsto+ind)= rotbyte(*(zsto+ind),((unsigned char)(ind))&7); //rotator
          ind = hind; //use last index as new reference index
      }
      memcpy(blad,zsto,blolen); //overwrite first loop result with second loop result
      free(zsto); // free intermediate storage
  }
  memcpy(pertxt,blad,blolen);// copy result to output block
  if(blad != NULL) free( blad);
  return true;
}
/**************************************************************/

bool bytmult_128(int blolen, char* bladwell, char* pertxt, int rounds)
{
  unsigned int ind,hind,ihl,oi,magic;
  int k,mask;
  unsigned char *blad,*zsto;



  mask =blolen-1;

  if(blolen<2)return false;
  mask =blolen-1;
  blad=(unsigned char*)malloc(blolen);
  memcpy(blad,bladwell,blolen);
  magic=(unsigned int) (blolen/2);
  for(k=0;k<rounds;k++)
  {
      for(oi=0;oi<(unsigned int)blolen;oi++) //first loop systematically
      {
		  // determine pointed-to pointed to index
          magic += (ulong32)   *(blad + (*(blad+oi)&mask));
          hind = (oi + magic)&mask; //add with oi for homogeneous index migration
          if(magic&4) { *(blad+oi) ^= 255;} //invert every now and then
          ihl= oi+ ((unsigned int) *(blad+hind)) * ((unsigned int) *(blad+oi)); //bitdiffusion by multiplication
         *(blad+oi)=  ((unsigned char)ihl)+ (unsigned char)(ihl>>8); //press into one byte
          if(hind&1) { *(blad+oi) ^= 255;} //entropy enhancer
          *(blad+hind)= rotbyte(*(blad+hind),((unsigned char)(oi))&7);  //rotator
      }
      ind = blolen/2; //initially set to middle of block
      zsto=(unsigned char *) malloc(blolen); //allocate space vor intermediate storage
      memcpy(zsto,blad,blolen);//fill with result of first loop
      for(oi=0;oi<(unsigned int)blolen;oi++) //second loop, jump self referentially
      {
          magic +=  *(blad +(((ulong32) *(blad +((oi+ind)&mask)))  &mask)) ;
          hind = (oi + magic)&mask; //add with oi for homogeneous index migration
          if(magic&2) { *(blad+(oi&mask)) ^= 255;} //invert passive block every now and then
          ihl=(unsigned int)oi +((unsigned int) *(blad+hind)) * ((unsigned int) *(blad+ind)); //diffusion by multiplication
          *(zsto+ind)=  ((unsigned char)ihl)+ (unsigned char)(ihl>>8); //press into one byte
          if(oi&1) { *(zsto+ind) ^= 255;} //entropy enhancer
          *(zsto+ind)= rotbyte(*(zsto+ind),((unsigned char)(ind))&7); //rotator
          ind = hind; //use last index as new reference index
      }
      memcpy(blad,zsto,blolen); //overwrite first loop result with second loop result
      free(zsto); // free intermediate storage
  }
  memcpy(pertxt,blad,blolen);// copy result to output block
  if(blad != NULL) free( blad);
  return true;
}
/******************************************************************************************//**************************************************************//*********************************************//*********************************************/
/******************************************************************************************//**************************************************************//*********************************************//*********************************************/
bool bytperm(int blolen, char* bladwell, char* pertxt, int rounds)
//ausgemagerte version ohne combination lock
{
  unsigned int ind,hind,ihl;
  int oi,k;
  unsigned char *blad,*zsto;

  if(ispowof2(blolen))
  {
      return bytperm_128(blolen,bladwell,pertxt,rounds);
  }
  if(blolen<2)return false;

  blad=(unsigned char*)malloc(blolen);

  memcpy(blad,bladwell,blolen);

  for(k=0;k<rounds;k++)
  {
      for(oi=0;oi<blolen;oi++) //erste schleife systematisch
      {
          hind = (ulong32)   *(blad + (*(blad+oi)%blolen))%blolen ; // pointed-to pointed to index bestimmen
          hind = (hind + oi)%blolen; //wegen homogener durchwanderung mit oi addieren
          // jetzt multipl. und splitten
          if(hind&4) { *(blad+oi) ^= 255;} //invertieren
          ihl= oi+ ((unsigned int) *(blad+hind)) * ((unsigned int) *(blad+oi)); //Bitdiffusion durch Multiplikation
          ihl%=257;
          ihl+=oi;
         *(blad+oi)=  ((unsigned char)ihl)+ ((unsigned char)ihl>>8); //bytzusammenfuehrung
          // jetzt der Entropiestep
          if(hind&1) { *(blad+oi) ^= 255;}
                     //jetzt der rotator
          *(blad+hind)= rotbyte(*(blad+hind),((unsigned char)(oi))&7);
      }
      ind = blolen/2;
      zsto=(unsigned char *) malloc(blolen);
      memcpy(zsto,blad,blolen);
      for(oi=0;oi<blolen;oi++) //zweite schleife wild gesprungwen
      {
          hind =  (*(blad +(ulong32) *(blad+((oi+ind)%blolen))  %blolen) )%blolen ; // pointed-to pointed to index bestimmen
          hind = (hind + oi)%blolen; //wegen homogener durchwanderung mit oi addieren
          // jetzt ein Entropiestep
          if(hind&4) { *(blad+(oi%blolen)) ^= 255;}
          //mit hind vertauschen
          ihl=  (unsigned int)oi +((unsigned int) *(blad+hind)) * ((unsigned int) *(blad+ind)); //Bitdiffusion durch Multiplikation
          ihl%=257;
          *(zsto+ind)= (unsigned char) oi+ ((unsigned char)ihl&255)+ (unsigned char)(ihl>>8); //unteres byte
          // jetzt der Entropiestep
          if(hind&1) { *(zsto+ind) ^= 255;}
           //jetzt der rotator
          *(zsto+ind)= rotbyte(*(zsto+ind),((unsigned char)(ind))&7);
          //einmal durchwechseln
          ind = hind;
      }
      memcpy(blad,zsto,blolen);
      free(zsto);
  }
  memcpy(pertxt,blad,blolen);
  if(blad != NULL) free( blad);
  return true;
}
/**************************************************************//**************************************************************//*********************************************//*********************************************/
bool bytperm_x1(int blolen, char* bladwell, char* pertxt, int rounds)

//Funktion setzt einen blolen großen Block in
//einen blolengroßen Permutationsarray um
//1.wohin zeigt byte-wohin zeigt das wieder+ index, original wird mit dem vertauscht
//2. wohin zeigt byte, wird mit dem gexort, wohin zeigt das, wird mit dem gexort etc blolen mal
// bei zwei heisst zeigt: alter index + neuer + index modulo blolen

{
  unsigned int ind,hind,ihl;
  int oi,k;
  unsigned char *blad,h1,h2,*zsto;

  if(ispowof2(blolen))
  {
      return bytperm_x1_128(blolen,bladwell,pertxt,rounds);
  }
  if(blolen<2)return false;

  blad=(unsigned char*)malloc(blolen);

  memcpy(blad,bladwell,blolen);

  for(k=0;k<rounds;k++)
  {
      for(oi=0;oi<blolen;oi++) //erste schleife systematisch
      {
          hind = (ulong32)   *(blad + (*(blad+oi)%blolen))%blolen ; // pointed-to pointed to index bestimmen
          hind = (hind + oi)%blolen; //wegen homogener durchwanderung mit oi addieren
          // jetzt multipl. und splitten
          if(hind&4) { *(blad+oi) ^= 255;} //invertieren
          ihl= oi+ ((unsigned int) *(blad+hind)) * ((unsigned int) *(blad+oi)); //Bitdiffusion durch Multiplikation
          ihl%=257;
          ihl+=oi;
          //if(ihl==0){ihl = oi*oi+ (*(blad+hind)*  *(blad+hind) ) + (*(blad+oi) * *(blad+oi));}
         *(blad+oi)=  ((unsigned char)ihl)+ ((unsigned char)ihl>>8); //bytzusammenfuehrung
          //if(*(blad+oi)==0) *(blad+oi)=(unsigned char) (ihl+hind);
          // jetzt der Entropiestep
          if(hind&1) { *(blad+oi) ^= 255;}
                     //jetzt der rotator
          *(blad+hind)= rotbyte(*(blad+hind),((unsigned char)(oi))&7);
      }
      ind = blolen/2;
      zsto=(unsigned char *) malloc(blolen);
      memcpy(zsto,blad,blolen);
      for(oi=0;oi<blolen;oi++) //zweite schleife wild gesprungwen
      {
          hind =  (*(blad +(ulong32) *(blad+((oi+ind)%blolen))  %blolen) )%blolen ; // pointed-to pointed to index bestimmen
          hind = (hind + oi)%blolen; //wegen homogener durchwanderung mit oi addieren
          // jetzt ein Entropiestep
          if(hind&4) { *(blad+(oi%blolen)) ^= 255;}
          //mit hind vertauschen
          ihl=  (unsigned int)oi +((unsigned int) *(blad+hind)) * ((unsigned int) *(blad+ind)); //Bitdiffusion durch Multiplikation
          ihl%=257;
          //if(ihl<2){ihl = oi*oi+ (*(blad+hind)*  *(blad+hind) ) + (*(blad+ind) * *(blad+ind));}
          *(zsto+ind)= (unsigned char) oi+ ((unsigned char)ihl&255)+ (unsigned char)(ihl>>8); //unteres byte
          //*(blad+hind)= (unsigned char) (ihl>>8);  //oberes byte lieber nicht nehmen
          //Nullenfresser
          //if(*(blad+ind)==0) *(blad+ind)=(unsigned char) hind;
          // jetzt der Entropiestep
          if(hind&1) { *(zsto+ind) ^= 255;}
           //jetzt der rotator
          *(zsto+ind)= rotbyte(*(zsto+ind),((unsigned char)(ind))&7);
          //einmal durchwechseln
          ind = hind;
      }
      memcpy(blad,zsto,blolen);
      free(zsto);
     //now statistically closing the lid
      h1=(unsigned char) ind;  h2=(unsigned char) (hind+ind);
      for(oi=0;oi<2*blolen;oi++) //dritte schleife, retrace buster
      {
          hind = ((int) *(blad+((oi+ind)%blolen)))  %blolen; // pointed-to index bestimmen
          //nicht sicher rueckabwickelbare entscheidung
          if( ((*(blad + hind) >> (oi&7))&1) ==1)
          {
              //der rotator
              h2+=*(blad+hind);
              *(blad+hind)= rotbyte(*(blad+hind),h1&7);
          }
          else
          {
              // der alternativer rotator
              h1+= *(blad+hind);
              *(blad+hind) =rotbyte(*(blad+hind),h2&7);
          }
          ind = hind;
      }
  }
  memcpy(pertxt,blad,blolen);
  if(blad != NULL) free( blad);
  return true;
}
/**************************************************************/
/**************************************************************/
/*********************************************//*********************************************/
bool bytperm_128( int blolen, char* bladwell, char* pertxt, int rounds)
//Schnellversion für 128 lockgroesse/

{
  unsigned int ind,hind,ihl,mask;
  int oi,k;
  unsigned char *blad,*zsto;


  if(blolen<2)return false;
  mask =blolen-1;

  blad=(unsigned char*)malloc(blolen);

  memcpy(blad,bladwell,blolen);


  for(k=0;k<rounds;k++)
  {
      for(oi=0;oi<blolen;oi++) //erste schleife systematisch
      {
          hind = (ulong32)   *(blad + (*(blad+oi)&mask))&mask ; // pointed-to pointed to index bestimmen
          hind = (hind + oi)&mask; //wegen homogener durchwanderung mit oi addieren
          // jetzt multipl. und splitten
          if(hind&4) { *(blad+oi) ^= 255;} //invertieren
          ihl= oi+ ((unsigned int) *(blad+hind)) * ((unsigned int) *(blad+oi)); //Bitdiffusion durch Multiplikation
          ihl%=257;
          ihl+=oi;
         *(blad+oi)=  ((unsigned char)ihl)+ ((unsigned char)ihl>>8); //bytzusammenfuehrung
          if(hind&1) { *(blad+oi) ^= 255;}
                     //jetzt der rotator
          *(blad+hind)= rotbyte(*(blad+hind),((unsigned char)(oi))&7);
      }
      ind = blolen/2;
      zsto=(unsigned char *) malloc(blolen);
      memcpy(zsto,blad,blolen);
      for(oi=0;oi<blolen;oi++) //zweite schleife wild gesprungwen
      {
          hind =  (*(blad +((ulong32) *(blad+((oi+ind)&mask)) &mask)) )&mask ; // pointed-to pointed to index bestimmen
          hind = (hind + oi)&mask; //wegen homogener durchwanderung mit oi addieren
          // jetzt ein Entropiestep
          if(hind&4) { *(blad+(oi&mask)) ^= 255;}
          //mit hind vertauschen
          ihl=  (unsigned int)oi +((unsigned int) *(blad+hind)) * ((unsigned int) *(blad+ind)); //Bitdiffusion durch Multiplikation
          ihl%=257;
          *(zsto+ind)= (unsigned char) oi+ ((unsigned char)ihl&255)+ (unsigned char)(ihl>>8); //unteres byte
          if(hind&1) { *(zsto+ind) ^= 255;}
           //jetzt der rotator
          *(zsto+ind)= rotbyte(*(zsto+ind),((unsigned char)(ind))&7);
          //einmal durchwechseln
          ind = hind;
      }
      memcpy(blad,zsto,blolen);
      free(zsto);
  }
  memcpy(pertxt,blad,blolen);
  if(blad != NULL) free( blad);
  return true;
}
/**************************************************************/
/**************************************************************/
/*********************************************//*********************************************/
bool bytperm_x1_128( int blolen, char* bladwell, char* pertxt, int rounds)
//Schnellversion für 128 lockgroesse/

{
  unsigned int ind,hind,ihl,mask;
  int oi,k;
  unsigned char *blad,h1,h2,*zsto;


  if(blolen<2)return false;
  mask =blolen-1;

  blad=(unsigned char*)malloc(blolen);

  memcpy(blad,bladwell,blolen);


  for(k=0;k<rounds;k++)
  {
      for(oi=0;oi<blolen;oi++) //erste schleife systematisch
      {
          hind = (ulong32)   *(blad + (*(blad+oi)&mask))&mask ; // pointed-to pointed to index bestimmen
          hind = (hind + oi)&mask; //wegen homogener durchwanderung mit oi addieren
          // jetzt multipl. und splitten
          if(hind&4) { *(blad+oi) ^= 255;} //invertieren
          ihl= oi+ ((unsigned int) *(blad+hind)) * ((unsigned int) *(blad+oi)); //Bitdiffusion durch Multiplikation
          ihl%=257;
          ihl+=oi;
          //if(ihl==0){ihl = oi*oi+ (*(blad+hind)*  *(blad+hind) ) + (*(blad+oi) * *(blad+oi));}
         *(blad+oi)=  ((unsigned char)ihl)+ ((unsigned char)ihl>>8); //bytzusammenfuehrung
          //if(*(blad+oi)==0) *(blad+oi)=(unsigned char) (ihl+hind);
          // jetzt der Entropiestep
          if(hind&1) { *(blad+oi) ^= 255;}
                     //jetzt der rotator
          *(blad+hind)= rotbyte(*(blad+hind),((unsigned char)(oi))&7);
      }
      ind = blolen/2;
      zsto=(unsigned char *) malloc(blolen);
      memcpy(zsto,blad,blolen);
      for(oi=0;oi<blolen;oi++) //zweite schleife wild gesprungwen
      {
          hind =  (*(blad +((ulong32) *(blad+((oi+ind)&mask)) &mask)) )&mask ; // pointed-to pointed to index bestimmen
          hind = (hind + oi)&mask; //wegen homogener durchwanderung mit oi addieren
          // jetzt ein Entropiestep
          if(hind&4) { *(blad+(oi&mask)) ^= 255;}
          //mit hind vertauschen
          ihl=  (unsigned int)oi +((unsigned int) *(blad+hind)) * ((unsigned int) *(blad+ind)); //Bitdiffusion durch Multiplikation
          ihl%=257;
          //if(ihl<2){ihl = oi*oi+ (*(blad+hind)*  *(blad+hind) ) + (*(blad+ind) * *(blad+ind));}
          *(zsto+ind)= (unsigned char) oi+ ((unsigned char)ihl&255)+ (unsigned char)(ihl>>8); //unteres byte
          //*(blad+hind)= (unsigned char) (ihl>>8);  //oberes byte lieber nicht nehmen
          //Nullenfresser
          //if(*(blad+ind)==0) *(blad+ind)=(unsigned char) hind;
          // jetzt der Entropiestep
          if(hind&1) { *(zsto+ind) ^= 255;}
           //jetzt der rotator
          *(zsto+ind)= rotbyte(*(zsto+ind),((unsigned char)(ind))&7);
          //einmal durchwechseln
          ind = hind;
      }
      memcpy(blad,zsto,blolen);
      free(zsto);
     //now statistically closing the lid
      h1=(unsigned char) ind;  h2=(unsigned char) (hind+ind);
      for(oi=0;oi<2*blolen;oi++) //dritte schleife, retrace buster
      {
          hind = ((int) *(blad+((oi+ind)&mask)))  &mask; // pointed-to index bestimmen
          //nicht sicher rueckabwickelbare entscheidung
          if( ((*(blad + hind) >> (oi&7))&1) ==1)
          {
              //der rotator
              h2+=*(blad+hind);
              *(blad+hind)= rotbyte(*(blad+hind),h1&7);
          }
          else
          {
              // der alternativer rotator
              h1+= *(blad+hind);
              *(blad+hind) =rotbyte(*(blad+hind),h2&7);
          }
          ind = hind;
      }
  }
  memcpy(pertxt,blad,blolen);
  if(blad != NULL) free( blad);
  return true;
}
/**************************************************************/
bool bytinteg(int blolen, char* blo, int keylen, char* keytxt, unsigned char offs, unsigned char inc)
// integriert einmal rauf und einmal runter, add offset am Start und add inc bei jedem Schritt
// dient zum "entcodieren" von Textinfo
// dient auch zur Bitdispersion:Ändere ein Bit am Anfang -> beeinglusse jedes Byte

{
  unsigned int ind,totlen;
  unsigned char *blad;


  if(blolen<2)return false;
  totlen=blolen+keylen;
  blad=(unsigned char*)malloc(totlen);

  memcpy(blad,blo,blolen);
  memcpy(blad+blolen,keytxt,keylen);
  *blad += offs;
  for(ind=1;ind<totlen;ind++) //raufintegriert
  {
    *(blad+ind) += *(blad+ind-1) + inc;
  }
  for(ind=totlen-1;ind>0;ind--) //runterintegriert
  {
    *(blad+ind-1) += *(blad+ind) + inc;
  }
  memcpy(blo,blad,blolen);
  if(blad != NULL) free(blad);
  return true;
}
/*********************************************/
/**************************************************************/
bool bytinteg(int blolen, char* blo, unsigned char offs, unsigned char inc)
// integriert einmal rauf und einmal runter, add offset am Start und add inc bei jedem Schritt
// dient zum "entcodieren" von Textinfo
// dient auch zur Bitdispersion:Ändere ein Bit am Anfang -> beeinglusse jedes Byte

{
  unsigned int ind;

  if(blolen<2)return false;

  *blo += (char) offs;
  for(ind=1;ind<(unsigned int)blolen;ind++) //raufintegriert
  {
    *(blo+ind) += *(blo+ind-1) + (char )inc;
  }
  for(ind=blolen-1;ind>0;ind--) //runterintegriert
  {
    *(blo+ind-1) += *(blo+ind) + (char) inc;
  }
  return true;
}
/*******************************************************************************/
unsigned int get_code_4_algo(wxString algo)
{
    unsigned int aflag;

    if(algo == _("Fleas_1_8")) aflag=18;
    else if(algo == _("Fleas_3")) aflag=19;
    else if(algo == _("Fleas_4")) aflag=0;
    else if(algo == _("Fleas_5")) aflag=1;
    else if(algo == _("Fleas_x2")) aflag=2;
    else if(algo == _("Fleas_x5")) aflag=3;
    else if(algo == _("Fleas_o2")) aflag=4;
    else if(algo == _("Fleas_o5")) aflag=5;
    else if(algo == _("Fleas_l")) aflag=6;
    else if(algo == _("Fleas_ls")) aflag=7;
    else if(algo == _("Fleas_l3")) aflag=8;
    else if(algo == _("Fleas_lc")) aflag=9;
    else if(algo == _("Fleas_ld")) aflag=10;
    else if(algo == _("Fleas_lb")) aflag=11;
    else if(algo == _("flight")) aflag=15;
    else if(algo == _("flightx")) aflag=16;
    else if(algo == _("threefish")) aflag=17;
    else if(algo == _("chimera")) aflag=20;
    else
    {
        //throwout(_("unknown algorithm in roundfun"));
        return  1001;
    }
    return aflag;
}
/*******************************************************************************/
bool roundfun(int bllen, int keylen, char* ptxt, char* key, int laps, int cpa_blen, char* cpa_bl, wxString algo)
{
  unsigned int aflag;

  aflag= get_code_4_algo(algo);
  if(aflag>1000) return false;
  return( roundfun(bllen, keylen, ptxt, key, laps, cpa_blen, cpa_bl, aflag));

}
/**************************************************************/
/*******************************************************************************/
bool roundfun(int bllen, int keylen, char* ptxt, char* key, int laps, int cpa_blen, char* cpa_bl, unsigned int algoflag)
{
  char *pertxt,*lk;
  int i;


  pertxt=(char*)malloc(bllen);
  lk=(char *)malloc(keylen);

  memcpy(lk,key,keylen);

  switch(algoflag)
  {
      case 18: //Fleas 1_8
            kbytmixx2(bllen, ptxt, pertxt, keylen, lk, laps,20,cpa_blen,cpa_bl);
            for(i=0;i<bllen;i++)
            {
                *(ptxt+i) = *(pertxt+i);
            }
            break;

    case 19:  //Fleas_3
       if(!k_develop_b_f3(bllen, ptxt, keylen, lk, laps,20,cpa_blen,cpa_bl))
       {
           throwout(_("Fehler a1 in kdevelop_b_f3"));
            free((char *)pertxt);
            free((char *)lk);
            return false;
       }
       break;

     case 0: //Fleas_4
     //the fleas 4 routine
       if(!k_develop_f4(bllen, ptxt, cpa_blen,cpa_bl, keylen, lk, laps,0))
       {
           throwout(_("Fehler a2 in k_develop_f4"));
            free((char *)pertxt);
            free((char *)lk);
            return false;
       }
    break;
    case 1:
 //the fleas 5 routine
       if(!k_develop_f4(bllen, ptxt, cpa_blen,cpa_bl, keylen, lk, laps, 1))
       {
           throwout(_("Fehler a3 in k_develop_f4"));
            free((char *)pertxt);
            free((char *)lk);
            return false;
       }
        break;
    case 2:
  //the dead fleas x2 routine
       if(!k_develop_f4(bllen, ptxt, cpa_blen,cpa_bl, keylen, lk, laps, 2))
       {
           throwout(_("Fehler a4 in k_develop_f4"));
            free((char *)pertxt);
            free((char *)lk);
            return false;
       }
       break;
    case 3:
  //the dead fleas x5 routine
       if(!k_develop_f4(bllen, ptxt, cpa_blen,cpa_bl, keylen, lk, laps, 3))
       {
           throwout(_("Fehler a5 in k_develop_f4"));
            free((char *)pertxt);
            free((char *)lk);
            return false;
       }
        break;
    case 4:
  //the fleas o2 routine
       if(!k_develop_f4(bllen, ptxt, cpa_blen,cpa_bl, keylen, lk, laps, 4))
       {
           throwout(_("Fehler a6 in k_develop_f4"));
            free((char *)pertxt);
            free((char *)lk);
            return false;
       }
        break;
    case 5:
//the fleas o5 routine
       if(!k_develop_f4(bllen, ptxt, cpa_blen,cpa_bl, keylen, lk, laps, 5))
       {
           throwout(_("Fehler a7 in k_develop_f4"));
            free((char *)pertxt);
            free((char *)lk);
            return false;
       }
    break;
    case 6:
  //the fast and good fleas lean routine
       if(!k_develop_f4(bllen, ptxt, cpa_blen,cpa_bl, 0, lk, laps, 6))
       {
           throwout(_("Fehler a8 in k_develop_f4"));
            free((char *)pertxt);
            free((char *)lk);
            return false;
       }
    break;
    case 7:
 //the fleas (a little bit obsolete) lean-supersafe routine Fleas_ls
       if(!k_develop_f4(bllen, ptxt, cpa_blen,cpa_bl, keylen, lk, laps, 7))
       {
           throwout(_("Fehler a9 in k_develop_f4"));
            free((char *)pertxt);
            free((char *)lk);
            return false;
       }
    break;
    //12 reserved for aes
    case 8:
  //the fleas (a little bit obsolete) lean-supersafe 3 path routine (für Hash: Fleas_lx3) und cipher l3
       if(!k_develop_f4(bllen, ptxt, cpa_blen,cpa_bl, keylen, lk, laps, 8))
       {
           throwout(_("Fehler a10 in k_develop_f4"));
            free((char *)pertxt);
            free((char *)lk);
            return false;
       }
    break;
    case 9:
 //the fleas lean true 3 path-supersafe routine Fleas_lc
    case 20: //also chimera default
      if(!k_develop_f4(bllen, ptxt, cpa_blen,cpa_bl, keylen, lk, laps, 9))
       {
           throwout(_("Fehler a11 in k_develop_f4"));
            free((char *)pertxt);
            free((char *)lk);
            return false;
       }
    break;
    case 10:
    case 13:
 //the fleas lean-4-path-supersafe routine Fleas_ld
        if(!k_develop_f4(bllen, ptxt, cpa_blen,cpa_bl, keylen, lk, laps, 10))
       {
           throwout(_("Fehler a12 in k_develop_f4"));
            free((char *)pertxt);
            free((char *)lk);
            return false;
       }
    break;
    case 11:
 //the fleas lean-2-path routine Fleas_lb
        if(!k_develop_f4(bllen, ptxt, cpa_blen,cpa_bl, keylen, lk, laps, 11))
       {
           throwout(_("Fehler a13 in k_develop_f4"));
            free((char *)pertxt);
            free((char *)lk);
            return false;
       }
    break;
    case 15:
 //the fleas super lean-2-path routine flight
        if(!k_develop_f4(bllen, ptxt, cpa_blen,cpa_bl, keylen, lk, laps, 15))
       {
           throwout(_("Fehler a14 in k_develop_f4"));
            free((char *)pertxt);
            free((char *)lk);
            return false;
       }
    break;
    case 16:
 //the fleas optimized super lean-2-path routine flightx
        if(!k_develop_f4(bllen, ptxt, cpa_blen,cpa_bl, keylen, lk, laps, 16))
       {
           throwout(_("Fehler a15 in k_develop_f4"));
            free((char *)pertxt);
            free((char *)lk);
            return false;
       }
    break;
    default:
        {
            free((char *)pertxt);
            free((char *)lk);
            throwout(_("Fehler: Unbekannte Algorithmus id in roundfun!"));
            return false;
        }
  }
  if(pertxt != NULL) free((char *)pertxt);
  if(lk != NULL) free((char *)lk);
  return true;
}
/**************************************************************/
/******************************************/
/******************************************/
bool getbit(char* blad, ulong32 ind)
//Funktion gibt den Wert des ind´ten bits zurück
{
  ulong32 offset;
  unsigned char bitoff,mask;

  offset = ind >> 3;
  bitoff = ((unsigned char) ind)&7;
  mask = 1>>bitoff;
  if((mask & ((unsigned char) *(blad+offset))) >0) return TRUE;
  return FALSE;
}
/******************************************/
bool flipbit(char* blad, ulong32 ind)
//Funktion flipt den Wert des ind´ten bits
// returned neuen Wert
{
  ulong32 offset;
  unsigned char bitoff,mask;
  bool retval=FALSE;

  offset = ind >> 3;
  bitoff = ((unsigned char) ind)&7;
  mask = 1>>bitoff;
  if((mask & *(blad+offset)) <0) retval= TRUE;
  *(blad+offset) ^= mask;
  return retval;
}
/******************************************/
void setbit(char* blad, ulong32 ind, bool val)
//Funktion setzt den Wert des ind´ten bits
{
  ulong32 offset;
  unsigned char bitoff,mask;

  offset = ind >> 3;
  bitoff = ((unsigned char) ind)&7;
  mask = 1>>bitoff;
  if(val) *(blad+offset) |= mask;
  else *(blad+offset) &= ~mask;
}
/******************************************/
void recodebits(char* blad, ulong32 bitno)
//Funktion recodiert bis bitno nach:
//1.bit bleibt, unverändert gegen letztes ->1 (sonst 0)
{
  ulong32 index;
  bool lastval,currval=true;
  lastval=getbit(blad,0);
  for(index=1;index<=bitno;index++)
  {
    currval=getbit(blad,index);
    if(currval == lastval)
         flipbit(blad,index);
    lastval=currval;
  }
  if(currval==getbit(blad,0))flipbit(blad,0);
}
/****************************************/
unsigned int getnum(char* blad, ulong32 maxbytes, ulong32 indx,char bitzahl)
//holt eine referenzzahl durch wildes auslesen im bitraster
{
  unsigned int result,mask;
  char i;
  indx= indx%(maxbytes*8);
  mask=1; result=0;
  if(bitzahl>16)bitzahl=16; //Fehlerkorrektur

  for (i=0;i<bitzahl;i++)
  {
    if (getbit(blad,indx+(ulong32)i))
           result |= mask;
    mask >>= 1;
  }
  return result;
}
/******************************************/
/***********************************/
char konv_lett_to_hex(unsigned char lett)
{
   switch(lett)
     {
         case 70: //F
          return 15;
        case 102: //f
          return 15;
        case 'E': // E
          return 14;
        case 'e':
          return 14;
        case 'D' :// D
          return 13;
        case 'd':
          return 13;
        case 'C' :// C
          return 12;
        case 'c':
          return 12;
        case 'B' :// B
          return 11;
        case 'b':
          return 11;
        case 'A': // A
          return 10;
        case 'a':
          return 10;
        case '9' :
          return 9;
        case '8':
          return 8;
        case '7' :
          return 7;
        case '6':
          return 6;
        case '5' :
          return 5;
        case '4':
          return 4;
        case '3' :
          return 3;
        case '2':
          return 2;
        case '1' :
          return 1;
        case '0':
          return 0;
        default:
          return -1;
    }
}
/***************************************************/
int read_hex_bytes_from_file(FILE* fp, unsigned char* buffer, int maxbytes)
// maximal maxbytes leerzeichengetrennte Hex-bytes von einem File lesen und in den buffer hineinschreiben
//returnwert: gelesene bytes
{
  int k,i=0;
  unsigned char bf[2], a_byt, r_byt;
  int fl1;
  char a1;

  while(i<maxbytes)
  {
     fl1= fscanf(fp," %2s", bf);
     if(fl1 != 1) //lesevorgang nicht erfolgreich
     {
         break;
     }

     // jetzt umwandeln
     a1 = konv_lett_to_hex(bf[0]);
     if(a1 <0 ) break;
     a_byt=(unsigned char) a1;
     r_byt=a_byt<<4;
     a1 = konv_lett_to_hex(bf[1]);
     if(a1<0 ) break;
     r_byt+=(unsigned char)a1;
     *(buffer+i)= r_byt;
     i++;
  }
  //invert buffer
  for(k=0;k<=(i-1)/2;k++)
  {
      a_byt=*(buffer+(i-1)-k);
      *(buffer+(i-1)-k)= *(buffer+k);
      *(buffer+k)=a_byt;
  }

  return i;
}
/*****************************************************/
bool encipher_file_cntmode(char* fnam, char* key, int keylen, int k_steps, int blolen, int mode, wxString algo, unsigned char shoprog)
//k_steps: number of owf-steps,   mode:0-clear number, 1number of lonulock steps for number
//
{
  long long filen=0, blockzahl, j;
  FILE *plainfile, *cipherfile;
  char *sfnam, *iv, *ic,*ip,hc;
  int i,algoflag,ichar, currpos,lastmax=0;
  longnumber ivno;
  bool goodp=true;

//check new mt version now
 if( (algo== _("F_cnt_1c"))|| (algo== _("F_cnt_1d"))) return encipher_file_cntmode_mt( fnam, key, keylen, k_steps, blolen, mode, algo, shoprog);
 if( algo == _("threefish")) return encipher_file_threefish( fnam, key, keylen, k_steps, blolen, mode, algo, shoprog);
 if(algo == _("chimera")) return encipher_file_fleafish( fnam, key, keylen, k_steps, blolen, mode, algo, shoprog);
    //some primitive error checking
  if(strlen(fnam) > 398) return false;
  wxString fnwx(fnam, wxConvFile);
  if(!wxFile::Exists(fnwx))
  {
      throwout(_("Datei existiert nicht!!\nAbbruch."),2);
      return false;
  }
  //determine Length
  wxFile tmpfl(fnwx);
  filen=tmpfl.Length();
  tmpfl.Close();
  if((plainfile = fopen(fnam, "rb"))== NULL) return false;
  // jetzt cipherfile öffnen mit anhängsel "_1c" etc und setze algoflag
    //create new sfnam "shadow fnam" buffer
    sfnam=(char *)malloc(strlen(fnam)+10);
    strcpy(sfnam,fnam);
  if(algo ==_("F_cnt_1c"))   {strncat(sfnam,"_1c",3); algoflag=9;}
  else if(algo ==_("F_cnt_1d"))   {strncat(sfnam,"_1d",3); algoflag=10;}
  else if(algo ==_("F_cnt_1b"))   {strncat(sfnam,"_1b",3); algoflag=11;}
   else if(algo ==_("flight"))   {strncat(sfnam,"_fl",3); algoflag=15;}
    else if(algo ==_("flightx"))   {strncat(sfnam,"_fx",3); algoflag=16;}
     /*else if(algo ==_("aes"))   {strncat(sfnam,"_ae",3);blolen=8;algoflag=12;}*/
       else{throwout(_("Unbek. Algorithmus! \nAbbruch (0)"));free(sfnam);return false;}
  if((cipherfile = fopen(sfnam, "wb"))== NULL)
  {
    free(sfnam);
    return false;
  }
  free(sfnam);

//initialize Progress bar
  wxProgressDialog pbar(_("Verschl.-Fortschritt"),_("Computer arbeitet hart :-)"),100);
  //determine blockzahl
  blockzahl=filen/blolen;
  //create IV from random then lock with key
  iv=(char *) malloc(blolen);
  ic=(char *) malloc(blolen);
  ip=(char *) malloc(blolen);
  zuf.push_bytes(blolen, (unsigned char *)iv);
  wxString primalgo;
  if(algoflag==9) primalgo=_("Fleas_lc");
  else if(algoflag==10) primalgo=_("Fleas_ld");
  else if(algoflag==11) primalgo=_("Fleas_lb");
  else if(algoflag==15) primalgo=_("flight");
  else if(algoflag==16) primalgo=_("flightx");
  else if(algoflag==17) primalgo=_("flightx");

  else throwout(_("Fehler! \nunbekannter Primaeralgorithmus in countermode Verschluesslg."));


  if(mode>0)
  {
      dofeistel(blolen, keylen, iv,ic, key, 4, mode, 0, NULL, primalgo);//4Steps, mode-fold owf, no cpa blocker
  }
  else memcpy(ic,iv,blolen);
  //write locked IV
  for(i=0;i<blolen;i++)
  {
      fputc( *(ic + i),cipherfile) ;
  }
  //put IV into longnumber
  ivno.resizelonu(blolen+2);
  memcpy(ivno.ad, iv, blolen);
  ivno.setsize(true);
  //loop complete blocks with progbar update
  for(j=0;j<blockzahl;j++)
  {
       if((shoprog>0)&&(blockzahl>1))
       {
         //prog bar propagation in main loop
         currpos=(int)( (100*j)/(blockzahl-1));
         if((currpos >lastmax)&&goodp)
         {
            goodp=pbar.Update(currpos);
            lastmax=currpos;
         }
       }
      //read in plaintext block
      for(i=0;i<blolen;i++)
      {
          *(ip + i)= (char) fgetc(plainfile);
      }
      //get key-developed number block from ivno
      memcpy(ic,(char *)(ivno.ad),blolen);
      k_develop_f4(blolen, ic,0,NULL,keylen,key,k_steps,algoflag); //9 ->Fleas_lc 10 ->fleas ld
      //xor with plaintext block
      //write to cipher file
      for(i=0;i<blolen;i++)
      {

          hc=*(ic + i);
          hc ^= *(ip+i);
          fputc( hc,cipherfile) ;
      }
      //increment ivno
      ivno.inc();
  }
  //get rest block from plainfile
  //get key-developed number block from ivno
  //xor bytewise and write to cipherfile
  memcpy(ic,(char *)(ivno.ad),blolen);
  k_develop_f4(blolen, ic,0,NULL,keylen,key,k_steps,algoflag); //9 ->Fleas_lc 10 ->fleas ld
  i=0;
  do
  {
      ichar=fgetc(plainfile);
      if(ichar==EOF) break;
      hc=(char) ichar;
      hc ^= *(ic + i);
      fputc( hc,cipherfile) ;
      i++;
  } while(ichar != EOF);
  //close files
  fclose(cipherfile);
  fclose(plainfile);
    //free buffers
    free(iv); free(ip); free(ic);
  return true;
}
/*****************************************************/
bool encipher_file_threefish(char* fnam, char* key, int keylen, int k_steps, int blolen, int mode, wxString algo, unsigned char shoprog)
//k_steps: number of owf-steps,   mode:0-clear number, 1number of lonulock steps for number
//
{
  long long filen=0, blockzahl, j;
  FILE *plainfile, *cipherfile;
  char *sfnam, *iv, *ic,*ip,hc, *thrfkey;
  int i,algoflag,ichar, currpos,lastmax=0;
  longnumber ivno;
  bool goodp=true;
    // Threefish cipher context data
    ThreefishKey_t keyCtx;

//check new mt version now
 if (algo!= _("threefish")) { throwout(_("error 1 in threefish call"), 10); return false;}
 if (blolen!= 128) { throwout(_("blocklength error  in threefish call"), 10); return false;}
 algoflag=17;

  //some primitive error checking
  if(strlen(fnam) > 398) return false;
  wxString fnwx(fnam, wxConvFile);
  if(!wxFile::Exists(fnwx))
  {
      throwout(_("Datei existiert nicht!!\nAbbruch."),2);
      free(thrfkey);
      return false;
  }
  //determine Length
  wxFile tmpfl(fnwx);
  filen=tmpfl.Length();
  tmpfl.Close();
  if((plainfile = fopen(fnam, "rb"))== NULL) return false;
  // jetzt cipherfile öffnen mit anhängsel "_1c" etc und setze algoflag
    //create new sfnam "shadow fnam" buffer
    sfnam=(char *)malloc(strlen(fnam)+10);
    strcpy(sfnam,fnam);
    strncat(sfnam,"_tf",3);
  if((cipherfile = fopen(sfnam, "wb"))== NULL)
  {
    free(sfnam);
    return false;
  }
  free(sfnam);
  // reform key to 128 byte
  thrfkey=develop_o_malloc(keylen,key,2,2,144);
  threefishSetKey(&keyCtx, Threefish1024, (uint64_t *)thrfkey, (uint64_t *)(thrfkey+128));  //use first 128 byte of key as key, next 16 as tweak

//initialize Progress bar
  wxProgressDialog pbar(_("Verschluesselungs-Fortschritt"),_("Computer arbeitet hart :-)"),100);
  //determine blockzahl
  blockzahl=filen/blolen;
  //create IV from random then lock with key
  iv=(char *) malloc(blolen);
  ic=(char *) malloc(blolen);
  ip=(char *) malloc(blolen);
  zuf.push_bytes(blolen, (unsigned char *)iv);
  //wxString primalgo;
  //primalgo=_("Fleas_lc");

  if(mode>0)
  {
      //dofeistel(blolen, 128, iv,ic, thrfkey, 4, mode, 0, NULL, primalgo);//4Steps, mode-fold owf, no cpa blocker
        threefishEncryptBlockBytes(&keyCtx, (uint8_t *)iv, (uint8_t *) ic);
  }
  else memcpy(ic,iv,blolen);
  //write locked IV
  for(i=0;i<blolen;i++)
  {
      fputc( *(ic + i),cipherfile) ;
  }
  //put IV into longnumber
  ivno.resizelonu(blolen+2);
  memcpy(ivno.ad, iv, blolen);
  ivno.setsize(true);
  ivno.inc();
  //loop complete blocks with progbar update
  for(j=0;j<blockzahl;j++)
  {
       if((shoprog>0)&&(blockzahl>1))
       {
         //prog bar propagation in main loop
         currpos=(int)( (100*j)/(blockzahl-1));
         if((currpos >lastmax)&&goodp)
         {
            goodp=pbar.Update(currpos);
            lastmax=currpos;
         }
       }
      //read in plaintext block
      for(i=0;i<blolen;i++)
      {
          *(ip + i)= (char) fgetc(plainfile);
      }
      //get key-developed number block from ivno
      threefishEncryptBlockBytes(&keyCtx, (uint8_t *)(ivno.ad), (uint8_t *) ic);
      //xor with plaintext block
      //write to cipher file
      for(i=0;i<blolen;i++)
      {

          hc=*(ic + i);
          hc ^= *(ip+i);
          fputc( hc,cipherfile) ;
      }
      //increment ivno
      ivno.inc();
  }
  //get rest block from plainfile
  //get key-developed number block from ivno
  //xor bytewise and write to cipherfile
  memcpy(ip,ivno.ad,blolen);
  threefishEncryptBlockBytes(&keyCtx, (uint8_t *) ip, (uint8_t *) ic);
  i=0;
  do
  {
      ichar=fgetc(plainfile);
      if(ichar==EOF) break;
      hc=(char) ichar;
      hc ^= *(ic + i);
      fputc( hc,cipherfile) ;
      i++;
  } while(ichar != EOF);
  //close files
  fclose(cipherfile);
  fclose(plainfile);
    //free buffers
    free(iv); free(ip); free(ic); free(thrfkey);
  return true;
}
/******************************************************/
/*****************************************************/
bool encipher_file_fleafish(char* fnam, char* key, int keylen, int k_steps, int blolen, int mode, wxString algo, unsigned char shoprog)
//k_steps: number of owf-steps,   mode:0-clear number, 1number of lonulock steps for number
//
{
  long long filen=0, blockzahl, j;
  FILE *plainfile, *cipherfile;
  char *sfnam, *iv, *ic, *ffic,*ip,hc, *thrfkey, *flightxkey, *flightxblock;
  int i,algoflag,ichar, currpos,lastmax=0;
  longnumber ivno,flivno;
  bool goodp=true;
    // Threefish cipher context data
    ThreefishKey_t keyCtx;

//check new mt version now
 if (algo!= _("chimera")) { throwout(_("error 1 in fleafish call"), 10); return false;}
 if (blolen!= 512) { throwout(_("blocklength error  in fleafish call"), 10); return false;}
 algoflag=17;

  //some primitive error checking
  if(strlen(fnam) > 398) return false;
  wxString fnwx(fnam, wxConvFile);
  if(!wxFile::Exists(fnwx))
  {
      throwout(_("Datei existiert nicht!!\nAbbruch."),2);
      return false;
  }
  //determine Length
  wxFile tmpfl(fnwx);
  filen=tmpfl.Length();
  tmpfl.Close();
  if((plainfile = fopen(fnam, "rb"))== NULL) return false;
  // jetzt cipherfile öffnen mit anhängsel "_1c" etc und setze algoflag
    //create new sfnam "shadow fnam" buffer
    sfnam=(char *)malloc(strlen(fnam)+10);
    if(sfnam == NULL){throwout(_(" failed malloc in enc fleafish"));return false;}
    strcpy(sfnam,fnam);
    strncat(sfnam,"_ff",3);
  if((cipherfile = fopen(sfnam, "wb"))== NULL)
  {
    free(sfnam);
    return false;
  }
  free(sfnam);
  // reform key to 128 byte
  thrfkey=develop_o_malloc(keylen,key,2,2,144);
  if(thrfkey==NULL) {throwout(_(" failed malloc in enc fleafish(2)"));return false;}
  threefishSetKey(&keyCtx, Threefish1024, (uint64_t *)thrfkey, (uint64_t *)(thrfkey+128));  //use first 128 byte of key as key, next 16 as tweak
  flightxkey=develop_o_malloc(keylen,key,3,2,512);
  if(flightxkey==NULL) {free(thrfkey); throwout(_(" failed malloc in enc fleafish(3)"));return false;}
//initialize Progress bar
  wxProgressDialog pbar(_("Verschluesselungs-Fortschritt"),_("Computer arbeitet hart :-)"),100);
  //determine blockzahl
  blockzahl=filen/blolen;
  //create IV from random then lock with key
  iv=(char *) malloc(blolen);
    if(iv==NULL)
      { free(thrfkey);free(flightxkey); return false;}
  ic=(char *) malloc(blolen);
      if(ic==NULL)
      { free(iv); free(thrfkey);free(flightxkey); return false;}
  ffic=(char *) malloc(blolen);
      if(ffic==NULL)
      { free(ic); free(iv); free(thrfkey);free(flightxkey); return false;}
  ip=(char *) malloc(blolen);
      if(ip==NULL)
      { free(ffic); free(ic); free(iv); free(thrfkey); free(flightxkey);return false;}
  flightxblock= (char *) malloc(blolen+512);
      if(flightxblock==NULL)
      { free(ip); free(ffic); free(ic); free(iv); free(thrfkey);free(flightxkey); return false;}

  zuf.push_bytes(blolen, (unsigned char *)iv);
  //wxString primalgo;
  //primalgo=_("Fleas_lc");

  if(mode>0)
  {
      dofeistel(blolen, 512, iv,ic, flightxkey, 4, mode, 0, NULL, _("flightx"));//4Steps, mode-fold owf, no cpa blocker
  }
  else memcpy(ic,iv,blolen);
  //write locked IV
  for(i=0;i<blolen;i++)
  {
      fputc( *(ic + i),cipherfile) ;
  }
  //put IV into longnumber
  ivno.resizelonu(128+2);
  flivno.resizelonu(blolen+2);
  memcpy(ivno.ad, iv, 128);
  memcpy(flivno.ad, iv, blolen);

  ivno.setsize(true);
  ivno.inc();
  flivno.setsize(true);
  flivno.inc();
  //loop complete blocks with progbar update
  for(j=0;j<blockzahl;j++)
  {
       if((shoprog>0)&&(blockzahl>1))
       {
         //prog bar propagation in main loop
         currpos=(int)( (100*j)/(blockzahl-1));
         if((currpos >lastmax)&&goodp)
         {
            goodp=pbar.Update(currpos);
            lastmax=currpos;
         }
       }
      //read in plaintext block
      for(i=0;i<blolen;i++)
      {
          *(ip + i)= (char) fgetc(plainfile);
      }
      //get key-developed number block from ivno
      threefishEncryptBlockBytes(&keyCtx, (uint8_t *)(ivno.ad), (uint8_t *) ic);
      ivno.inc();
      threefishEncryptBlockBytes(&keyCtx, (uint8_t *)(ivno.ad), (uint8_t *) (ic+128));
      ivno.inc();
      threefishEncryptBlockBytes(&keyCtx, (uint8_t *)(ivno.ad), (uint8_t *) (ic+256));
      ivno.inc();
      threefishEncryptBlockBytes(&keyCtx, (uint8_t *)(ivno.ad), (uint8_t *) (ic+384));
      ivno.inc();
      //now develop flightx-half
      //fflivno.ad nach ffic mit flightxkey
      memcpy(flightxblock,flightxkey,512); //copy
      memcpy(flightxblock+512,flivno.ad,512); //copy
      develop_flightx_onearm( flightxblock, blolen+512, 1);
      flivno.inc();
      //pick center section
      memcpy(ffic,flightxblock+(blolen+512)/2,512);
      //xor with plaintext block
      //write to cipher file
      for(i=0;i<blolen;i++)
      {
          hc=*(ic + i);
          hc ^= *(ip+i);
          hc ^= *(ffic+i);
          fputc( hc,cipherfile) ;
      }
      //increment ivno
  }
  //get rest block from plainfile
  //get key-developed number block from ivno
  //xor bytewise and write to cipherfile
  threefishEncryptBlockBytes(&keyCtx, (uint8_t *) ivno.ad, (uint8_t *) ic);
      ivno.inc();
  threefishEncryptBlockBytes(&keyCtx, (uint8_t *)(ivno.ad), (uint8_t *) (ic+128));
      ivno.inc();
  threefishEncryptBlockBytes(&keyCtx, (uint8_t *)(ivno.ad), (uint8_t *) (ic+256));
      ivno.inc();
  threefishEncryptBlockBytes(&keyCtx, (uint8_t *)(ivno.ad), (uint8_t *) (ic+384));
      ivno.inc();
  i=0;
      memcpy(flightxblock,flightxkey,512); //copy
      memcpy(flightxblock+512,flivno.ad,512); //copy
      develop_flightx_onearm( flightxblock, blolen+512, 1);
      flivno.inc();
      //pick center section
      memcpy(ffic,flightxblock+(blolen+512)/2,512);
  do
  {
      ichar=fgetc(plainfile);
      if(ichar==EOF) break;
      hc=(char) ichar;
      hc ^= *(ic + i);
      hc ^= *(ffic+i);
      fputc( hc,cipherfile) ;
      i++;
  } while(ichar != EOF);
  //close files
  fclose(cipherfile);
  fclose(plainfile);
    //free buffers
    free(iv); free(ip); free(ic); free(thrfkey); free(ffic); free(flightxkey); free(flightxblock);
  return true;
}
/******************************************************//******************************************************/
bool encipher_cnt_pair(wxString appendto, wxString header1, wxString plainwxfile, char* key, int keylen, int k_steps, int blolen,int mode, wxString algo,unsigned char shoprog)
{
  long long filen=0, blockzahl, j,headlen,plainlen, readbyts=0;
  FILE *plainfile, *cipherfile, *inner_hdfile;
  char *cfnam, *fnam,*hfnam, *iv, *ic,*ic1,*ic2,*ic3,*ip,hc;
  int i,algoflag,ichar, currpos,lastmax=0;
  longnumber ivno;
  bool goodp=true;
  kdev_thr *pt0, *pt1,*pt2,*pt3;
  bool r0,r1,r2,r3;

  //some primitive error checking
  if(!wxFile::Exists(appendto))
  {
      throwout(_("File to append cipher to does not exist!!\nAborting."),2);
      return false;
  }
  if(!wxFile::Exists(header1))
  {
      throwout(_("File for first level header does not exist!!\nAborting."),2);
      return false;
  }
  if(!wxFile::Exists(plainwxfile))
  {
      throwout(_("Plainfile does not exist!!\nAborting."),2);
      return false;
  }
  //determine Lengths
  wxFile tmpfl(header1);
  headlen=tmpfl.Length();
  tmpfl.Close();
  wxFile tmpfl2(plainwxfile);
  plainlen=tmpfl2.Length();
  tmpfl2.Close();
  filen= headlen+plainlen;
  //create all char filenames
  cfnam= (char *) malloc(4*appendto.Length()+2);
  if(cfnam==NULL)
  {
      throwout(_("error! Could not allocate filename buffer?"),2);
      return false;
  }
  strcpy( cfnam, (const char*)appendto.mb_str(wxConvLocal) );// schreibt den Filenamen in fnambuff rein

  fnam= (char *) malloc(4*plainwxfile.Length()+2);
  if(fnam==NULL)
  {
      throwout(_("error! Could not allocate plain-filename buffer?"),2);
      free(cfnam);
      return false;
  }
  strcpy( fnam, (const char*)plainwxfile.mb_str(wxConvLocal) );// schreibt den Filenamen in fnambuff rein

  hfnam= (char *) malloc(4*header1.Length()+2);
  if(hfnam==NULL)
  {
      throwout(_("error! Could not allocate header1-filename buffer?"),2);
      free(cfnam);free(fnam);
      return false;
  }
  strcpy( hfnam, (const char*)header1.mb_str(wxConvLocal) );// schreibt den Filenamen in fnambuff rein
//open all files
  if((cipherfile = fopen(cfnam, "ab"))== NULL)
  {
      free(cfnam);free(fnam);free(hfnam);
      throwout(_("could not open cipherfile"),2);
      return false;
  }
  if((plainfile = fopen(fnam, "rb"))== NULL)
  {
      fclose(cipherfile);
      free(cfnam);free(fnam);free(hfnam);
      throwout(_("could not open plainfile"),2);
      return false;
  }
  if((inner_hdfile = fopen(hfnam, "rb"))== NULL)
  {
      fclose(cipherfile);fclose(plainfile);
      free(cfnam);free(fnam);free(hfnam);
      throwout(_("could not open header_1_file"),2);
      return false;
  }
  if(algo ==_("F_cnt_1c"))   { algoflag=9;}
  else if(algo ==_("F_cnt_1d"))   { algoflag=13;}
  else if(algo ==_("F_cnt_1b"))   { algoflag=11;}
  else if(algo ==_("flight"))   { algoflag=15;}
  else if(algo ==_("flightx"))   { algoflag=16;}
  else if(algo ==_("chimera"))   { algoflag=20;}

//initialize Progress bar
  wxProgressDialog pbar(_("Verschluesselungs-Fortschritt"),_("Computer arbeitet hart :-)"),100);
  //determine blockzahl
  blockzahl = filen/blolen;
  //create IV from random then lock with key
  iv=(char *) malloc(blolen);
  ic=(char *) malloc(blolen);
  ic1=(char *) malloc(blolen);
  ic2=(char *) malloc(blolen);
  ic3=(char *) malloc(blolen);
  ip=(char *) malloc(blolen);
  zuf.push_bytes(blolen, (unsigned char *)iv);
  wxString primalgo;
  if(algoflag==9) primalgo=_("Fleas_lc");
  else if(algoflag==13) primalgo=_("Fleas_ld");
  else if(algoflag==11) primalgo=_("Fleas_lb");
  else if(algoflag==15) primalgo=_("flight");
  else if(algoflag==16) primalgo=_("flightx");
  else if(algoflag==20) primalgo=_("chimera");

  else throwout(_("Fehler! \nUnbekannter Primaeralgorithmus in countermode Verschluesselg."));
  if(mode>0)
  {
      dofeistel_n(blolen, keylen, iv,ic, key, 4, mode, 0, NULL, (unsigned int)algoflag);//4Steps, mode-fold owf, no cpa blocker
  }
  else memcpy(ic,iv,blolen);
  //write locked IV
  for(i=0;i<blolen;i++)
  {
      fputc( *(ic + i),cipherfile) ;
  }
  //put IV into longnumber
  ivno.resizelonu(blolen+2);
  memcpy(ivno.ad, iv, blolen);
  ivno.setsize(true);
  //loop complete blocks with progbar update

  for(j=0;j+4<blockzahl;j+=4)
  {
       if((shoprog>0)&&(blockzahl>1))
       {
         //prog bar propagation in main loop
         currpos=(int)( (100*j)/(blockzahl-1));
         if((currpos >lastmax)&&goodp)
         {
            goodp=pbar.Update(currpos);
            lastmax=currpos;
         }
       }
      //get key-developed number block from ivno
      memcpy(ic,(char *)(ivno.ad),blolen);
      //do it threaded
      //k_develop_f4(blolen, ic,0,NULL,keylen,key,k_steps,algoflag); //14 ->Fleas_lc 15 ->fleas ld
        pt0=new kdev_thr(&r0,blolen, ic,0,NULL,keylen,key,k_steps,algoflag);
        if ( pt0->Create() != wxTHREAD_NO_ERROR )
            {
                throwout(_("Can't create the thread!"),3);
                delete pt0;
                pt0=NULL;
            }
        else
          {
            if ( pt0->Run() != wxTHREAD_NO_ERROR )
            {
                throwout(_("Can't Run the thread!"),3);
                delete pt0;
                pt0 = NULL;
            }
          }
          if(pt0==NULL)  //error  go to harakiri
          {
               //close files
                fclose(cipherfile);
                fclose(plainfile);
                fclose(inner_hdfile);
                //free buffers
                free(iv); free(ip); free(ic);free(ic1);free(ic2);free(ic3);
                free(cfnam);free(hfnam);free(fnam);
                throwout(_("error in encipher file cntmode multithread t0"),3);
                return false;
          }


      ivno.inc();
      memcpy(ic1,(char *)(ivno.ad),blolen);
      //k_develop_f4(blolen, ic1,0,NULL,keylen,key,k_steps,algoflag); //14 ->Fleas_lc 15 ->fleas ld
        pt1=new kdev_thr(&r1,blolen, ic1,0,NULL,keylen,key,k_steps,algoflag);
        if ( pt1->Create() != wxTHREAD_NO_ERROR )
            {
                throwout(_("Can't create the thread!"),3);
                delete pt1;
                pt1=NULL;
            }
        else
          {
            if ( pt1->Run() != wxTHREAD_NO_ERROR )
            {
                throwout(_("Can't Run the thread!"),3);
                delete pt1;
                pt1 = NULL;
            }
          }
          if(pt1==NULL)  //error  go to harakiri
          {
               pt0->Wait();
               //close files
                fclose(cipherfile);
                fclose(plainfile);
                //free buffers
                free(iv); free(ip); free(ic);free(ic1);free(ic2);free(ic3);
                fclose(inner_hdfile);
                free(cfnam);free(hfnam);free(fnam);
                throwout(_("error in encipher file cntmode multithread t1"),3);
                return false;
          }

      ivno.inc();
      memcpy(ic2,(char *)(ivno.ad),blolen);
      //k_develop_f4(blolen, ic2,0,NULL,keylen,key,k_steps,algoflag); //14 ->Fleas_lc 15 ->fleas ld
        pt2=new kdev_thr(&r2,blolen, ic2,0,NULL,keylen,key,k_steps,algoflag);
        if ( pt2->Create() != wxTHREAD_NO_ERROR )
            {
                throwout(_("Can't create the thread!"),3);
                delete pt2;
                pt2=NULL;
            }
        else
          {
            if ( pt2->Run() != wxTHREAD_NO_ERROR )
            {
                throwout(_("Can't Run the thread!"),3);
                delete pt2;
                pt2 = NULL;
            }
          }
          if(pt2==NULL)  //error  go to harakiri
          {
               pt0->Wait();
               pt1->Wait();
               //close files
                fclose(cipherfile);
                fclose(plainfile);
                fclose(inner_hdfile);
                //free buffers
                free(iv); free(ip); free(ic);free(ic1);free(ic2);free(ic3);
                free(cfnam);free(hfnam);free(fnam);
                throwout(_("error in encipher file cntmode multithread t1"),3);
                return false;
          }

      ivno.inc();
      memcpy(ic3,(char *)(ivno.ad),blolen);
      //k_develop_f4(blolen, ic3,0,NULL,keylen,key,k_steps,algoflag); //14 ->Fleas_lc 15 ->fleas ld
        pt3=new kdev_thr(&r3,blolen, ic3,0,NULL,keylen,key,k_steps,algoflag);
        if ( pt3->Create() != wxTHREAD_NO_ERROR )
            {
                throwout(_("Can't create the thread!"),3);
                delete pt3;
                pt3=NULL;
            }
        else
          {
            if ( pt3->Run() != wxTHREAD_NO_ERROR )
            {
                throwout(_("Can't Run the thread!"),3);
                delete pt3;
                pt3 = NULL;
            }
          }
          if(pt3==NULL)  //error  go to harakiri
          {
               pt0->Wait();
               pt1->Wait();
               pt2->Wait();
               //close files
                fclose(cipherfile);
                fclose(plainfile);
                //free buffers
                fclose(inner_hdfile);
                free(cfnam);free(hfnam);free(fnam);
                free(iv); free(ip); free(ic);free(ic1);free(ic2);free(ic3);
                throwout(_("error in encipher file cntmode multithread t1"),3);
                return false;
          }
      //xor with plaintext block
      //write to cipher file
      //increment ivno
      ivno.inc();
//wait for proc0 to finish
       pt0->Wait();
      //read in plaintext block
      for(i=0;i<blolen;i++)
      {
          if(readbyts< headlen)
          {
             *(ip + i)= (char) fgetc(inner_hdfile);
            readbyts++;
          }
          else
          {
             *(ip + i)= (char) fgetc(plainfile);
            readbyts++;
          }
          hc=*(ic + i);
          hc ^= *(ip+i);
          fputc( hc,cipherfile) ;
      }
//wait for proc1 to finish
       pt1->Wait();
      //read in plaintext block
      for(i=0;i<blolen;i++)
      {
          if(readbyts< headlen)
          {
             *(ip + i)= (char) fgetc(inner_hdfile);
            readbyts++;
          }
          else
          {
             *(ip + i)= (char) fgetc(plainfile);
            readbyts++;
          }
          hc=*(ic1 + i);
          hc ^= *(ip+i);
          fputc( hc,cipherfile) ;
      }
//wait for proc2 to finish
       pt2->Wait();
      //read in plaintext block
      for(i=0;i<blolen;i++)
      {
          if(readbyts< headlen)
          {
             *(ip + i)= (char) fgetc(inner_hdfile);
            readbyts++;
          }
          else
          {
             *(ip + i)= (char) fgetc(plainfile);
            readbyts++;
          }
          hc=*(ic2 + i);
          hc ^= *(ip+i);
          fputc( hc,cipherfile) ;
      }
//wait for proc3 to finish
       pt3->Wait();
      //read in plaintext block
      for(i=0;i<blolen;i++)
      {
          if(readbyts< headlen)
          {
             *(ip + i)= (char) fgetc(inner_hdfile);
            readbyts++;
          }
          else
          {
             *(ip + i)= (char) fgetc(plainfile);
            readbyts++;
          }
          hc=*(ic3 + i);
          hc ^= *(ip+i);
          fputc( hc,cipherfile) ;
      }

  }
  //now do the rest in the traditional way
  // reached j was GE blockzahl -3
  for(;j<blockzahl;j++)
  {
       if((shoprog>0)&&(blockzahl>1))
       {
         //prog bar propagation in main loop
         currpos=(int)( (100*j)/(blockzahl-1));
         if((currpos >lastmax)&&goodp)
         {
            goodp=pbar.Update(currpos);
            lastmax=currpos;
         }
       }
      //read in plaintext block
      for(i=0;i<blolen;i++)
      {
          if(readbyts< headlen)
          {
             *(ip + i)= (char) fgetc(inner_hdfile);
            readbyts++;
          }
          else
          {
             *(ip + i)= (char) fgetc(plainfile);
            readbyts++;
          }
      }
      //get key-developed number block from ivno
      memcpy(ic,(char *)(ivno.ad),blolen);
      k_develop_f4(blolen, ic,0,NULL,keylen,key,k_steps,algoflag); //14 ->Fleas_lc 15 ->fleas ld 16 ->fleas ld
      //xor with plaintext block
      //write to cipher file
      for(i=0;i<blolen;i++)
      {

          hc=*(ic + i);
          hc ^= *(ip+i);
          fputc( hc,cipherfile) ;
      }
      //increment ivno
      ivno.inc();
  }

  //get rest block from plainfile
  //get key-developed number block from ivno
  //xor bytewise and write to cipherfile
  memcpy(ic,(char *)(ivno.ad),blolen);
  k_develop_f4(blolen, ic,0,NULL,keylen,key,k_steps,algoflag); //14 ->Fleas_lc 15 ->fleas ld
  i=0;
  do
  {
      if(readbyts< headlen)
      {
          ichar=fgetc(inner_hdfile);
          readbyts++;
      }
      else
      {
            ichar=fgetc(plainfile);
            readbyts++;
      }
      if(ichar==EOF) break;
      hc=(char) ichar;
      hc ^= *(ic + i);
      fputc( hc,cipherfile);
      i++;
  } while(ichar != EOF);
  //close files
  fclose(cipherfile);
  fclose(plainfile);
  fclose(inner_hdfile);
    //free buffers
  free(cfnam);free(hfnam);free(fnam);
  free(iv); free(ip); free(ic);free(ic1);free(ic2);free(ic3);
  return true;
}
/******************************************************/
bool encipher_cnt_pair_mono(wxString appendto, wxString header1, wxString plainwxfile, char* key, int keylen, int k_steps, int blolen,int mode, wxString algo,unsigned char shoprog)
{
  long long filen=0, blockzahl, j,headlen,plainlen, readbyts=0;
  FILE *plainfile, *cipherfile, *inner_hdfile;
  char *cfnam, *fnam,*hfnam, *iv, *ic,*ip,*ffic,*flightxkey, *flightxblock,hc,*thrfkey=NULL;
  int i,algoflag,ichar, currpos,lastmax=0;
  longnumber ivno,tfivno;
  bool goodp=true;



      // Extra for Threefish cipher context data
  ThreefishKey_t keyCtx;
  //some primitive error checking
  if(!wxFile::Exists(appendto))
  {
      throwout(_("File to append cipher to does not exist!!\nAborting."),2);
      return false;
  }
  if(!wxFile::Exists(header1))
  {
      throwout(_("File for first level header does not exist!!\nAborting."),2);
      return false;
  }
  if(!wxFile::Exists(plainwxfile))
  {
      throwout(_("Plainfile does not exist!!\nAborting."),2);
      return false;
  }
  //determine Lengths
  wxFile tmpfl(header1);
  headlen=tmpfl.Length();
  tmpfl.Close();
  wxFile tmpfl2(plainwxfile);
  plainlen=tmpfl2.Length();
  tmpfl2.Close();
  filen= headlen+plainlen;
  //create all char filenames
  cfnam= (char *) malloc(4*appendto.Length()+2);
  if(cfnam==NULL)
  {
      throwout(_("error! Could not allocate filename buffer?"),2);
      return false;
  }
  strcpy( cfnam, (const char*)appendto.mb_str(wxConvLocal) );// schreibt den Filenamen in fnambuff rein

  fnam= (char *) malloc(4*plainwxfile.Length()+2);
  if(fnam==NULL)
  {
      throwout(_("error! Could not allocate plain-filename buffer?"),2);
      free(cfnam);
      return false;
  }
  strcpy( fnam, (const char*)plainwxfile.mb_str(wxConvLocal) );// schreibt den Filenamen in fnambuff rein

  hfnam= (char *) malloc(4*header1.Length()+2);
  if(hfnam==NULL)
  {
      throwout(_("error! Could not allocate header1-filename buffer?"),2);
      free(cfnam);free(fnam);
      return false;
  }
  strcpy( hfnam, (const char*)header1.mb_str(wxConvLocal) );// schreibt den Filenamen in fnambuff rein
//open all files
  if((cipherfile = fopen(cfnam, "ab"))== NULL)
  {
      free(cfnam);free(fnam);free(hfnam);
      throwout(_("Konnte cipherfile nicht oeffnen"),2);
      return false;
  }
  if((plainfile = fopen(fnam, "rb"))== NULL)
  {
      fclose(cipherfile);
      free(cfnam);free(fnam);free(hfnam);
      throwout(_("Konnte Klartext nicht oeffnen"),2);
      return false;
  }
  if((inner_hdfile = fopen(hfnam, "rb"))== NULL)
  {
      fclose(cipherfile);fclose(plainfile);
      free(cfnam);free(fnam);free(hfnam);
      throwout(_("could not open header_1_file"),2);
      return false;
  }
  if(algo ==_("F_cnt_1c"))   { algoflag=9;}
  else if(algo ==_("F_cnt_1d"))   { algoflag=10;}
  else if(algo ==_("F_cnt_1b"))   { algoflag=11;}
  else if(algo ==_("flight"))   { algoflag=15;}
  else if(algo ==_("flightx"))   { algoflag=16;}
  else if(algo ==_("chimera")) //set threefish derived key and fightxkey derived keys
  {
        algoflag=20;
        if (blolen!= 512) { throwout(_("blocklength error  in chimera call"), 10); return false;}
        thrfkey=develop_o_malloc(keylen,key,2,2,144);
        threefishSetKey(&keyCtx, Threefish1024, (uint64_t *)thrfkey, (uint64_t *)(thrfkey+128));  //use first 128 byte of key as key, next 16 as tweak
        flightxkey=develop_o_malloc(keylen,key,3,2,512);
        if(flightxkey==NULL) {free(thrfkey); throwout(_(" failed malloc in enc fleafish(3)"));return false;}
        flightxblock= (char *) malloc(blolen+512);
        if(flightxblock==NULL)
        { free(thrfkey);free(flightxkey); return false;}
        ffic=(char *) malloc(blolen);
        if(ffic==NULL)
        { free(flightxblock); free(thrfkey);free(flightxkey); return false;}
  }
  else if(algo ==_("threefish"))   //set key and do special preparationes
  {
      algoflag=17;
      if (blolen!= 128) { throwout(_("blocklength error  in threefish call"), 10); return false;}
      thrfkey=develop_o_malloc(keylen,key,2,2,144);
      threefishSetKey(&keyCtx, Threefish1024, (uint64_t *)thrfkey, (uint64_t *)(thrfkey+128));  //use first 128 byte of key as key, next 16 as tweak
      //if(thrfkey != NULL) free(thrfkey); //free the threefish key stuff
  }

//initialize Progress bar
  wxProgressDialog pbar(_("Verschluesselungs-Fortschritt"),_("Hart arbeitender Computer :-)"),100);
  //determine blockzahl
  blockzahl = filen/blolen;
  //create IV from random then lock with key
  iv=(char *) malloc(blolen);
  ic=(char *) malloc(blolen);
  ip=(char *) malloc(blolen);
  zuf.push_bytes(blolen, (unsigned char *)iv);

  if(mode>0) //lock IV
  {
      if(algoflag==17)
      {
          //dofeistel(blolen, keylen, iv,ic, key, 4, mode, 0, NULL, primalgo); //special for threefish
          threefishEncryptBlockBytes(&keyCtx, (uint8_t *)iv,  (uint8_t *)ic);
      }
      else if(algoflag==20)   //lockiv
      {
         dofeistel(blolen, 512, iv,ic, flightxkey, 4, mode, 0, NULL, _("flightx"));//4Steps, mode-fold owf, no cpa blocker
      }
      else dofeistel_n(blolen, keylen, iv,ic, key, 4, mode, 0, NULL, (unsigned int)algoflag);//4Steps, mode-fold owf, no cpa blocker
  }
  else memcpy(ic,iv,blolen);
  //write locked IV
  for(i=0;i<blolen;i++)
  {
      fputc( *(ic + i),cipherfile) ;
  }
  //put IV into longnumber
  if(algoflag==20)
  {
      tfivno.resizelonu(128+2); //use this as threefish counter
      memcpy(tfivno.ad, iv, 128);
      tfivno.setsize(true);
  }
  ivno.resizelonu(blolen+2);
  memcpy(ivno.ad, iv, blolen);
  ivno.setsize(true);
  if((algoflag==17)||(algoflag==20)) {ivno.inc(); tfivno.inc();} //don't use same iv for first block xor!
  //loop complete blocks with progbar update

  for(j=0;j<blockzahl;j++)
  {
       if((shoprog>0)&&(blockzahl>1))
       {
         //prog bar propagation in main loop
         currpos=(int)( (100*j)/(blockzahl-1));
         if((currpos >lastmax)&&goodp)
         {
            goodp=pbar.Update(currpos);
            lastmax=currpos;
         }
       }
      //get key-developed number block from ivno
      if(algoflag != 20)  memcpy(ic,(char *)(ivno.ad),blolen);  //obsolete for chimera
      /** put in direct code for chimera here  **/
      if(algoflag==20)
      {
        //get key-developed number block from ivno
        threefishEncryptBlockBytes(&keyCtx, (uint8_t *)(tfivno.ad), (uint8_t *) ic);
        tfivno.inc();
        threefishEncryptBlockBytes(&keyCtx, (uint8_t *)(tfivno.ad), (uint8_t *) (ic+128));
        tfivno.inc();
        threefishEncryptBlockBytes(&keyCtx, (uint8_t *)(tfivno.ad), (uint8_t *) (ic+256));
        tfivno.inc();
        threefishEncryptBlockBytes(&keyCtx, (uint8_t *)(tfivno.ad), (uint8_t *) (ic+384));
        tfivno.inc();

        //now develop flightx-half
        //fflivno.ad nach ffic mit flightxkey
        memcpy(flightxblock,flightxkey,512); //copy
        memcpy(flightxblock+512,ivno.ad,512); //copy
        develop_flightx_onearm( flightxblock, blolen+512, 1);
        //pick center section
        memcpy(ffic,flightxblock+(blolen+512)/2,512);
        //unite via xor with threefish result blocks in ic
        for(i=0; i<blolen;i++)
        {
            *(ic+i) ^=  *(ffic+i);
        }
      }
      else if(algoflag==17) //threefish
      {
          threefishEncryptBlockBytes(&keyCtx, (uint8_t *)ic, (uint8_t *) ic);
      }
      else k_develop_f4(blolen, ic,0,NULL,keylen,key,k_steps,algoflag); //14 ->Fleas_lc 15 ->fleas ld
      ivno.inc();
      //read in plaintext block
      for(i=0;i<blolen;i++)
      {
          if(readbyts< headlen)
          {
             *(ip + i)= (char) fgetc(inner_hdfile);
            readbyts++;
          }
          else
          {
             *(ip + i)= (char) fgetc(plainfile);
            readbyts++;
          }
          hc=*(ic + i);
          hc ^= *(ip+i);
          fputc( hc,cipherfile) ;
      }

  }
 //get rest block from plainfile
  //get key-developed number block from ivno
  //xor bytewise and write to cipherfile
  if(algoflag != 20) memcpy(ic,(char *)(ivno.ad),blolen);  //obsolete for chimera
  if(algoflag==17) //threefish
  {
    threefishEncryptBlockBytes(&keyCtx, (uint8_t *)ic, (uint8_t *) ic);
  }
  else if(algoflag == 20) //for chimera
  {
        //get key-developed number block from ivno
        threefishEncryptBlockBytes(&keyCtx, (uint8_t *)(tfivno.ad), (uint8_t *) ic);
        tfivno.inc();
        threefishEncryptBlockBytes(&keyCtx, (uint8_t *)(tfivno.ad), (uint8_t *) (ic+128));
        tfivno.inc();
        threefishEncryptBlockBytes(&keyCtx, (uint8_t *)(tfivno.ad), (uint8_t *) (ic+256));
        tfivno.inc();
        threefishEncryptBlockBytes(&keyCtx, (uint8_t *)(tfivno.ad), (uint8_t *) (ic+384));
        tfivno.inc();
        //now develop flightx-half
        //fflivno.ad nach ffic mit flightxkey
        memcpy(flightxblock,flightxkey,512); //copy
        memcpy(flightxblock+512,ivno.ad,512); //copy
        develop_flightx_onearm( flightxblock, blolen+512, 1);
        //pick center section
        memcpy(ffic,flightxblock+(blolen+512)/2,512);
        //unite via xor with threefish result blocks in ic
        for(i=0; i<blolen;i++)
        {
            *(ic+i) ^=  *(ffic+i);
        }
  }
  else k_develop_f4(blolen, ic,0,NULL,keylen,key,k_steps,algoflag); //14 ->Fleas_lc 15 ->fleas ld
  i=0;
  do
  {
      if(readbyts< headlen)
      {
          ichar=fgetc(inner_hdfile);
          readbyts++;
      }
      else
      {
            ichar=fgetc(plainfile);
            readbyts++;
      }
      if(ichar==EOF) break;
      hc=(char) ichar;
      hc ^= *(ic + i);
      fputc( hc,cipherfile);
      i++;
  } while(ichar != EOF);
  //close files
  fclose(cipherfile);
  fclose(plainfile);
  fclose(inner_hdfile);
    //free buffers
  free(cfnam);free(hfnam);free(fnam);
  free(iv); free(ip); free(ic);
  if(thrfkey != NULL) free(thrfkey); //free the threefish key stuff
  if(algoflag==20) //free the chimera stuff
  {
      free(ffic); free(flightxkey); free(flightxblock);
  }
  return true;
}
/*****************************************************/
/*****************************************************/
bool encipher_file_cntmode_mt(char* fnam, char* key, int keylen, int k_steps, int blolen, int mode, wxString algo, unsigned char shoprog)
//k_steps: number of owf-steps,   mode:0-clear number, 1number of lonulock steps for number
//
{
  long long filen=0, blockzahl, j;
  FILE *plainfile, *cipherfile;
  char *sfnam, *iv, *ic,*ic1,*ic2,*ic3,*ip,hc;
  int i,algoflag,ichar, currpos,lastmax=0;
  longnumber ivno;
  bool goodp=true;
  kdev_thr *pt0, *pt1,*pt2,*pt3;
  bool r0,r1,r2,r3;

  //some primitive error checking
  if(strlen(fnam) > 598) return false;
  wxString fnwx(fnam, wxConvFile);
  if(!wxFile::Exists(fnwx))
  {
      throwout(_("Datei existiert nicht!!\nAbbruch."),2);
      return false;
  }
  //determine Length
  wxFile tmpfl(fnwx);
  filen=tmpfl.Length();
  tmpfl.Close();
  if((plainfile = fopen(fnam, "rb"))== NULL) return false;
  // jetzt cipherfile öffnen mit anhängsel "_1c" etc und setze algoflag
    //create new sfnam "shadow fnam" buffer
    sfnam=(char *)malloc(strlen(fnam)+10);
    strcpy(sfnam,fnam);
  if(algo ==_("F_cnt_1c"))   {strncat(sfnam,"_1c",3); algoflag=9;}
  else if(algo ==_("F_cnt_1d"))   {strncat(sfnam,"_1d",3); algoflag=10;}
  else if(algo ==_("F_cnt_1b"))   {strncat(sfnam,"_1b",3); algoflag=11;}
   else if(algo ==_("flight"))   {strncat(sfnam,"_fl",3); algoflag=15;}
    else if(algo ==_("flightx"))   {strncat(sfnam,"_fx",3); algoflag=16;}
    /*else if(algo ==_("Fleas_3"))   {strncat(sfnam,"_f3",3); algoflag=3;}
     else if(algo ==_("aes"))   {strncat(sfnam,"_ae",3);blolen=8;algoflag=12;}*/
       else{throwout(_("unknown algorithm! \n aborting (0)"));free(sfnam);return false;}
  if((cipherfile = fopen(sfnam, "wb"))== NULL)
  {
    free(sfnam);
    return false;
  }
  free(sfnam);

//initialize Progress bar
  wxProgressDialog pbar(_("Verschluesselungs-Fortschritt"),_("Sehr hart arbeitender Computer :-)"),100);
  //determine blockzahl
  blockzahl=filen/blolen;
  //create IV from random then lock with key
  iv=(char *) malloc(blolen);
  ic=(char *) malloc(blolen);
  ic1=(char *) malloc(blolen);
  ic2=(char *) malloc(blolen);
  ic3=(char *) malloc(blolen);
  ip=(char *) malloc(blolen);
  zuf.push_bytes(blolen, (unsigned char *)iv);
  wxString primalgo;
  if(algoflag==9) primalgo=_("Fleas_lc");
  else if(algoflag==10) primalgo=_("Fleas_ld");
  else if(algoflag==11) primalgo=_("Fleas_lb");
  else if(algoflag==15) primalgo=_("flight");
  else if(algoflag==16) primalgo=_("flightx");

  else throwout(_("Fehler! \nunbekannter Primaeralgorithmus in countermode Verschluesselung"));
  if(mode>0)
  {
      dofeistel_n(blolen, keylen, iv,ic, key, 4, mode, 0, NULL, (unsigned int)algoflag);//4Steps, mode-fold owf, no cpa blocker
  }
  else memcpy(ic,iv,blolen);
  //write locked IV
  for(i=0;i<blolen;i++)
  {
      fputc( *(ic + i),cipherfile) ;
  }
  //put IV into longnumber
  ivno.resizelonu(blolen+2);
  memcpy(ivno.ad, iv, blolen);
  ivno.setsize(true);
  //loop complete blocks with progbar update

  for(j=0;j+4<blockzahl;j+=4)
  {
       if((shoprog>0)&&(blockzahl>1))
       {
         //prog bar propagation in main loop
         currpos=(int)( (100*j)/(blockzahl-1));
         if((currpos >lastmax)&&goodp)
         {
            goodp=pbar.Update(currpos);
            lastmax=currpos;
         }
       }
      //get key-developed number block from ivno
      memcpy(ic,(char *)(ivno.ad),blolen);
      //do it threaded
      //k_develop_f4(blolen, ic,0,NULL,keylen,key,k_steps,algoflag); //14 ->Fleas_lc 15 ->flight
        pt0=new kdev_thr(&r0,blolen, ic,0,NULL,keylen,key,k_steps,algoflag);
        if ( pt0->Create() != wxTHREAD_NO_ERROR )
            {
                throwout(_("Can't create the thread!"),3);
                delete pt0;
                pt0=NULL;
            }
        else
          {
            if ( pt0->Run() != wxTHREAD_NO_ERROR )
            {
                throwout(_("Can't Run the thread!"),3);
                delete pt0;
                pt0 = NULL;
            }
          }
          if(pt0==NULL)  //error  go to harakiri
          {
               //close files
                fclose(cipherfile);
                fclose(plainfile);
                //free buffers
                free(iv); free(ip); free(ic);free(ic1);free(ic2);free(ic3);
                throwout(_("error in encipher file cntmode multithread t0"),3);
                return false;
          }


      ivno.inc();
      memcpy(ic1,(char *)(ivno.ad),blolen);
      //k_develop_f4(blolen, ic1,0,NULL,keylen,key,k_steps,algoflag); //14 ->Fleas_lc 15 ->fleas ld
        pt1=new kdev_thr(&r1,blolen, ic1,0,NULL,keylen,key,k_steps,algoflag);
        if ( pt1->Create() != wxTHREAD_NO_ERROR )
            {
                throwout(_("Can't create the thread!"),3);
                delete pt1;
                pt1=NULL;
            }
        else
          {
            if ( pt1->Run() != wxTHREAD_NO_ERROR )
            {
                throwout(_("Can't Run the thread!"),3);
                delete pt1;
                pt1 = NULL;
            }
          }
          if(pt1==NULL)  //error  go to harakiri
          {
               pt0->Wait();
               //close files
                fclose(cipherfile);
                fclose(plainfile);
                //free buffers
                free(iv); free(ip); free(ic);free(ic1);free(ic2);free(ic3);
                throwout(_("error in encipher file cntmode multithread t1"),3);
                return false;
          }

      ivno.inc();
      memcpy(ic2,(char *)(ivno.ad),blolen);
      //k_develop_f4(blolen, ic2,0,NULL,keylen,key,k_steps,algoflag); //14 ->Fleas_lc 15 ->fleas ld
        pt2=new kdev_thr(&r2,blolen, ic2,0,NULL,keylen,key,k_steps,algoflag);
        if ( pt2->Create() != wxTHREAD_NO_ERROR )
            {
                throwout(_("Can't create the thread!"),3);
                delete pt2;
                pt2=NULL;
            }
        else
          {
            if ( pt2->Run() != wxTHREAD_NO_ERROR )
            {
                throwout(_("Can't Run the thread!"),3);
                delete pt2;
                pt2 = NULL;
            }
          }
          if(pt2==NULL)  //error  go to harakiri
          {
               pt0->Wait();
               pt1->Wait();
               //close files
                fclose(cipherfile);
                fclose(plainfile);
                //free buffers
                free(iv); free(ip); free(ic);free(ic1);free(ic2);free(ic3);
                throwout(_("error in encipher file cntmode multithread t1"),3);
                return false;
          }

      ivno.inc();
      memcpy(ic3,(char *)(ivno.ad),blolen);
      //k_develop_f4(blolen, ic3,0,NULL,keylen,key,k_steps,algoflag); //14 ->Fleas_lc 15 ->fleas ld
        pt3=new kdev_thr(&r3,blolen, ic3,0,NULL,keylen,key,k_steps,algoflag);
        if ( pt3->Create() != wxTHREAD_NO_ERROR )
            {
                throwout(_("Can't create the thread!"),3);
                delete pt3;
                pt3=NULL;
            }
        else
          {
            if ( pt3->Run() != wxTHREAD_NO_ERROR )
            {
                throwout(_("Can't Run the thread!"),3);
                delete pt3;
                pt3 = NULL;
            }
          }
          if(pt3==NULL)  //error  go to harakiri
          {
               pt0->Wait();
               pt1->Wait();
               pt2->Wait();
               //close files
                fclose(cipherfile);
                fclose(plainfile);
                //free buffers
                free(iv); free(ip); free(ic);free(ic1);free(ic2);free(ic3);
                throwout(_("error in encipher file cntmode multithread t1"),3);
                return false;
          }
      //xor with plaintext block
      //write to cipher file
      //increment ivno
      ivno.inc();
//wait for proc0 to finish
       pt0->Wait();
      //read in plaintext block
      for(i=0;i<blolen;i++)
      {
          *(ip + i)= (char) fgetc(plainfile);
          hc=*(ic + i);
          hc ^= *(ip+i);
          fputc( hc,cipherfile) ;
      }
//wait for proc1 to finish
       pt1->Wait();
      //read in plaintext block
      for(i=0;i<blolen;i++)
      {
          *(ip + i)= (char) fgetc(plainfile);
          hc=*(ic1 + i);
          hc ^= *(ip+i);
          fputc( hc,cipherfile) ;
      }
//wait for proc2 to finish
       pt2->Wait();
      //read in plaintext block
      for(i=0;i<blolen;i++)
      {
          *(ip + i)= (char) fgetc(plainfile);
          hc=*(ic2 + i);
          hc ^= *(ip+i);
          fputc( hc,cipherfile) ;
      }
//wait for proc3 to finish
       pt3->Wait();
      //read in plaintext block
      for(i=0;i<blolen;i++)
      {
          *(ip + i)= (char) fgetc(plainfile);
          hc=*(ic3 + i);
          hc ^= *(ip+i);
          fputc( hc,cipherfile) ;
      }

  }
  //now do the rest in the traditional way
  // reached j was GE blockzahl -3
  for(;j<blockzahl;j++)
  {
       if((shoprog>0)&&(blockzahl>1))
       {
         //prog bar propagation in main loop
         currpos=(int)( (100*j)/(blockzahl-1));
         if((currpos >lastmax)&&goodp)
         {
            goodp=pbar.Update(currpos);
            lastmax=currpos;
         }
       }
      //read in plaintext block
      for(i=0;i<blolen;i++)
      {
          *(ip + i)= (char) fgetc(plainfile);
      }
      //get key-developed number block from ivno
      memcpy(ic,(char *)(ivno.ad),blolen);
      k_develop_f4(blolen, ic,0,NULL,keylen,key,k_steps,algoflag); //14 ->Fleas_lc 15 ->fleas ld 16 ->fleas ld
      //xor with plaintext block
      //write to cipher file
      for(i=0;i<blolen;i++)
      {

          hc=*(ic + i);
          hc ^= *(ip+i);
          fputc( hc,cipherfile) ;
      }
      //increment ivno
      ivno.inc();
  }

  //get rest block from plainfile
  //get key-developed number block from ivno
  //xor bytewise and write to cipherfile
  memcpy(ic,(char *)(ivno.ad),blolen);
  k_develop_f4(blolen, ic,0,NULL,keylen,key,k_steps,algoflag); //14 ->Fleas_lc 15 ->fleas ld
  i=0;
  do
  {
      ichar=fgetc(plainfile);
      if(ichar==EOF) break;
      hc=(char) ichar;
      hc ^= *(ic + i);
      fputc( hc,cipherfile) ;
      i++;
  } while(ichar != EOF);
  //close files
  fclose(cipherfile);
  fclose(plainfile);
    //free buffers
    free(iv); free(ip); free(ic);free(ic1);free(ic2);free(ic3);
  return true;
}
/*****************************************************/
int eax_enc_file(wxString fnam, wxString outnam, char* key, int keylen, wxString algo, unsigned char shoprog)
{
    int fr,kr,h_len, namlen;
    wxString hx,halgo,extn, cm_str, cfnam;
    bool ciphres;
    longnumber lky,hmacky,nonce, calc_mac;
    FILE *pkf;
    char *bfnam;


    if(!wxFileExists(fnam))
    {
        if(shoprog==1) throwout(_("Datei existiert nicht!\nAbbruch."),4);
        return -1;
    }
    if(algo==_("aes")){fr=0; kr=1; h_len=16;extn=_("_ae");}
      else if(algo== _("Fleas_1_8")) {fr=4;kr=2;h_len=65; extn=_("_c3");}
      else if(algo== _("Fleas_3")) {fr=4;kr=2;h_len=65; extn=_("_f3");}
      else if(algo== _("Fleas_o2")){fr=4;kr=2;h_len=512; extn=_("_o2");}
      else if(algo== _("Fleas_o5")){fr=4;kr=2;h_len=512; extn=_("_o5");}
      else if(algo== _("Fleas_l")){fr=4;kr=2;h_len=512; extn=_("_l");}
      else if(algo== _("Fleas_ls")){fr=4;kr=2;h_len=512; extn=_("_ls");}
      else if(algo== _("Fleas_l3")){fr=4;kr=3;h_len=512; extn=_("_l3");}
      else if(algo== _("Fleas_lb")){fr=4;kr=3;h_len=512; extn=_("_lb");}
      else if(algo== _("Fleas_lc")){fr=4;kr=3;h_len=512; extn=_("_lc");}
      else if(algo== _("Fleas_ld")){fr=4;kr=3;h_len=512; extn=_("_ld");}
      else if(algo== _("F_cnt_1c")){fr=4;kr=3;h_len=512; extn=_("_1c");}
      else if(algo== _("F_cnt_1d")){fr=4;kr=4;h_len=512; extn=_("_1d");}
      else if(algo== _("F_cnt_1b")){fr=4;kr=4;h_len=512; extn=_("_1b");}
      else if(algo== _("flight")){fr=4;kr=1;h_len=512; extn=_("_fl");}
      else if(algo== _("flightx")){fr=4;kr=1;h_len=512; extn=_("_fx");}
        else{ throwout(_("No valid or known algorithm specified! \nAborting.")); return -1;}

    namlen=fnam.Len();
    if(namlen>1000) {throwout(_("Dateiname zu lang, Abbruch.")); return -1;}
    else bfnam = (char *) malloc(2*fnam.Len()+2);
    strcpy( bfnam, (const char*)fnam.mb_str(wxConvLocal) );

    lky.resizelonu((ulong32)keylen+2);
    memcpy(lky.ad, key, keylen);
    lky.setsize(true);
    lky.shrinktofit();
    //if(keylen<ss_klen) lky.resizelonu(ss_klen);
    if(lky.length< (ulong32)ss_klen) lky.resizelonu(ss_klen);
    ciphres=encipher_file(bfnam, (char *)lky.ad, lky.length, (char) fr, kr, h_len, 1, algo,1);
    free(bfnam);
    //give message
    if(ciphres) { if(shoprog==1) throwout(_("Verschluesselt!"),1);}
     else {
         { if(shoprog==1) throwout(_("Verschluesselung fehlgeschlagen!"));}
         return -1;
     }
    //create prepend file
    hx=  sec_path + _("prep_file.pre");
    if((pkf=fopen((const char*)hx.mb_str(wxConvLocal),"w")) == NULL)
    {
        throwout(_("Error! opening prepend-file failed!?!"),5);
        return -1;
    }
    fprintf(pkf,"algo: %s \n",(const char*)algo.mb_str(wxConvUTF8));

    //determine hmac key
    nonce.makerandom(48);
    nonce.setsize(true);
    hmacky.copynum(&lky);
    hmacky.resizelonu(lky.length);
    hmacky.setsize(true);
    hmacky.shrinktofit();
    co_develop_ln2((char *)(hmacky.ad),(int)(hmacky.length), (char *)(nonce.ad),(int)(nonce.size),2,3 );
    halgo = algo;
    if(halgo == _("aes")) halgo= _("sha2");
       else if(halgo == _("Fleas_l")) halgo=_("Fleas_b");  //obsolete
         else if(halgo== _("F_cnt_1b")) halgo= _("Fleas_b");
           else if(halgo== _("F_cnt_1c")) halgo= _("Fleas_c");
             else if(halgo== _("F_cnt_1d")) halgo= _("Fleas_d");
              else if(halgo== _("Fleas_lb")) halgo= _("Fleas_b");
               else if(halgo== _("Fleas_ld")) halgo= _("Fleas_d");
                 else if(halgo== _("flight")) halgo= _("sha4");
                   else  halgo= _("Fleas_lc"); //older ciphers
    cfnam= fnam+extn;
    hmacky.setsize(true);
    hmacky.shrinktofit();
    if((ulong32) h_len> hmacky.length) hmacky.resizelonu(h_len);
    get_lonu_hmac_from_filename_x(&calc_mac,&cfnam, &hmacky, h_len, halgo, 1,0);
    calc_mac.resizelonu(48,0);
    calc_mac.writehex(&cm_str);
    fprintf(pkf,"MAC: %s  %s \n",(const char*)halgo.mb_str(wxConvUTF8),(const char*)cm_str.mb_str(wxConvUTF8));
    nonce.writehex(&cm_str);  //reusing string
    fprintf(pkf,"nonce: %s \n",(const char*)cm_str.mb_str(wxConvUTF8));

    fclose(pkf);
    //concatenate files and cleanup
    if(!fileconcat(hx, fnam+extn, outnam, _("end_header_info::")))
        {
          throwout(_("Error!\nFailed to create ciph-file from components"),20);
        }
    else
    {
       if(shoprog==1) throwout(_("Chiffrat wurde im Verzeichnis\ndes Klartextes abgelegt!"),2);
    }
        //delete concat wells
    wxRemoveFile(hx);
    wxRemoveFile(fnam+extn);
    return 0;
}
/****************************************/
int eax_dec_file(wxString fnam, wxString outnam, char* key, int keylen,unsigned char shoprog, bool strict)
{
    ulong32 deadhead;
    char buff1[2002],*pnam,*cnam;
    int rcnt,fr,kr,h_len,trydec;
    FILE *pkf;
    wxString algo,hs,macnam,h2;
    longnumber nonce,skey, c_mac,lky,hmacky, calc_mac;
    bool ciphres;


    if(!wxFileExists(fnam))
    {
        if(shoprog==1) throwout(_("Chiffrat-Datei existiert nicht!\n Tu nix."),4);
        return -1;
    }
    if(!file_pos_past(fnam,_("end_header_info::"),&deadhead,2000))
    {
        throwout(_("header mark not found!\naborting"),5);
        return -1;
    }
    pkf=fopen((const char*)fnam.mb_str(wxConvLocal), "r");
    if(pkf==NULL){throwout(_("error opening head-file of cipher"),4); return -1;}
    rcnt= fscanf(pkf,"algo: %400s ",buff1);
    if(rcnt != 1) {throwout(_("error reading algo"),4); fclose(pkf); return -1;}
    algo=wxString::FromUTF8(buff1);
    //check for MAC tag
    rcnt= fscanf(pkf,"%10s ",buff1);
    if(rcnt != 1)
    {
            throwout(_("Warnung, kaputte Kopf-Datei!\nAbbruch."),2);
            fclose(pkf);
            return -1;
    }
    else
    {
         h2=wxString::FromUTF8(buff1);
         if(h2==_("MAC:"))
         {
             rcnt= fscanf(pkf,"%400s ",buff1);
             if(rcnt == 1)
             {
                macnam=wxString::FromUTF8(buff1);
                rcnt= fscanf(pkf,"%2000s ",buff1);
                if(rcnt != 1) {throwout(_("Fehler beim Lesen des MAC \nKEINE Entschluesselung."),2);fclose(pkf); return -1;}
                h2=wxString::FromUTF8(buff1);
                c_mac.storhex(&h2);
                rcnt = fscanf(pkf," nonce: %2000s ",buff1);
                if(rcnt != 1) {throwout(_("Fehler beim Lesen der NONCE\nKEINE Entschluesselung."),2);fclose(pkf); return -1;}
                h2=wxString::FromUTF8(buff1);
                nonce.storhex(&h2);
             }
             else
             {
                 throwout(_("Fehler beim Lesen des MAC-Typs \nKEINE Entschluesselung."),4);
                 fclose(pkf);
                 return -1;
             }


         }
         else
         {
             throwout(_("Warnung, Integritaetstest nicht moeglich."),2);
             fclose(pkf);
             return -1;
         }
    }
    fclose(pkf);
    if(algo==_("aes")){fr=0; kr=1; h_len=16;}
      else if(algo== _("Fleas_1_8")) {fr=4;kr=2;h_len=65;}
      else if(algo== _("Fleas_3")) {fr=4;kr=2;h_len=65;}
      else if(algo== _("Fleas_o2")){fr=4;kr=2;h_len=512;}
      else if(algo== _("Fleas_o5")){fr=4;kr=2;h_len=512;}
      else if(algo== _("Fleas_l")){fr=4;kr=2;h_len=512;}
      else if(algo== _("Fleas_ls")){fr=4;kr=2;h_len=512;}
      else if(algo== _("Fleas_l3")){fr=4;kr=3;h_len=512;}
      else if(algo== _("Fleas_lb")){fr=4;kr=3;h_len=512;}
      else if(algo== _("Fleas_lc")){fr=4;kr=3;h_len=512;}
      else if(algo== _("Fleas_ld")){fr=4;kr=3;h_len=512;}
      else if(algo== _("F_cnt_1c")){fr=4;kr=3;h_len=512;}
      else if(algo== _("F_cnt_1d")){fr=4;kr=4;h_len=512;}
      else if(algo== _("F_cnt_1b")){fr=4;kr=4;h_len=512;}
        else{ throwout(_("No valid or known algorithm specified! \nAborting.")); return -1;}


    lky.resizelonu((ulong32)keylen+2);  //copy key to lky
    memcpy(lky.ad, key, keylen);
    lky.setsize(true);
    lky.shrinktofit();
    //if(keylen<ss_klen) lky.resizelonu(ss_klen);
    if(lky.length< (ulong32)ss_klen) lky.resizelonu(ss_klen);

    hmacky.copynum(&lky);
    hmacky.setsize(true);
    hmacky.shrinktofit();
    //develop key with nonce
    ciphres=co_develop_ln2((char *)(hmacky.ad),(int)(hmacky.length),(char *)(nonce.ad),(int)(nonce.size), 2,3 );
    if(!ciphres) throwout(_("error in co_develop_ln2)"),4);
    hmacky.setsize(true);
    hmacky.shrinktofit();
    if((ulong32)h_len> hmacky.length) hmacky.resizelonu(h_len);
    get_lonu_hmac_from_filename_x(&calc_mac,&fnam, &hmacky, h_len, macnam, 1, deadhead );
    calc_mac.resizelonu(48,0);
    if( calc_mac.compare(&c_mac) != 0 ) //wrong mac!
    {
        if(shoprog) throwout(_("Keine Integritaet der Chiffre!"),1);
        throwout(_("Ernste Warnung!\nIntegritaetsverletzung der Chiffre!"));
        if(strict) return -2;
        else  //if not strict
        {
             AskDialog adlg(NULL,_("Kann Integritaet nicht verifizieren!!\nOK klicken, wenn Du abbrechen willst!\nRiskiere es und versuche trotzdem zu dechiffrieren(klick NO)?"));
             trydec =adlg.ShowModal();
             if(trydec==1) return -2;
             else
             {
               throwout(_(" Du musst die evtl. gewonnene Klardatei mit groesstem Misstrauen betrachten!\nHol tief Luft uns bete....8-0"),4);
             }
        }
    }
    pnam = (char *) malloc(4*outnam.Len()+2);
    cnam = (char *) malloc(4*fnam.Len()+2);
    strcpy( pnam, (const char*)outnam.mb_str(wxConvLocal) );
    strcpy( cnam, (const char*)fnam.mb_str(wxConvLocal) );

    ciphres=dec_fil(deadhead,cnam,pnam, (char *)lky.ad, lky.length, (char) fr, kr, h_len, 1, algo,1);
    free(pnam); free(cnam);
    //give message
    if(ciphres)
    {
        if(shoprog==1) throwout(_("Dechiffriert."),1);
    }
     else {
         throwout(_("Dechiffrierung GESCHEITERT!"));
         return-3;
     }
    return 0;
}
/*****************************************************/
bool encipher_file(char* fnam, char* key, int keylen, char feistelsteps, int k_steps, int blolen, int mode, wxString algo, unsigned char shoprog)
// verschlüsselt ein File mit key
// wahre Blocklänge 2*blolen, blolen muss größer 25 sein
// alte filelänge long len wird vorangestellt
{
  long long filen=0, blockzahl, j,newfilen,blinds,bytcount=0;
  FILE *plainfile, *cipherfile;
  char *bff1, *bff_o, *bff_last, *cpa_bl, *key2, *sfnam;
  int restlen,i,cpa_blen, algoflag;
  int currpos, lastmax=0;
  bool goodp=true;
  aes_context aesc;
  unsigned int algocode;



  if( (algo==_("F_cnt_1c"))||(algo==_("F_cnt_1d"))||(algo==_("F_cnt_1b")) ) //-> countermode
  {
      return( encipher_file_cntmode_mt(fnam, key, keylen, k_steps, blolen, 2, algo, shoprog) );
  }
  else if( (algo==_("flight"))||(algo==_("flightx"))||(algo==_("threefish"))||(algo==_("chimera")) ) //-> countermode
  {
      return( encipher_file_cntmode(fnam, key, keylen, k_steps, blolen, 2, algo, shoprog) );
  }
  algocode=get_code_4_algo(algo); //
  if((blolen<26)&&(algo!=_("aes"))) return false;// kein Platz für die Filelänge!
  if(strlen(fnam) > 398) return false;
  wxString fnwx(fnam,wxConvFile);
  if(!wxFile::Exists(fnwx))
  {
      throwout(_("File does not exist!!\nAborting."),2);
      return false;
  }
  //determine Length
  wxFile tmpfl(fnwx);
  filen=tmpfl.Length();
  tmpfl.Close();
  //now normal old code
  if((plainfile = fopen(fnam, "rb"))== NULL) return false;
  // jetzt cipherfile öffnen mit anhängsel "_c2" etc und setze algoflag
    //create new sfnam "shadow fnam" buffer
    sfnam=(char *)malloc(strlen(fnam)+10);
    strcpy(sfnam,fnam);
  if(algo ==_("Fleas_1_8"))   {strncat(sfnam,"_c3",3); algoflag=0;}
    else if(algo ==_("Fleas_3"))   {strncat(sfnam,"_f3",3); algoflag=3;}
     else if(algo ==_("Fleas_4"))   {strncat(sfnam,"_f4",3); algoflag=4;}
     else if(algo ==_("Fleas_5"))   {strncat(sfnam,"_f5",3); algoflag=5;}
     else if(algo ==_("Fleas_x2"))   {strncat(sfnam,"_x2",3); algoflag=6;}
     else if(algo ==_("Fleas_x5"))   {strncat(sfnam,"_x5",3); algoflag=7;}
     else if(algo ==_("Fleas_o2"))   {strncat(sfnam,"_o2",3); algoflag=8;}
     else if(algo ==_("Fleas_o5"))   {strncat(sfnam,"_o5",3); algoflag=9;}
     else if(algo ==_("Fleas_l"))   {strncat(sfnam,"_l",2); algoflag=10;}
     else if(algo ==_("Fleas_ls"))   {strncat(sfnam,"_ls",3);  algoflag=11;}
     else if(algo ==_("Fleas_l3"))   {strncat(sfnam,"_l3",3);  algoflag=13;}
     else if(algo ==_("Fleas_lc"))   {strncat(sfnam,"_lc",3);  algoflag=14;}
     else if(algo ==_("Fleas_ld"))   {strncat(sfnam,"_ld",3);  algoflag=15;}
     else if(algo ==_("Fleas_lb"))   {strncat(sfnam,"_lb",3);  algoflag=16;}
     else if(algo ==_("aes"))   {strncat(sfnam,"_ae",3);blolen=8;algoflag=12;}
       else{throwout(_("unknown algorithm! \n aborting (1)"));free(sfnam);return false;}
  if((cipherfile = fopen(sfnam, "wb"))== NULL)
  {
    free(sfnam);
    return false;
  }
  free(sfnam);

//initialize Progress bar
  wxProgressDialog pbar(_("Verschluesselungs-Fortschritt"),_("Sehr hart arbeitender Prozessor :-)"),100);

  cpa_blen=blolen; //cpa_blen is matching blolen
  /** blolen must be 8 for aes!!  **/
  if((algo == _("aes"))&&(blolen != 8))
  {
      throwout(_("fatal error in aes blocksize!")); return false;
  }
  //create cpa_blocker block and possibly derived key
  key2= (char *) malloc(keylen);
  memcpy(key2,key,keylen); //key2 filled for use
  switch( algoflag)
  {
    case 4 :
      // build derived key for use with Fleas_4
      develop_c(key2,keylen,2);
      // build cpa_blocker with different algorithm
      cpa_bl= develop_b_malloc(keylen, key,4, cpa_blen);
      break;

    case 5 :
      // build derived key for use with Fleas_4
      develop_x(key2,keylen,2,3);
      // build cpa_blocker with different algorithm
      cpa_bl= develop_b_malloc(keylen, key,4, cpa_blen);
      break;

    case 6 :
      // build derived key for use with Fleas_x2
      develop_x1(key2,keylen,2,3);
      // build cpa_blocker with different algorithm
      cpa_bl= develop_x_malloc(keylen, key,4,2, cpa_blen);
      break;

    case 7 :
      // build derived key for use with Fleas_x5
      develop_x1(key2,keylen,2,3);
      // build cpa_blocker with different algorithm
      cpa_bl= develop_x_malloc(keylen, key,4,5, cpa_blen);
      break;

    case 8 :
      // build derived key for use with Fleas_o2
      develop_o(key2,keylen,2,3,0);
      // build cpa_blocker with different algorithm
      cpa_bl= develop_o_malloc(keylen, key,3,2, cpa_blen,0);
      break;

    case 9 :
      // build derived key for use with Fleas_o5
      develop_o(key2,keylen,2,3,2);
      // build cpa_blocker with different algorithm
      cpa_bl= develop_o_malloc(keylen, key,3,5, cpa_blen,2);
      break;

    case 10 :
      // build derived key for use with Fleas_l
      develop_l(key2,keylen,2);
      // build cpa_blocker with different algorithm
      cpa_bl= develop_l_malloc(keylen, key, 3,cpa_blen);
      break;

    case 11 :
      // build derived key for use with Fleas_ls
      develop_l(key2,keylen,2);
      // build cpa_blocker with different algorithm
      cpa_bl= develop_l_malloc(keylen, key, 3,cpa_blen);
      break;

    case 12 :
      // build derived key for use with aes
      develop_o(key2,keylen,2,3,2); //not used
      // build cpa_blocker with different algorithm
      cpa_bl= develop_o_malloc(keylen, key, 3,5,6*blolen,2); //cpablen, erste 16 für cpablock, 2.&3. 16 für key in 2 rounds
      break;

    case 13 :
      // build derived key for use with Fleas_l3
      develop_ls(key2,keylen,3,3,1);
      // build cpa_blocker with different algorithm
      cpa_bl= develop_l_malloc(keylen, key, 3,cpa_blen);
      break;

    case 14 :
      // build derived key for use with Fleas_lc
      develop_ln(key2,keylen,3,3,1);
      // build cpa_blocker with different algorithm
      cpa_bl= develop_l_malloc(keylen, key, 3,cpa_blen);
      break;

    case 15 :
      // build derived key for use with Fleas_ld
      develop_ln2_mt(key2,keylen,3,3,1);
      // build cpa_blocker with different algorithm
      cpa_bl= develop_l_malloc(keylen, key, 4,cpa_blen);
      break;

    case 16 :
      // build derived key for use with Fleas_ld
      develop_ln2_mt(key2,keylen,3,3,1);
      // build cpa_blocker with different algorithm
      cpa_bl= develop_l_malloc(keylen, key, 2,cpa_blen);
      break;

    default:
      cpa_bl=bytmixxmalloc(keylen,key,6,cpa_blen);
      break;
  }


//neue Routine zur Filelaenge already set filen
  /*while (!feof(plainfile)) {   // bestimme die Filelänge
      fgetc (plainfile);
      filen++;
      }
      filen--; //eins zuviel gezaehlt!
      rewind(plainfile);
*/
  //bestimme jetzt die Blockzahl
  blockzahl = (filen + 50) / (2*(long long)blolen);
  //
  if( ((filen + 50) % (2*(long long)blolen)) !=0) { blockzahl++; } //jetzt ham wir die richtige Blockzahl
  newfilen = blockzahl*2*(long long)blolen; //neue Filelaenge
  blinds = newfilen - filen;
  bff1 = (char *)malloc(2*blolen); bff_o = (char *)malloc(2*blolen);  //Buffer in and out allozieren
  if(mode == 1) bff_last = (char *)malloc(2*blolen); //bei CBC-mode noch ein zusätzlicher Buffer
  //srand((unsigned int) *(bff_o)); // randomvariable mit unbestimmtem seeden

#if defined(__WXMSW__)
  sprintf(bff1,"si3:%Ld ",filen); // hat jetzt die echtlänge an den Anfang geschrieben
#else
  sprintf(bff1,"si3:%lld ",filen); // hat jetzt die echtlänge an den Anfang geschrieben
#endif
  // jetzt restlänge bestimmen
  restlen=(int)(filen-(blockzahl-1)*2*(long long)blolen);
  //differenz mit Zufallszahlen auffüllen
  // kann auch negativ sein: dann Zufallszahlen AUCH in den nächsten Block eintragen
  bytcount=strlen(bff1);
  for( i=strlen(bff1);i<2*blolen;i++) //von ende der Längenangabe bis blockende
  {
      bytcount++;
      if(bytcount>blinds) *(bff1 + i)= (char) fgetc(plainfile);
      else *(bff1 + i)= zuf.get_rbyte(); // mit zufallszahlen auffüllen
  } // füllt den Rest mit dem Filerest auf, dazwischen wirds mit zufallszahlen aufgefüllt und wird bei decipher auch weggeschmissen
  // jetzt rausschreiben in cipherfile
  /********* muss noch verschl. werden ************/
//ersten block verschlüsseln
  if(algo==_("aes"))
  {
      //set encipher structure
        if (aes_setkey_enc( &aesc, (const unsigned char *)(cpa_bl+16), 256 )!= 0)
          throwout(_("error in setting aes key context"));
      //xor cpa_blocker to input block
      if(mode==1) for(i=0;i<16;i++) *(bff1+i) ^=  *(cpa_bl +i);
      //use aes in ecb mode
        if(aes_crypt_ecb( &aesc, AES_ENCRYPT,(const unsigned char *)bff1,(unsigned char*)bff_o)!=0) throwout(_("error in aes transformation"));
  }
  else if(!dofeistel_n(2*blolen, keylen,bff1, bff_o, key2, feistelsteps,k_steps,cpa_blen, cpa_bl, algocode))
      { printf("Error in dofeistel");}
  //else{throwout(_("fatal error in encipher file-> must debug properly !!!")); }


   //falls CBC-Mode in bff_last aufheben
  if(mode==1) memcpy(bff_last,bff_o,2*blolen);

  for( i=0;i<2*blolen;i++)
  {
     fputc((char) *(bff_o + i),cipherfile) ;
  } // schreibt das Produkt der ersten Blockverschl. in den ersten Block des Cipherfiles

  //jetzt das restliche file
  for(j=1;j<blockzahl;j++)
  {
       if((shoprog>0)&&(blockzahl>1))
       {
         //prog bar propagation in main loop
         currpos=(int)( (100*j)/(blockzahl-1));
         if((currpos >lastmax)&&goodp)
         {
            goodp=pbar.Update(currpos);
            lastmax=currpos;
         }
       }

      for(i=0;i<2*blolen;i++)
      {
          bytcount++;
          if(bytcount <= blinds)
          {
              //*(bff1 + i)=(char) (rand() % 256);
              *(bff1 + i)=zuf.get_rbyte();
          }
          else
            *(bff1 + i)= (char) fgetc(plainfile);
      }
      // jetzt bei CBC mode XOREN mit vorigem Cipherblock
      if(mode == 1) //wenn in CBC Mode
      {
        for(i=0;i<2*blolen;i++)
        {
          *(bff1 + i) ^= *(bff_last + i);
        }
      }
      if(algo==_("aes"))
      {
        //use aes in ecb mode, CBC conversion already done
        if(aes_crypt_ecb( &aesc, AES_ENCRYPT,(const unsigned char*)bff1,(unsigned char*)bff_o)!=0) throwout(_("error in aes transformation"));
       }
       else if(!dofeistel_n(2*blolen, keylen,bff1, bff_o, key2, feistelsteps,k_steps,cpa_blen, cpa_bl,algocode))
          { printf("Error in dofeistel");}
// cipher in last sichern
      if(mode == 1) memcpy(bff_last,bff_o,2*blolen);
      for(i=0;i<2*blolen;i++)
      {
         fputc((char) *(bff_o + i),cipherfile) ;
      }
  }
  free(bff1); free(bff_o); free(cpa_bl), free(key2);
  if(mode == 1) free(bff_last);
  fclose(plainfile);
  fclose(cipherfile);
  return true;
}
/****************************************/
/*****************************************************/
bool encipher_pair(wxString appendto, wxString header1, wxString plainwxfile, char* key, int keylen, char feistelsteps, int k_steps, int blolen,int mode, wxString algo,unsigned char shoprog)
                 // verschlüsselt ein File mit key
// wahre Blocklänge 2*blolen, blolen muss größer 25 sein
// alte filelänge long len wird vorangestellt
{
  long long filen=0, plainlen, headlen, blockzahl, readbyts=0, j,newfilen,blinds,bytcount=0;
  FILE *plainfile, *cipherfile, *inner_hdfile;
  char *bff1, *bff_o, *bff_last, *cpa_bl, *key2, *fnam,*hfnam,*cfnam;
  int restlen,i,cpa_blen, algoflag;
  int currpos, lastmax=0;
  bool goodp=true;
  aes_context aesc;
  unsigned int algocode;



  if( (algo==_("F_cnt_1c"))||(algo==_("F_cnt_1d"))||(algo==_("F_cnt_1b")) ) //-> countermode
  {
      return( encipher_cnt_pair(appendto, header1, plainwxfile,key, keylen, k_steps, blolen, 2, algo, shoprog) );
  }
  else if( (algo==_("flight"))||(algo==_("flightx"))||(algo==_("threefish"))||(algo==_("chimera")) ) //-> countermode
  {
      return( encipher_cnt_pair_mono(appendto, header1, plainwxfile,key, keylen, k_steps, blolen, 2, algo, shoprog) );
  }
  algocode=get_code_4_algo(algo); //
  if((blolen<26)&&(algo!=_("aes"))) return false;// kein Platz für die Filelänge!
  if(!wxFile::Exists(appendto))
  {
      throwout(_("File to append cipher to does not exist!!\nAborting."),2);
      return false;
  }
  if(!wxFile::Exists(header1))
  {
      throwout(_("File for first level header does not exist!!\nAborting."),2);
      return false;
  }
  if(!wxFile::Exists(plainwxfile))
  {
      throwout(_("Klardatei existiert nicht!!\nAbbruch."),2);
      return false;
  }
  //determine Lengths
  wxFile tmpfl(header1);
  headlen=tmpfl.Length();
  tmpfl.Close();
  wxFile tmpfl2(plainwxfile);
  plainlen=tmpfl2.Length();
  tmpfl2.Close();
  filen= headlen+plainlen;
  //create all char filenames
  cfnam= (char *) malloc(4*appendto.Length()+2);
  if(cfnam==NULL)
  {
      throwout(_("error! Could not allocate filename buffer?"),2);
      return false;
  }
  strcpy( cfnam, (const char*)appendto.mb_str(wxConvLocal) );// schreibt den Filenamen in fnambuff rein

  fnam= (char *) malloc(4*plainwxfile.Length()+2);
  if(fnam==NULL)
  {
      throwout(_("error! Could not allocate plain-filename buffer?"),2);
      free(cfnam);
      return false;
  }
  strcpy( fnam, (const char*)plainwxfile.mb_str(wxConvLocal) );// schreibt den Filenamen in fnambuff rein

  hfnam= (char *) malloc(4*header1.Length()+2);
  if(hfnam==NULL)
  {
      throwout(_("error! Could not allocate header1-filename buffer?"),2);
      free(cfnam);free(fnam);
      return false;
  }
  strcpy( hfnam, (const char*)header1.mb_str(wxConvLocal) );// schreibt den Filenamen in fnambuff rein
//open all files
  if((cipherfile = fopen(cfnam, "ab"))== NULL)
  {
      free(cfnam);free(fnam);free(hfnam);
      throwout(_("could not open cipherfile"),2);
      return false;
  }
  if((plainfile = fopen(fnam, "rb"))== NULL)
  {
      fclose(cipherfile);
      free(cfnam);free(fnam);free(hfnam);
      throwout(_("could not open plainfile"),2);
      return false;
  }
  if((inner_hdfile = fopen(hfnam, "rb"))== NULL)
  {
      fclose(cipherfile);fclose(plainfile);
      free(cfnam);free(fnam);free(hfnam);
      throwout(_("could not open header_1_file"),2);
      return false;
  }
  // jetzt  setze algoflag
  if(algo ==_("Fleas_1_8"))   { algoflag=0;}
    else if(algo ==_("Fleas_3"))   { algoflag=3;}
     else if(algo ==_("Fleas_4"))   { algoflag=4;}
     else if(algo ==_("Fleas_5"))   { algoflag=5;}
     else if(algo ==_("Fleas_x2"))   { algoflag=6;}
     else if(algo ==_("Fleas_x5"))   { algoflag=7;}
     else if(algo ==_("Fleas_o2"))   { algoflag=8;}
     else if(algo ==_("Fleas_o5"))   { algoflag=9;}
     else if(algo ==_("Fleas_l"))   {algoflag=10;}
     else if(algo ==_("Fleas_ls"))   {  algoflag=11;}
     else if(algo ==_("Fleas_l3"))   {  algoflag=13;}
     else if(algo ==_("Fleas_lc"))   {  algoflag=14;}
     else if(algo ==_("Fleas_ld"))   {  algoflag=15;}
     else if(algo ==_("Fleas_lb"))   { algoflag=16;}
     else if(algo ==_("aes"))   {blolen=8;algoflag=12;}
       else{
            throwout(_("unknown algorithm! \n aborting (1)"));fclose(cipherfile);fclose(plainfile);
            fclose(inner_hdfile);free(cfnam);free(fnam);free(hfnam);return false;
            }
//initialize Progress bar
  wxProgressDialog pbar(_("Verschluesslgs.-Fortschritt"),_("Prozessor muss arbeiten :-)"),100);

  cpa_blen=blolen; //cpa_blen is matching blolen
  /** blolen must be 8 for aes!!  **/
  if((algo == _("aes"))&&(blolen != 8))
  {
      throwout(_("fatal error in aes blocksize!")); return false;
  }
  //create cpa_blocker block and possibly derived key
  key2= (char *) malloc(keylen);
  memcpy(key2,key,keylen); //key2 filled for use
  switch( algoflag)
  {
    case 4 :
      // build derived key for use with Fleas_4
      develop_c(key2,keylen,2);
      // build cpa_blocker with different algorithm
      cpa_bl= develop_b_malloc(keylen, key,4, cpa_blen);
      break;

    case 5 :
      // build derived key for use with Fleas_4
      develop_x(key2,keylen,2,3);
      // build cpa_blocker with different algorithm
      cpa_bl= develop_b_malloc(keylen, key,4, cpa_blen);
      break;

    case 6 :
      // build derived key for use with Fleas_x2
      develop_x1(key2,keylen,2,3);
      // build cpa_blocker with different algorithm
      cpa_bl= develop_x_malloc(keylen, key,4,2, cpa_blen);
      break;

    case 7 :
      // build derived key for use with Fleas_x5
      develop_x1(key2,keylen,2,3);
      // build cpa_blocker with different algorithm
      cpa_bl= develop_x_malloc(keylen, key,4,5, cpa_blen);
      break;

    case 8 :
      // build derived key for use with Fleas_o2
      develop_o(key2,keylen,2,3,0);
      // build cpa_blocker with different algorithm
      cpa_bl= develop_o_malloc(keylen, key,3,2, cpa_blen,0);
      break;

    case 9 :
      // build derived key for use with Fleas_o5
      develop_o(key2,keylen,2,3,2);
      // build cpa_blocker with different algorithm
      cpa_bl= develop_o_malloc(keylen, key,3,5, cpa_blen,2);
      break;

    case 10 :
      // build derived key for use with Fleas_l
      develop_l(key2,keylen,2);
      // build cpa_blocker with different algorithm
      cpa_bl= develop_l_malloc(keylen, key, 3,cpa_blen);
      break;

    case 11 :
      // build derived key for use with Fleas_ls
      develop_l(key2,keylen,2);
      // build cpa_blocker with different algorithm
      cpa_bl= develop_l_malloc(keylen, key, 3,cpa_blen);
      break;

    case 12 :
      // build derived key for use with aes
      develop_o(key2,keylen,2,3,2); //not used
      // build cpa_blocker with different algorithm
      cpa_bl= develop_o_malloc(keylen, key, 3,5,6*blolen,2); //cpablen, erste 16 für cpablock, 2.&3. 16 für key in 2 rounds
      break;

    case 13 :
      // build derived key for use with Fleas_l3
      develop_ls(key2,keylen,3,3,1);
      // build cpa_blocker with different algorithm
      cpa_bl= develop_l_malloc(keylen, key, 3,cpa_blen);
      break;

    case 14 :
      // build derived key for use with Fleas_lc
      develop_ln(key2,keylen,3,3,1);
      // build cpa_blocker with different algorithm
      cpa_bl= develop_l_malloc(keylen, key, 3,cpa_blen);
      break;

    case 15 :
      // build derived key for use with Fleas_ld
      develop_ln2_mt(key2,keylen,3,3,1);
      // build cpa_blocker with different algorithm
      cpa_bl= develop_l_malloc(keylen, key, 4,cpa_blen);
      break;

    case 16 :
      // build derived key for use with Fleas_ld
      develop_ln2_mt(key2,keylen,3,3,1);
      // build cpa_blocker with different algorithm
      cpa_bl= develop_l_malloc(keylen, key, 2,cpa_blen);
      break;

    default:
      cpa_bl=bytmixxmalloc(keylen,key,6,cpa_blen);
      break;
  }


  //bestimme jetzt die Blockzahl
  blockzahl = (filen + 50) / (2*(long long)blolen);
  //
  if( ((filen + 50) % (2*(long long)blolen)) !=0) { blockzahl++; } //jetzt ham wir die richtige Blockzahl
  newfilen = blockzahl*2*(long long)blolen; //neue Filelaenge
  blinds = newfilen - filen;
  bff1 = (char *)malloc(2*blolen); bff_o = (char *)malloc(2*blolen);  //Buffer in and out allozieren
  if(mode == 1) bff_last = (char *)malloc(2*blolen); //bei CBC-mode noch ein zusätzlicher Buffer

#if defined(__WXMSW__)
  sprintf(bff1,"si3:%Ld ",filen); // hat jetzt die echtlänge an den Anfang geschrieben, passt immer in den Buffer min 64 byte
#else
  sprintf(bff1,"si3:%lld ",filen); // hat jetzt die echtlänge an den Anfang geschrieben, passt immer in den Buffer
#endif
  // jetzt restlänge bestimmen
  restlen=(int)(filen-(blockzahl-1)*2*(long long)blolen);
  //differenz mit Zufallszahlen auffüllen
  // kann auch negativ sein: dann Zufallszahlen AUCH in den nächsten Block eintragen
  bytcount=strlen(bff1);
  for( i=strlen(bff1);i<2*blolen;i++) //von ende der Längenangabe bis blockende
  {
      bytcount++;
      if(bytcount>blinds)
      {
          if(readbyts< headlen)
          {
            *(bff1 + i)= (char) fgetc(inner_hdfile);
            readbyts++;
          }
          else
          {
            *(bff1 + i)= (char) fgetc(plainfile);
            readbyts++;
          }
      }
      else *(bff1 + i)= zuf.get_rbyte(); // mit zufallszahlen auffüllen
  } // füllt den Rest mit dem Filerest auf, dazwischen wirds mit zufallszahlen aufgefüllt und wird bei decipher auch weggeschmissen
  // jetzt rausschreiben in cipherfile
  /********* muss noch verschl. werden ************/
//ersten block verschlüsseln
  if(algo==_("aes"))
  {
      //set encipher structure
        if (aes_setkey_enc( &aesc, (const unsigned char *)(cpa_bl+16), 256 )!= 0)
          throwout(_("error in setting aes key context"));
      //xor cpa_blocker to input block
      if(mode==1) for(i=0;i<16;i++) *(bff1+i) ^=  *(cpa_bl +i);
      //use aes in ecb mode
        if(aes_crypt_ecb( &aesc, AES_ENCRYPT,(const unsigned char *)bff1,(unsigned char*)bff_o)!=0) throwout(_("error in aes transformation"));
  }
  else if(!dofeistel_n(2*blolen, keylen,bff1, bff_o, key2, feistelsteps,k_steps,cpa_blen, cpa_bl, algocode))
      { printf("Error in dofeistel");}
  //else{throwout(_("fatal error in encipher file-> must debug properly !!!")); }


   //falls CBC-Mode in bff_last aufheben
  if(mode==1) memcpy(bff_last,bff_o,2*blolen);

  for( i=0;i<2*blolen;i++)
  {
     fputc((char) *(bff_o + i),cipherfile) ;
  } // schreibt das Produkt der ersten Blockverschl. in den ersten Block des Cipherfiles

  //jetzt das restliche file
  for(j=1;j<blockzahl;j++)
  {
       if((shoprog>0)&&(blockzahl>1))
       {
         //prog bar propagation in main loop
         currpos=(int)( (100*j)/(blockzahl-1));
         if((currpos >lastmax)&&goodp)
         {
            goodp=pbar.Update(currpos);
            lastmax=currpos;
         }
       }

      for(i=0;i<2*blolen;i++)
      {
          bytcount++;
          if(bytcount <= blinds)
          {
              //*(bff1 + i)=(char) (rand() % 256);
              *(bff1 + i)=zuf.get_rbyte();
          }
          else
          {
            if(readbyts< headlen)
            {
              *(bff1 + i)= (char) fgetc(inner_hdfile);
              readbyts++;
            }
            else
            {
              *(bff1 + i)= (char) fgetc(plainfile);
              readbyts++;
            }
          }
      }
      // jetzt bei CBC mode XOREN mit vorigem Cipherblock
      if(mode == 1) //wenn in CBC Mode
      {
        for(i=0;i<2*blolen;i++)
        {
          *(bff1 + i) ^= *(bff_last + i);
        }
      }
      if(algo==_("aes"))
      {
        //use aes in ecb mode, CBC conversion already done
        if(aes_crypt_ecb( &aesc, AES_ENCRYPT,(const unsigned char*)bff1,(unsigned char*)bff_o)!=0) throwout(_("error in aes transformation"));
       }
       else if(!dofeistel_n(2*blolen, keylen,bff1, bff_o, key2, feistelsteps,k_steps,cpa_blen, cpa_bl,algocode))
          { printf("Error in dofeistel");}
// cipher in last sichern
      if(mode == 1) memcpy(bff_last,bff_o,2*blolen);
      for(i=0;i<2*blolen;i++)
      {
         fputc((char) *(bff_o + i),cipherfile) ;
      }
  }
  //cleanup
  free(bff1); free(bff_o); free(cpa_bl), free(key2);
  if(mode == 1) free(bff_last);
  fclose(cipherfile);fclose(plainfile);fclose(inner_hdfile);
  free(cfnam);free(fnam);free(hfnam);
  return true;
}
/****************************************/
/**************************************************************/
char * develop_b_malloc(int blolen, char* bladwell, int rounds, int outlen)
// Funktion mixert und permutiert den block massiv durch
// output erfolgt in der Länge outlen,
{
    int i,len;
    char *inad;

    if(outlen>blolen) len=outlen;
    else len=blolen;
    inad= (char *)malloc(len);
    memcpy(inad,bladwell,blolen);//cage fill 1
    for(i=blolen;i<len;i++) //cage fill rest
    {
        *(inad+i)= *(bladwell + i%blolen);
    }
    if(!develop_b(inad,len, rounds)) {   //do oneway function
        throwout(_("error in bytmixmalloc"));
        free(inad);
        return NULL;
    };
    inad= (char *) realloc(inad,outlen);
    return inad;
}
/*********************************************/
char * develop_x_malloc(int blolen, char* bladwell, int rounds, int paths, int outlen)
// Funktion mixert und permutiert den block massiv durch
// output erfolgt in der Länge outlen,
{
    int i,len;
    char *inad;

    if(outlen>blolen) len=outlen;
    else len=blolen;
    inad= (char *)malloc(len);
    memcpy(inad,bladwell,blolen);//cage fill 1
    for(i=blolen;i<len;i++) //cage fill rest
    {
        *(inad+i)= *(bladwell + i%blolen);
    }
    if(!develop_x1(inad,len, rounds,paths)) {   //do oneway function
        throwout(_("error in bytmixmalloc"));
        free(inad);
        return NULL;
    };
    inad= (char *) realloc(inad,outlen);
    return inad;
}
/*********************************************/
char * develop_o_malloc(int blolen, char* bladwell, int rounds, int paths, int outlen, char alg_flag)
// Funktion mixert und permutiert den block massiv durch
// output erfolgt in der Länge outlen,
{
    int i,len;
    char *inad;

    if(outlen>blolen) len=outlen;
    else len=blolen;
    inad= (char *)malloc(len);
    memcpy(inad,bladwell,blolen);//cage fill 1
    for(i=blolen;i<len;i++) //cage fill rest
    {
        *(inad+i)= *(bladwell + i%blolen);
    }
    if(!develop_o(inad,len, rounds,paths, alg_flag)) {   //do oneway function
        throwout(_("error in bytmixmalloc"));
        free(inad);
        return NULL;
    };
    inad= (char *) realloc(inad,outlen);
    return inad;
}
/*********************************************/
char * develop_l_malloc(int blolen, char* bladwell, int rounds, int outlen)
// output erfolgt in der Länge outlen,
{
    int i,len,j;
    char *inad;

    if(outlen>blolen) len=outlen;
    else len=blolen;
    inad= (char *)malloc(len);
    memcpy(inad,bladwell,blolen);//cage fill 1
    j=0;
    for(i=blolen;i<len;i++) //cage fill rest
    {
        if(j>=blolen) j-=blolen;
        *(inad+i)= *(bladwell + j);
        j++;
    }
    if(!develop_l(inad,len, rounds)) {   //do oneway function
        throwout(_("error in develop_l_malloc"));
        free(inad);
        return NULL;
    };
    inad= (char *) realloc(inad,outlen);
    return inad;
}
/**************************************************************//**************************************************************/
char * bytmixxmalloc(int blolen, char* bladwell, int rounds, int outlen)
// Funktion mixert und permutiert den block massiv durch
// output erfolgt in der Länge outlen,
{
    int i,len;
    char *outad,*inad;

    if(outlen>blolen) len=outlen;
    else len=blolen;
    outad= (char *)malloc(len);
    inad= (char *)malloc(len);
    memcpy(inad,bladwell,blolen);//cage fill 1
    for(i=blolen;i<len;i++) //cage fill rest
    {
        *(inad+i)= *(bladwell + i%blolen);
    }
    if(!bytmixx_d(len,inad,outad, rounds)) {   //do oneway function
        throwout(_("error in bytmixmalloc"));
        free(outad); free(inad);
        return NULL;
    };
    free(inad);
    outad= (char *) realloc(outad,outlen);
    return outad;
}
/*********************************************/
bool bytmixx_d(int blolen, char* bladwell, char* pertxt, int rounds)
// Funktion mixert und permutiert den block massiv durch
//zwei reihenfolgen, zum schluss xoren
// soll: Sicherheit erhoehen
//to do: Test chiquad
{
   int i;
   char *pertxt2;

    // zweiten block allozieren
   pertxt2= (char *) malloc(blolen);

   memcpy(pertxt,bladwell,blolen);
   memcpy(pertxt2,bladwell,blolen); //parallel auf die Schiene setzen

   for(i=0;i<rounds;i++)
   {
     if( !bytinteg(blolen,pertxt,127,1)) return false;
     if( !bytperm_x1(blolen, pertxt, pertxt, 1)) return false;
     if( !bytxor_x1(blolen, pertxt, pertxt, 1)) return false;
   }
   //zweiter pfad in anderer Reihenfolge
   for(i=0;i<rounds;i++)
   {
     if( !bytinteg(blolen,pertxt2,63,7)) return false;;
     if( !bytxor_x1(blolen, pertxt2, pertxt2, 1)) return false;
     if( !bytperm_x1(blolen, pertxt2, pertxt2, 1)) return false;
   }
   //ergebnis xoren
   for(i=0;i<blolen;i++) *(pertxt +i) ^= *(pertxt2+i);
   free(pertxt2);
   return true;
}

/**************************************************************/
bool bytmixx(int blolen, char* bladwell, char* pertxt, int rounds)
// Funktion mixert und permutiert den block massiv durch
{
   int i;

   memcpy(pertxt,bladwell,blolen);

   for(i=0;i<rounds;i++)
   {
     if( !bytinteg(blolen,pertxt,127,1));
     if( !bytperm_x1(blolen, pertxt, pertxt, 1)) return false;
     //if( !bytinteg(blolen,pertxt,127,1));
     if( !bytxor_x1(blolen, pertxt, pertxt, 1)) return false;
   }
   return true;
}
/*********************************************/
/*****************************************************/
bool decipher_file_cntmode_mt4ecc(ulong32 deadhead, wxString cnam, wxString* pnampt, char* key, int keylen, int k_steps, int blolen, int mode, wxString algo, unsigned char shoprog)
//terrible spaghetti to optimize ecc deciphering(maintenance nightmare optimization sigh!!)
{
  long long filen=0, blockzahl, j;
  FILE *plainfile, *cipherfile;
  wxString temppname,temphname,secsep=_("X:"), pathcomp,hs;
  char *iv, *ic, *ic1,*ic2,*ic3,*ip,hc,buff[1602],c;
  int i,ichar, currpos,lastmax=0,fl1,buflen,lbl,i2;
  unsigned int algoflag;
  char *chkbuf,*outbuf;
  longnumber ivno;
  bool goodp=true,switched=false;
  kdev_thr *pt0, *pt1,*pt2,*pt3;
  bool r0,r1,r2,r3;

  //some primitive error checking
  if(blolen<1) return false;
  if(!wxFile::Exists(cnam))
  {
      throwout(_("Chiffrat-Datei existiert nicht!!\nAbbruch."),2);
      return false;
  }
  //determine Length
  wxFile tmpfl(cnam);
  filen=tmpfl.Length();
  tmpfl.Close();
  // jetzt plainfile öffnen ohne anhängsel "_1c" etc und setze algoflag
  wxFFile ciph_fil(cnam,_("rb"));
  if(!ciph_fil.IsOpened()){throwout(_("Konnte Chiffrat-Datei nicht oeffnen\nAbbruch"),5); return false;}
  cipherfile=ciph_fil.fp();
  temphname=cnam+_("_hl");
  temppname=cnam+_("_hp");

  if(algo ==_("F_cnt_1c"))   { algoflag=9;}
  else if(algo ==_("F_cnt_1d"))   { algoflag=10;}
  else if(algo ==_("F_cnt_1b"))   { algoflag=11;}
  else if(algo ==_("flight"))   { algoflag=15;}
  else if(algo ==_("flightx"))   { algoflag=16;}
       else{throwout(_("unknown algorithm! \n aborting (2)"));return false;}

  wxFFile plain_fil(temphname,_("wb"));
  if(!plain_fil.IsOpened()) {throwout(_("couldn't open plainheader for writing\naborting"),5); return false;}
  plainfile=plain_fil.fp();

//initialize Progress bar
  wxProgressDialog pbar(_("Entschluesselungs-Fortschritt"),_("Prozessor arbeitet sich warm :-)"),100);
// set checksequence for header end
  chkbuf= (char*) malloc(4*secsep.Len()+2);
  buflen=secsep.Len();
  outbuf= (char*) malloc(buflen+2);
  //sequence set
  strcpy( chkbuf, (const char*)secsep.mb_str(wxConvUTF8) );// schreibt den Filenamen in fnambuff rein
  for(i=0;i<buflen;i++){outbuf[i]=0;} //set starter value
  //determine blockzahl
  blockzahl=(filen-deadhead)/blolen;
  if(blockzahl >0)blockzahl--;
  //create IV from random then lock with key
  iv=(char *) malloc(blolen);
  ic=(char *) malloc(blolen);
  ic1=(char *) malloc(blolen);
  ic2=(char *) malloc(blolen);
  ic3=(char *) malloc(blolen);
  ip=(char *) malloc(blolen);
  //read locked IV
  fl1=fseek(cipherfile,deadhead, SEEK_SET); //dead header überspringen
  for(i=0;i<blolen;i++)
  {
      ichar=fgetc(cipherfile) ;
      if(ichar==EOF)
      {
          free(iv);free(ic);free(ip);free(chkbuf);
          throwout(_("Fehler, konnte IV nicht auslesen"),5);
          return false;
      }
      *(ic+i)=(char) ichar;
  }
  if(mode>0)
      dofeistel_n(blolen, keylen, ic,iv, key, -4, mode, 0, NULL, algoflag);//-4Steps, mode-fold owf, no cpa blocker
  else memcpy(iv,ic,blolen);


  //put IV into longnumber
  ivno.resizelonu(blolen+2);
  memcpy(ivno.ad, iv, blolen);
  ivno.setsize(true);
  //loop complete blocks with progbar update
  for(j=0;j<blockzahl-4;j+=4)
  {
       if((shoprog>0)&&(blockzahl>1))
       {
         //prog bar propagation in main loop
         currpos=(int)( (100*j)/(blockzahl-1));
         if((currpos >lastmax)&&goodp)
         {
            goodp=pbar.Update(currpos);
            lastmax=currpos;
         }
       }
      //get key-developed number block from ivno
      memcpy(ic,(char *)(ivno.ad),blolen);
      //do it threaded
      //k_develop_f4(blolen, ic,0,NULL,keylen,key,k_steps,algoflag); //14 ->Fleas_lc 15 ->fleas ld
        pt0=new kdev_thr(&r0,blolen, ic,0,NULL,keylen,key,k_steps,algoflag);
        if ( pt0->Create() != wxTHREAD_NO_ERROR )
            {
                throwout(_("Can't create the thread!"),3);
                delete pt0;
                pt0=NULL;
            }
        else
          {
            if ( pt0->Run() != wxTHREAD_NO_ERROR )
            {
                throwout(_("Can't Run the thread!"),3);
                delete pt0;
                pt0 = NULL;
            }
          }
          if(pt0==NULL)  //error  go to harakiri
          {
               //close files
                ciph_fil.Close();
                plain_fil.Close();
                //free buffers
                free(iv); free(ip); free(ic);free(ic1);free(ic2);free(ic3);free(chkbuf);
                throwout(_("error in encipher file cntmode multithread t0"),3);
                return false;
          }


      ivno.inc();
      memcpy(ic1,(char *)(ivno.ad),blolen);
      //k_develop_f4(blolen, ic1,0,NULL,keylen,key,k_steps,algoflag); //14 ->Fleas_lc 15 ->fleas ld
        pt1=new kdev_thr(&r1,blolen, ic1,0,NULL,keylen,key,k_steps,algoflag);
        if ( pt1->Create() != wxTHREAD_NO_ERROR )
            {
                throwout(_("Can't create the thread!"),3);
                delete pt1;
                pt1=NULL;
            }
        else
          {
            if ( pt1->Run() != wxTHREAD_NO_ERROR )
            {
                throwout(_("Can't Run the thread!"),3);
                delete pt1;
                pt1 = NULL;
            }
          }
          if(pt1==NULL)  //error  go to harakiri
          {
               pt0->Wait();
               //close files
                ciph_fil.Close();
                plain_fil.Close();
                //free buffers
                free(iv); free(ip); free(ic);free(ic1);free(ic2);free(ic3);free(chkbuf);
                throwout(_("error in encipher file cntmode multithread t1"),3);
                return false;
          }

      ivno.inc();
      memcpy(ic2,(char *)(ivno.ad),blolen);
      //k_develop_f4(blolen, ic2,0,NULL,keylen,key,k_steps,algoflag); //14 ->Fleas_lc 15 ->fleas ld
        pt2=new kdev_thr(&r2,blolen, ic2,0,NULL,keylen,key,k_steps,algoflag);
        if ( pt2->Create() != wxTHREAD_NO_ERROR )
            {
                throwout(_("Can't create the thread!"),3);
                delete pt2;
                pt2=NULL;
            }
        else
          {
            if ( pt2->Run() != wxTHREAD_NO_ERROR )
            {
                throwout(_("Can't Run the thread!"),3);
                delete pt2;
                pt2 = NULL;
            }
          }
          if(pt2==NULL)  //error  go to harakiri
          {
               pt0->Wait();
               pt1->Wait();
               //close files
                ciph_fil.Close();
                plain_fil.Close();
                //free buffers
                free(iv); free(ip); free(ic);free(ic1);free(ic2);free(ic3);free(chkbuf);
                throwout(_("error in encipher file cntmode multithread t1"),3);
                return false;
          }

      ivno.inc();
      memcpy(ic3,(char *)(ivno.ad),blolen);
      //k_develop_f4(blolen, ic3,0,NULL,keylen,key,k_steps,algoflag); //14 ->Fleas_lc 15 ->fleas ld
        pt3=new kdev_thr(&r3,blolen, ic3,0,NULL,keylen,key,k_steps,algoflag);
        if ( pt3->Create() != wxTHREAD_NO_ERROR )
            {
                throwout(_("Can't create the thread!"),3);
                delete pt3;
                pt3=NULL;
            }
        else
          {
            if ( pt3->Run() != wxTHREAD_NO_ERROR )
            {
                throwout(_("Can't Run the thread!"),3);
                delete pt3;
                pt3 = NULL;
            }
          }
          if(pt3==NULL)  //error  go to harakiri
          {
               pt0->Wait();
               pt1->Wait();
               pt2->Wait();
               //close files
                ciph_fil.Close();
                plain_fil.Close();
                //free buffers
                free(iv); free(ip); free(ic);free(ic1);free(ic2);free(ic3);free(chkbuf);
                throwout(_("error in encipher file cntmode multithread t1"),3);
                return false;
          }
      //xor with ciphertext block
      //write to plain file
      //increment ivno
      ivno.inc();
//wait for proc0 to finish
       pt0->Wait();

      if(switched)
      {
      //read in ciphertext block
         for(i=0;i<blolen;i++)
         {
             hc= (char) fgetc(cipherfile);
             hc ^= *(ic+i);
             fputc( hc,plainfile) ;
         }
//wait for proc1 to finish
         pt1->Wait();
      //read in ciphertext block
         for(i=0;i<blolen;i++)
         {
            hc= (char) fgetc(cipherfile);
            hc ^= *(ic1+i);
            fputc( hc,plainfile) ;
         }

//wait for proc1 to finish
         pt2->Wait();
      //read in ciphertext block
         for(i=0;i<blolen;i++)
         {
             hc= (char) fgetc(cipherfile);
             hc ^= *(ic2+i);
             fputc( hc,plainfile) ;
         }

//wait for proc1 to finish
         pt3->Wait();
      //read in ciphertext block
         for(i=0;i<blolen;i++)
         {
             hc= (char) fgetc(cipherfile);
             hc ^= *(ic3+i);
             fputc( hc,plainfile) ;
         }
      }
      else //not switched yet
      {
      //read in ciphertext block
         for(i=0;i<blolen;i++)
         {
             hc= (char) fgetc(cipherfile);
             hc ^= *(ic+i);
             fputc( hc,plainfile) ;
//walk  marker
              memmove(outbuf,outbuf+1,buflen -1);
              *(outbuf+buflen-1)=hc;
              if(!switched && memcmp(outbuf,chkbuf,buflen)==0)  //key hit, now switch aoutput file
              {
                 switched= true;
              //close old plainfile and open real one
                 if(!plain_fil.Close())
                 {
                  throwout(_("Warning, could not close temp file"));
                 }
                 if(!plain_fil.Open(temppname,_("wb")) )
                 {
                  throwout(_("Fehler, konnte Klardatei nicht lesen!\nAbbruch."),5);
                  free(iv); free(ip); free(ic); free(ic1); free(ic2); free(ic3);free(chkbuf);

                   ciph_fil.Close();
                  return false;
                 }
                  if(!plain_fil.IsOpened())
                    {
                        throwout(_("couldn't open plainheader for writing\naborting"),5);
                        ciph_fil.Close();
                        return false;
                    }
                 plainfile=plain_fil.fp();
               }
         }
//wait for proc1 to finish
         pt1->Wait();
      //read in ciphertext block
         for(i=0;i<blolen;i++)
         {
            hc= (char) fgetc(cipherfile);
            hc ^= *(ic1+i);
            fputc( hc,plainfile) ;
//walk  marker
              memmove(outbuf,outbuf+1,buflen -1);
              *(outbuf+buflen-1)=hc;
              if(!switched && memcmp(outbuf,chkbuf,buflen)==0)  //key hit, now switch aoutput file
              {
                 switched= true;
              //close old plainfile and open real one
                 if(!plain_fil.Close())
                 {
                  throwout(_("Warning, could not close temp file"));
                 }
                 if(!plain_fil.Open(temppname,_("wb")) )
                 {
                  throwout(_("Error, could not open plain file!\naborting."),5);
                  free(iv); free(ip); free(ic); free(ic1); free(ic2); free(ic3);free(chkbuf);
                  return false;
                 }
                  if(!plain_fil.IsOpened()) {throwout(_("couldn't open plainheader for writing\naborting"),5); return false;}
                 plainfile=plain_fil.fp();
               }
         }

//wait for proc1 to finish
         pt2->Wait();
      //read in ciphertext block
         for(i=0;i<blolen;i++)
         {
             hc= (char) fgetc(cipherfile);
             hc ^= *(ic2+i);
             fputc( hc,plainfile) ;
//walk  marker
              memmove(outbuf,outbuf+1,buflen -1);
              *(outbuf+buflen-1)=hc;
              if(!switched && memcmp(outbuf,chkbuf,buflen)==0)  //key hit, now switch aoutput file
              {
                 switched= true;
              //close old plainfile and open real one
                 if(!plain_fil.Close())
                 {
                  throwout(_("Warning, could not close temp file"));
                 }
                 if(!plain_fil.Open(temppname,_("wb")) )
                 {
                  throwout(_("Error, could not open plain file!\naborting."),5);
                  free(iv); free(ip); free(ic); free(ic1); free(ic2); free(ic3);free(chkbuf);
                  ciph_fil.Close();
                  return false;
                 }
                  if(!plain_fil.IsOpened()) {throwout(_("couldn't open plainheader for writing\naborting"),5); return false;}
                 plainfile=plain_fil.fp();
               }
         }

//wait for proc1 to finish
         pt3->Wait();
      //read in ciphertext block
         for(i=0;i<blolen;i++)
         {
             hc= (char) fgetc(cipherfile);
             hc ^= *(ic3+i);
             fputc( hc,plainfile) ;
//walk  marker
              memmove(outbuf,outbuf+1,buflen -1);
              *(outbuf+buflen-1)=hc;
              if(!switched && memcmp(outbuf,chkbuf,buflen)==0)  //key hit, now switch aoutput file
              {
                 switched= true;
              //close old plainfile and open real one
                 if(!plain_fil.Close())
                 {
                  throwout(_("Warning, could not close temp file"));
                 }
                 if(!plain_fil.Open(temppname,_("wb")) )
                 {
                  throwout(_("Error, could not open plain file!\naborting."),5);
                  free(iv); free(ip); free(ic); free(ic1); free(ic2); free(ic3);free(chkbuf);
                  ciph_fil.Close();
                  return false;
                 }
                  if(!plain_fil.IsOpened()) {throwout(_("couldn't open plainheader for writing\naborting"),5); return false;}
                 plainfile=plain_fil.fp();
               }
         }
      }

  }
  //get rest block from plainfile
  //get key-developed number block from ivno
  //xor bytewise and write to cipherfile
  //do it with the rest
  for(;j<blockzahl;j++)
  {
        memcpy(ic1,(char *)(ivno.ad),blolen);
        k_develop_f4(blolen, ic1,0,NULL,keylen,key,k_steps,algoflag); //14 ->Fleas_lc 15->Fleas_ld
        for(i=0;i<blolen;i++)
        {
            ichar=fgetc(cipherfile);
            if(ichar==EOF) break;
            hc=(char) ichar;
            hc ^= *(ic1 + i);
            fputc( hc,plainfile) ;
            //walk  marker
            memmove(outbuf,outbuf+1,buflen -1);
            *(outbuf+buflen-1)=hc;
              if(!switched && memcmp(outbuf,chkbuf,buflen)==0)  //key hit, now switch aoutput file
              {
                 switched= true;
              //close old plainfile and open real one
                 if(!plain_fil.Close())
                 {
                  throwout(_("Warning, could not close temp file"));
                 }
                 if(!plain_fil.Open(temppname,_("wb")) )
                 {
                  throwout(_("Error, could not open plain file!\naborting."),5);
                  free(iv); free(ip); free(ic); free(ic1); free(ic2); free(ic3);free(chkbuf);
                  ciph_fil.Close();
                  return false;
                 }
                  if(!plain_fil.IsOpened())
                    {
                        throwout(_("couldn't open plainheader for writing\naborting"),5);
                        ciph_fil.Close();
                        return false;
                    }
                 plainfile=plain_fil.fp();
               }
        }
        ivno.inc();
  }
  memcpy(ic1,(char *)(ivno.ad),blolen);
  k_develop_f4(blolen, ic1,0,NULL,keylen,key,k_steps,algoflag); //14 ->Fleas_lc 15->Fleas_ld
  i=0;
  do
  {
      ichar=fgetc(cipherfile);
      if(ichar==EOF) break;
      hc=(char) ichar;
      hc ^= *(ic1 + i);
      fputc( hc,plainfile) ;
//walk  marker
              memmove(outbuf,outbuf+1,buflen -1);
              *(outbuf+buflen-1)=hc;
              if(!switched && memcmp(outbuf,chkbuf,buflen)==0)  //key hit, now switch aoutput file
              {
                 switched= true;
              //close old plainfile and open real one
                 if(!plain_fil.Close())
                 {
                  throwout(_("Warning, could not close temp file"));
                 }
                 if(!plain_fil.Open(temppname,_("wb")) )
                 {
                  throwout(_("Error, could not open plain file!\naborting."),5);
                  free(iv); free(ip); free(ic); free(ic1); free(ic2); free(ic3);free(chkbuf);
                  ciph_fil.Close();
                  return false;
                 }
                  if(!plain_fil.IsOpened())
                    {
                        throwout(_("couldn't open plainheader for writing\naborting"),5);
                        ciph_fil.Close();
                        return false;
                    }
                 plainfile=plain_fil.fp();
               }
      i++;
  } while(ichar != EOF);
    //free buffers
    free(iv); free(ip); free(ic); free(ic1); free(ic2); free(ic3);free(chkbuf);

  //now do some deletion and renaming
  //open helper file and read  in name
  if(!ciph_fil.Close())
              {
                 throwout(_("Warning, could not close cipher file"));
              }
  if(!plain_fil.Close())
              {
                 throwout(_("Warning, could not close plain file"));
              }
  if(!plain_fil.Open(temphname,_("rb")) )
              {
                  throwout(_("Error, could not open header file 4 reading!\naborting."),5);
              }
  plainfile=plain_fil.fp();
  for(i=0;i<30;i++)
         {
              lbl=fscanf(plainfile,"%c",&c);
         }
  lbl=fscanf(plainfile," fnam: %400s ",buff);
  if(lbl!=1)
     {
        throwout(_("Error could not read plain filename\n exiting without renaming and without removing plain header!"),5);
        plain_fil.Close();
        return false;
     }
  else
     {
      if((pnampt!=NULL)&&(*pnampt!= _("")))
      {
         if(!wxRenameFile(temppname,*pnampt))
         {
             //rename plainfile
             throwout(_("error, could not rename plain file"),5);
         }
         if(!plain_fil.Close())
         {
             throwout(_("warning!\ncould not close:\n")+temphname, 5);
         }
         if(!wxRemoveFile(temphname))
         {
             //remove plain header
             throwout(_("error, could not remove plain header"),5);
         }
      }
      else
      {
         hs=wxString::FromUTF8(buff);
         //prepend path and sep
         pathcomp=((wxFileName) temppname).GetPath(wxPATH_GET_VOLUME|wxPATH_GET_SEPARATOR);

//dirty patch: changing >< to underscore in hs
  i2=0;
  i2 += hs.Replace(_("<"), _("_"));
  i2 += hs.Replace(_(">"), _("_"));
  if(i2>0) throwout(_("Unsichere Zeichen im Dateinamen \n >,< ersetzt durch _"),5);
//end dirty patch

         *pnampt= pathcomp+ hs;

         if(wxFileExists(*pnampt))
         {
           throwout(_("Achtung, Datei mit originalem Namen existiert schon\nwird umbenannt zu *_old und fahre fort"));
           if(wxFileExists(*pnampt+_("_old"))) wxRemoveFile(*pnampt+_("_old"));
           wxRenameFile(*pnampt, *pnampt+_("_old"));
         }
         if(!wxRenameFile(temppname,*pnampt))
         {
             //rename plainfile
             throwout(_("error, could not rename plain file"),5);
         }
         if(!plain_fil.Close())
         {
             throwout(_("warning!\ncould not close:\n")+temphname, 5);
         }
         if(!wxRemoveFile(temphname))
         {
             //remove plain header
             throwout(_("error, could not remove plain header"),5);
         }
      }
  }
  return true;
}
/*****************************************************/
/***********************************************/
bool decipher_file_cntmode_mt(ulong32 deadhead, char* fnam, char* key, int keylen, int k_steps, int blolen, int mode, wxString algo, unsigned char shoprog)
//k_steps: number of owf-steps,   mode:0-clear number, 1number of lonulock steps for number
{
   int namlen;
   char *sfnam;
   bool res;

  //some primitive error checking
  if(strlen(fnam) > 598) return false;
  namlen=strlen(fnam);
  wxString fnwx(fnam, wxConvFile);
  if(!wxFile::Exists(fnwx))
  {
      throwout(_("Chiffratdatei existiert nicht!!\nAbbruch."),2);
      return false;
  }
  // jetzt plainfile öffnen ohne anhängsel "_1c" etc und setze algoflag
    //create new sfnam "shadow fnam" buffer
    sfnam=(char *)malloc(strlen(fnam)+2);
    strcpy(sfnam,fnam);
    if(fnwx.Right(3)== _("_1c")) memcpy(sfnam+namlen-3,"\0",1);
    else if(fnwx.Right(3)== _("_1d")) memcpy(sfnam+namlen-3,"\0",1);
      else if(fnwx.Right(3)== _("_1b")) memcpy(sfnam+namlen-3,"\0",1);
       else if(fnwx.Right(3)== _("_fl")) memcpy(sfnam+namlen-3,"\0",1);
        else if(fnwx.Right(3)== _("_fx")) memcpy(sfnam+namlen-3,"\0",1);
         else if(fnwx.Right(5)== _(".ciph")) memcpy(sfnam+namlen-5,"\0",1);
         else if(fnwx.Right(5)== _("_ciph")) memcpy(sfnam+namlen-5,"\0",1);
           else //weird cipher extension
            {
                throwout(_("Komische Namenserweiterung der Chiffratdatei!\n"),2);
                sfnam= (char *) realloc( sfnam, strlen(fnam + 8));
                memcpy(sfnam+namlen,".pt\0",1);
            }
  res= decipher_file_cntmode_mt(deadhead, fnam, sfnam, key, keylen, k_steps, blolen, mode, algo, shoprog);
  free(sfnam);
  return res;
}
/*********************************************/
bool decipher_file_cntmode_mt(ulong32 deadhead, char* fnam, char *pnam, char* key, int keylen, int k_steps, int blolen, int mode, wxString algo, unsigned char shoprog)
//k_steps: number of owf-steps,   mode:0-clear number, 1number of lonulock steps for number
//
{
  long long filen=0, blockzahl, j;
  FILE *plainfile, *cipherfile;
  char *iv, *ic,*ic1,*ic2,*ic3,*ip,hc;
  int i,ichar, currpos,namlen,lastmax=0;
  unsigned int algoflag;
  longnumber ivno;
  bool goodp=true;
  kdev_thr *pt0, *pt1,*pt2,*pt3;
  bool r0,r1,r2,r3;

  //some primitive error checking
  if(strlen(fnam) > 598) return false;
  namlen=strlen(fnam);
  wxString fnwx(fnam,wxConvFile);
  if(!wxFile::Exists(fnwx))
  {
      throwout(_("Chiffrat-Datei existiert nicht!!\nAbbruch."),2);
      return false;
  }
  //determine Length
  wxFile tmpfl(fnwx);
  filen=tmpfl.Length();
  tmpfl.Close();
  if((cipherfile = fopen(fnam, "rb"))== NULL) return false;

  if(algo ==_("F_cnt_1c"))   { algoflag=9;}
  else if(algo ==_("F_cnt_1d"))   { algoflag=10;}
  else if(algo ==_("F_cnt_1b"))   { algoflag=11;}
  else if(algo ==_("flight"))   { algoflag=15;}
  else if(algo ==_("flightx"))   { algoflag=16;}
    /*else if(algo ==_("Fleas_3"))   {strncat(sfnam,"_f3",3); algoflag=3;}
     else if(algo ==_("aes"))   {strncat(sfnam,"_ae",3);blolen=8;algoflag=12;}*/
       else{throwout(_("unknown algorithm! \n aborting (2)"));return false;}
  if((plainfile = fopen(pnam, "wb"))== NULL)
  {
    return false;
  }

//initialize Progress bar
  wxProgressDialog pbar(_("Entschluesselungs-Fortschritt"),_("Prozessor arbeitet hart :-)"),100);
  //determine blockzahl
  blockzahl=(filen-deadhead)/blolen;
  if(blockzahl >0)blockzahl--;
  //create IV from random then lock with key
  iv=(char *) malloc(blolen);
  ic=(char *) malloc(blolen);
  ic1=(char *) malloc(blolen);
  ic2=(char *) malloc(blolen);
  ic3=(char *) malloc(blolen);
  ip=(char *) malloc(blolen);
  //read locked IV
  fseek(cipherfile,deadhead, SEEK_SET); //dead header überspringen
  for(i=0;i<blolen;i++)
  {
      ichar=fgetc(cipherfile) ;
      if(ichar==EOF)
      {
          free(iv);free(ic);free(ip);
          throwout(_("Fehler, konnte IV nicht lesen."),5);
          return false;
      }
      *(ic+i)=(char) ichar;
  }
  wxString primalgo;
  if(algoflag==9) primalgo=_("Fleas_lc"); //obsolete
  else if(algoflag==10) primalgo=_("Fleas_ld"); //obsolete
  else if(algoflag==11) primalgo=_("Fleas_lb"); //obsolete
  else if(algoflag==15) primalgo=_("flight"); //obsolete
  else if(algoflag==16) primalgo=_("flightx"); //obsolete
  if(mode>0)
      dofeistel(blolen, keylen, ic,iv, key, -4, mode, 0, NULL, primalgo);//-4Steps, mode-fold owf, no cpa blocker
  else memcpy(iv,ic,blolen);
  //put IV into longnumber
  ivno.resizelonu(blolen+2);
  memcpy(ivno.ad, iv, blolen);
  ivno.setsize(true);
  //loop complete blocks with progbar update
  for(j=0;j<blockzahl-4;j+=4)
  {
       if((shoprog>0)&&(blockzahl>1))
       {
         //prog bar propagation in main loop
         currpos=(int)( (100*j)/(blockzahl-1));
         if((currpos >lastmax)&&goodp)
         {
            goodp=pbar.Update(currpos);
            lastmax=currpos;
         }
       }
      //get key-developed number block from ivno
      memcpy(ic,(char *)(ivno.ad),blolen);
      //do it threaded
      //k_develop_f4(blolen, ic,0,NULL,keylen,key,k_steps,algoflag); //14 ->Fleas_lc 15 ->fleas ld
        pt0=new kdev_thr(&r0,blolen, ic,0,NULL,keylen,key,k_steps,algoflag);
        if ( pt0->Create() != wxTHREAD_NO_ERROR )
            {
                throwout(_("Can't create the thread!"),3);
                delete pt0;
                pt0=NULL;
            }
        else
          {
            if ( pt0->Run() != wxTHREAD_NO_ERROR )
            {
                throwout(_("Can't Run the thread!"),3);
                delete pt0;
                pt0 = NULL;
            }
          }
          if(pt0==NULL)  //error  go to harakiri
          {
               //close files
                fclose(cipherfile);
                fclose(plainfile);
                //free buffers
                free(iv); free(ip); free(ic);free(ic1);free(ic2);free(ic3);
                throwout(_("error in encipher file cntmode multithread t0"),3);
                return false;
          }


      ivno.inc();
      memcpy(ic1,(char *)(ivno.ad),blolen);
      //k_develop_f4(blolen, ic1,0,NULL,keylen,key,k_steps,algoflag); //14 ->Fleas_lc 15 ->fleas ld
        pt1=new kdev_thr(&r1,blolen, ic1,0,NULL,keylen,key,k_steps,algoflag);
        if ( pt1->Create() != wxTHREAD_NO_ERROR )
            {
                throwout(_("Can't create the thread!"),3);
                delete pt1;
                pt1=NULL;
            }
        else
          {
            if ( pt1->Run() != wxTHREAD_NO_ERROR )
            {
                throwout(_("Can't Run the thread!"),3);
                delete pt1;
                pt1 = NULL;
            }
          }
          if(pt1==NULL)  //error  go to harakiri
          {
               pt0->Wait();
               //close files
                fclose(cipherfile);
                fclose(plainfile);
                //free buffers
                free(iv); free(ip); free(ic);free(ic1);free(ic2);free(ic3);
                throwout(_("error in encipher file cntmode multithread t1"),3);
                return false;
          }

      ivno.inc();
      memcpy(ic2,(char *)(ivno.ad),blolen);
      //k_develop_f4(blolen, ic2,0,NULL,keylen,key,k_steps,algoflag); //14 ->Fleas_lc 15 ->fleas ld
        pt2=new kdev_thr(&r2,blolen, ic2,0,NULL,keylen,key,k_steps,algoflag);
        if ( pt2->Create() != wxTHREAD_NO_ERROR )
            {
                throwout(_("Can't create the thread!"),3);
                delete pt2;
                pt2=NULL;
            }
        else
          {
            if ( pt2->Run() != wxTHREAD_NO_ERROR )
            {
                throwout(_("Can't Run the thread!"),3);
                delete pt2;
                pt2 = NULL;
            }
          }
          if(pt2==NULL)  //error  go to harakiri
          {
               pt0->Wait();
               pt1->Wait();
               //close files
                fclose(cipherfile);
                fclose(plainfile);
                //free buffers
                free(iv); free(ip); free(ic);free(ic1);free(ic2);free(ic3);
                throwout(_("error in encipher file cntmode multithread t1"),3);
                return false;
          }

      ivno.inc();
      memcpy(ic3,(char *)(ivno.ad),blolen);
      //k_develop_f4(blolen, ic3,0,NULL,keylen,key,k_steps,algoflag); //14 ->Fleas_lc 15 ->fleas ld
        pt3=new kdev_thr(&r3,blolen, ic3,0,NULL,keylen,key,k_steps,algoflag);
        if ( pt3->Create() != wxTHREAD_NO_ERROR )
            {
                throwout(_("Can't create the thread!"),3);
                delete pt3;
                pt3=NULL;
            }
        else
          {
            if ( pt3->Run() != wxTHREAD_NO_ERROR )
            {
                throwout(_("Can't Run the thread!"),3);
                delete pt3;
                pt3 = NULL;
            }
          }
          if(pt3==NULL)  //error  go to harakiri
          {
               pt0->Wait();
               pt1->Wait();
               pt2->Wait();
               //close files
                fclose(cipherfile);
                fclose(plainfile);
                //free buffers
                free(iv); free(ip); free(ic);free(ic1);free(ic2);free(ic3);
                throwout(_("error in encipher file cntmode multithread t1"),3);
                return false;
          }
      //xor with ciphertext block
      //write to plain file
      //increment ivno
      ivno.inc();
//wait for proc0 to finish
       pt0->Wait();
      //read in ciphertext block
      for(i=0;i<blolen;i++)
      {
          *(ip + i)= (char) fgetc(cipherfile);
      }
      //xor with ciphertext block
      //write to plain file
      for(i=0;i<blolen;i++)
      {

          hc=*(ip + i);
          hc ^= *(ic+i);
          fputc( hc,plainfile) ;
      }

//wait for proc1 to finish
       pt1->Wait();
      //read in ciphertext block
      for(i=0;i<blolen;i++)
      {
          *(ip + i)= (char) fgetc(cipherfile);
      }
      //xor with ciphertext block
      //write to plain file
      for(i=0;i<blolen;i++)
      {

          hc=*(ip + i);
          hc ^= *(ic1+i);
          fputc( hc,plainfile) ;
      }

//wait for proc1 to finish
       pt2->Wait();
      //read in ciphertext block
      for(i=0;i<blolen;i++)
      {
          *(ip + i)= (char) fgetc(cipherfile);
      }
      //xor with ciphertext block
      //write to plain file
      for(i=0;i<blolen;i++)
      {

          hc=*(ip + i);
          hc ^= *(ic2+i);
          fputc( hc,plainfile) ;
      }

//wait for proc1 to finish
       pt3->Wait();
      //read in ciphertext block
      for(i=0;i<blolen;i++)
      {
          *(ip + i)= (char) fgetc(cipherfile);
      }
      //xor with ciphertext block
      //write to plain file
      for(i=0;i<blolen;i++)
      {

          hc=*(ip + i);
          hc ^= *(ic3+i);
          fputc( hc,plainfile) ;
      }

  }
  //get rest block from plainfile
  //get key-developed number block from ivno
  //xor bytewise and write to cipherfile
  //do it with the rest
  for(;j<blockzahl;j++)
  {
        memcpy(ic1,(char *)(ivno.ad),blolen);
        k_develop_f4(blolen, ic1,0,NULL,keylen,key,k_steps,algoflag); //14 ->Fleas_lc 15->Fleas_ld
        for(i=0;i<blolen;i++)
        {
            ichar=fgetc(cipherfile);
            if(ichar==EOF) break;
            hc=(char) ichar;
            hc ^= *(ic1 + i);
            fputc( hc,plainfile) ;
        }
        ivno.inc();
  }
  memcpy(ic1,(char *)(ivno.ad),blolen);
  k_develop_f4(blolen, ic1,0,NULL,keylen,key,k_steps,algoflag); //14 ->Fleas_lc 15->Fleas_ld
  i=0;
  do
  {
      ichar=fgetc(cipherfile);
      if(ichar==EOF) break;
      hc=(char) ichar;
      hc ^= *(ic1 + i);
      fputc( hc,plainfile) ;
      i++;
  } while(ichar != EOF);
  //close files
  fclose(cipherfile);
  fclose(plainfile);
    //free buffers
    free(iv); free(ip); free(ic); free(ic1); free(ic2); free(ic3);
  return true;
}
/*****************************************************/
bool decipher_file_cntmode_4ecc(ulong32 deadhead, wxString cnam, wxString* pnampt, char* key, int keylen, int k_steps, int blolen, int mode, wxString algo, unsigned char shoprog)
{
  long long filen=0, blockzahl, j;
  FILE *plainfile, *cipherfile;
  wxString temppname,temphname,secsep=_("X:"), pathcomp,hs;
  char *iv, *ic,*ip,*ffic, *flightxkey,*flightxblock,hc,buff[1602],c,*thrfkey=NULL;
  int i,ichar, currpos,lastmax=0,fl1,buflen,lbl,i2;
  unsigned int algoflag;
  char *chkbuf,*outbuf;
  longnumber ivno, tfivno;
  bool goodp=true,switched=false;

      // Extra for Threefish cipher context data
  ThreefishKey_t keyCtx;
  //some primitive error checking
  if(blolen<1) return false;
  if(!wxFile::Exists(cnam))
  {
      throwout(_("Chiffratdatei existiert nicht!!\nAbbruch."),2);
      return false;
  }
  //determine Length
  wxFile tmpfl(cnam);
  filen=tmpfl.Length();
  tmpfl.Close();
  // jetzt plainfile öffnen ohne anhängsel "_1c" etc und setze algoflag
  wxFFile ciph_fil(cnam,_("rb"));
  if(!ciph_fil.IsOpened()){throwout(_("couldn't open cipherfile\naborting"),5); return false;}
  cipherfile=ciph_fil.fp();
  temphname=cnam+_("_hl");
  temppname=cnam+_("_hp");

  if(algo ==_("F_cnt_1c"))   { algoflag=9;}
  else if(algo ==_("F_cnt_1d"))   { algoflag=10;}
  else if(algo ==_("F_cnt_1b"))   { algoflag=11;}
  else if(algo ==_("threefish"))   { algoflag=17;}
  else if(algo ==_("flight"))   { algoflag=15;}
  else if(algo ==_("flightx"))   { algoflag=16;}
  else if(algo ==_("chimera"))   { algoflag=20;}
       else{throwout(_("unknown algorithm! \n aborting (2)"));return false;}

  wxFFile plain_fil(temphname,_("wb"));
  if(!plain_fil.IsOpened()) {throwout(_("couldn't open plainheader for writing\naborting"),5); return false;}
  plainfile=plain_fil.fp();

//initialize Progress bar
  wxProgressDialog pbar(_("Entschluesselungs-Fortschritt"),_("Computer arbeitet hart :-)"),100);
// set checksequence for header end
  chkbuf= (char*) malloc(4*secsep.Len()+2);
  buflen=secsep.Len();
  outbuf= (char*) malloc(buflen+2);
  //sequence set
  strcpy( chkbuf, (const char*)secsep.mb_str(wxConvUTF8) );// schreibt den Filenamen in fnambuff rein
  for(i=0;i<buflen;i++){outbuf[i]=0;} //set starter value
  //determine blockzahl
  blockzahl=(filen-deadhead)/blolen;
  if(blockzahl >0)blockzahl--;
  if(algoflag == 20) //allocate chimera stuff and set subkeys
  {
        if (blolen!= 512) { throwout(_("blocklength error  in chimera call"), 10); return false;}
        thrfkey=develop_o_malloc(keylen,key,2,2,144);
        threefishSetKey(&keyCtx, Threefish1024, (uint64_t *)thrfkey, (uint64_t *)(thrfkey+128));  //use first 128 byte of key as key, next 16 as tweak
        flightxkey=develop_o_malloc(keylen,key,3,2,512);
        if(flightxkey==NULL) {free(thrfkey); throwout(_(" failed malloc in enc fleafish(3)"));return false;}
        flightxblock= (char *) malloc(blolen+512);
        if(flightxblock==NULL)
        { free(thrfkey);free(flightxkey); return false;}
        ffic=(char *) malloc(blolen);
        if(ffic==NULL)
        { free(flightxblock); free(thrfkey);free(flightxkey); return false;}
  }
  iv=(char *) malloc(blolen);
  ic=(char *) malloc(blolen);
  ip=(char *) malloc(blolen);
  //recreate IV from locked vlock
  //read locked IV
  fl1=fseek(cipherfile,deadhead, SEEK_SET); //dead header überspringen
  for(i=0;i<blolen;i++)
  {
      ichar=fgetc(cipherfile) ;
      if(ichar==EOF)
      {
          free(iv);free(ic);free(ip);free(chkbuf),free(outbuf);
          throwout(_("Fehler, konnte IV nicht lesen."),5);
          // remove helper file
          fclose(cipherfile);
          if(wxFileExists(temphname)) wxRemoveFile(temphname);
          return false;
      }
      *(ic+i)=(char) ichar;
  }
  if(algo ==_("threefish"))   //set key and do special preparationes for threefish
  {
      if (blolen!= 128) { throwout(_("blocklength error  in threefish call"), 10); free(iv); free(ip); free(ic);free(chkbuf); return false;}
      thrfkey=develop_o_malloc(keylen,key,2,2,144);
      threefishSetKey(&keyCtx, Threefish1024, (uint64_t *)thrfkey, (uint64_t *)(thrfkey+128));  //use first 128 byte of key as key, next 16 as tweak
      if(thrfkey != NULL) free(thrfkey); //free the threefish key stuff
  }
  wxString primalgo;
  if(algoflag==9) primalgo=_("Fleas_lc"); //obsolete
  else if(algoflag==10) primalgo=_("Fleas_ld"); //obsolete
  else if(algoflag==11) primalgo=_("Fleas_lb"); //obsolete
  else if(algoflag==17) primalgo=_("Fleas_lc"); //obsolete
  else if(algoflag==20) primalgo=_("chimera"); //obsolete!
  if(mode>0) //unlock IV
  {
      if(algoflag==17)
      {
          threefishDecryptBlockBytes(&keyCtx, (uint8_t *)ic,  (uint8_t *)iv);
          //dofeistel(blolen, keylen, ic,iv, key, -4, mode, 0, NULL, primalgo); //special for threefish
      }
      else if(algoflag==20)   //unlock iv
      {
         dofeistel(blolen, 512, ic,iv, flightxkey, -4, mode, 0, NULL, _("flightx"));//4Steps, mode-fold owf, no cpa blocker
      }
      else dofeistel_n(blolen, keylen, ic,iv, key, -4, mode, 0, NULL, algoflag);//-4Steps, mode-fold owf, no cpa blocker
  }
  else memcpy(iv,ic,blolen);


  //put IV into longnumber
  ivno.resizelonu(blolen+2);
  memcpy(ivno.ad, iv, blolen);
  ivno.setsize(true);
  if(algoflag == 20)
  {
      tfivno.resizelonu(128+2); // use this as threefish counter
      memcpy(tfivno.ad, iv, 128);
      tfivno.setsize(true);
      ivno.inc();  //the inc's are still here for compatibility reasons, could also be removed everywhere.
      tfivno.inc();
  }
  if(algoflag==17) ivno.inc(); //Threefish: do not reuse first cipher block for xoring
  //loop complete blocks with progbar update
  for(j=0;j<blockzahl;j++)
  {
       if((shoprog>0)&&(blockzahl>1))
       {
         //prog bar propagation in main loop
         currpos=(int)( (100*j)/(blockzahl-1));
         if((currpos >lastmax)&&goodp)
         {
            goodp=pbar.Update(currpos);
            lastmax=currpos;
         }
       }
      //read in ciphertext block
      for(i=0;i<blolen;i++)
      {
          *(ic + i)= (char) fgetc(cipherfile);
      }
      //get key-developed number block from ivno
      if(algoflag != 20) memcpy(ip,(char *)(ivno.ad),blolen); //obsolete for chimera
      if(algoflag==20)
      {
        //get key-developed number block from ivno
        threefishEncryptBlockBytes(&keyCtx, (uint8_t *)(tfivno.ad), (uint8_t *) ip);
        tfivno.inc();
        threefishEncryptBlockBytes(&keyCtx, (uint8_t *)(tfivno.ad), (uint8_t *) (ip+128));
        tfivno.inc();
        threefishEncryptBlockBytes(&keyCtx, (uint8_t *)(tfivno.ad), (uint8_t *) (ip+256));
        tfivno.inc();
        threefishEncryptBlockBytes(&keyCtx, (uint8_t *)(tfivno.ad), (uint8_t *) (ip+384));
        tfivno.inc();

        //now develop flightx-half
        //fflivno.ad nach ffic mit flightxkey
        memcpy(flightxblock,flightxkey,512); //copy
        memcpy(flightxblock+512,ivno.ad,512); //copy
        develop_flightx_onearm( flightxblock, blolen+512, 1);
        //pick center section
        memcpy(ffic,flightxblock+(blolen+512)/2,512);
        //unite via xor with threefish result blocks in ic
        for(i=0; i<blolen;i++)
        {
            *(ip+i) ^=  *(ffic+i);
        }
      }
      else if(algoflag==17)
      {
          threefishEncryptBlockBytes(&keyCtx, (uint8_t *)ip, (uint8_t *) ip);
      }
      else k_develop_f4(blolen, ip,0,NULL,keylen,key,k_steps,algoflag); //9 ->Fleas_lc 10->Fleas_ld
      //xor with plaintext block
      //write to plain file
      if(!switched)
      {
        for(i=0;i<blolen;i++)
        {
          hc=*(ip + i);
          hc ^= *(ic+i);
          fputc( hc,plainfile) ;
//walk  marker
          memmove(outbuf,outbuf+1,buflen -1);
          *(outbuf+buflen-1)=hc;
          if(!switched && memcmp(outbuf,chkbuf,buflen)==0)  //key hit, now switch aoutput file
          {
              switched= true;
              //close old plainfile and open real one
              if(!plain_fil.Close())
              {
                 throwout(_("Warning, could not close temp file"));
              }
              if(!plain_fil.Open(temppname,_("wb")) )
              {
                  throwout(_("Error, could not open plain file!\naborting."),5);
                  free(iv); free(ip); free(ic);free(chkbuf);free(outbuf);
                  ciph_fil.Close();
                  return false;
              }
              if(!plain_fil.IsOpened())
                {
                  free(iv); free(ip); free(ic);free(chkbuf);free(outbuf);
                  ciph_fil.Close();
                    throwout(_("couldn't open plainheader for writing\naborting"),5);
                    return false;
                }
              plainfile=plain_fil.fp();
          }
        }
      }
      else
      {
          for(i=0;i<blolen;i++)
          {
            hc=*(ip + i);
            hc ^= *(ic+i);
            fputc( hc,plainfile) ;
          }
      }
      //increment ivno
      ivno.inc();
  }
  //get rest block from plainfile
  //get key-developed number block from ivno
  //xor bytewise and write to cipherfile
  memcpy(ip,(char *)(ivno.ad),blolen);
  if(algoflag==17)
  {
     threefishEncryptBlockBytes(&keyCtx, (uint8_t *)ip, (uint8_t *) ip);
  }
  else if(algoflag == 20) //for chimera
  {
        //get key-developed number block from ivno
        threefishEncryptBlockBytes(&keyCtx, (uint8_t *)(tfivno.ad), (uint8_t *) ip);
        tfivno.inc();
        threefishEncryptBlockBytes(&keyCtx, (uint8_t *)(tfivno.ad), (uint8_t *) (ip+128));
        tfivno.inc();
        threefishEncryptBlockBytes(&keyCtx, (uint8_t *)(tfivno.ad), (uint8_t *) (ip+256));
        tfivno.inc();
        threefishEncryptBlockBytes(&keyCtx, (uint8_t *)(tfivno.ad), (uint8_t *) (ip+384));
        tfivno.inc();
        //now develop flightx-half
        //fflivno.ad nach ffic mit flightxkey
        memcpy(flightxblock,flightxkey,512); //copy
        memcpy(flightxblock+512,ivno.ad,512); //copy
        develop_flightx_onearm( flightxblock, blolen+512, 1);
        //pick center section
        memcpy(ffic,flightxblock+(blolen+512)/2,512);
        //unite via xor with threefish result blocks in ic
        for(i=0; i<blolen;i++)
        {
            *(ip+i) ^=  *(ffic+i);
        }
  }
  else k_develop_f4(blolen, ip,0,NULL,keylen,key,k_steps,algoflag); //9 ->Fleas_lc 10->Fleas_ld
  i=0;
  if(switched)
  {
    do
    {
      ichar=fgetc(cipherfile);
      if(ichar==EOF) break;
      hc=(char) ichar;
      hc ^= *(ip + i);
      fputc( hc,plainfile) ;
      i++;
    } while(ichar != EOF);
  }
  else
  {
    do
    {
      ichar=fgetc(cipherfile);
      if(ichar==EOF) break;
      hc=(char) ichar;
      hc ^= *(ip + i);
      fputc( hc,plainfile) ;
  //walk  marker
          memmove(outbuf,outbuf+1,buflen -1);
          *(outbuf+buflen-1)=hc;
          if(!switched && memcmp(outbuf,chkbuf,buflen)==0)  //key hit, now switch aoutput file
          {
              switched= true;
              //close old plainfile and open real one
              if(!plain_fil.Close())
              {
                 throwout(_("Warning, could not close temp file"));
              }
              if(!plain_fil.Open(temppname,_("wb")) )
              {
                  throwout(_("Error, could not open plain file!\naborting."),5);
                  free(iv); free(ip); free(ic);free(chkbuf);free(outbuf);
                  ciph_fil.Close();
                  return false;
              }
              if(!plain_fil.IsOpened())
              {
                  throwout(_("couldn't open plainheader for writing\naborting"),5);
                  free(iv); free(ip); free(ic);free(chkbuf);free(outbuf);
                  ciph_fil.Close();
                  return false;
              }
              plainfile=plain_fil.fp();
          }
      i++;
    } while(ichar != EOF);
  }
    //free buffers
  free(iv); free(ip); free(ic);free(chkbuf);free(outbuf);
  if(algoflag==20) //free the chimera stuff
  {
      free(ffic); free(flightxkey); free(flightxblock);
  }
  //now do some deletion and renaming
  //open helper file and read  in name
  ciph_fil.Close();
  if(!plain_fil.Close())
              {
                 throwout(_("Warning, could not close temp file"));
              }
  if(!plain_fil.Open(temphname,_("rb")) )
              {
                  throwout(_("Error, could not open plain file!\naborting."),5);
              }
  plainfile=plain_fil.fp();
  for(i=0;i<30;i++)
         {
              lbl=fscanf(plainfile,"%c",&c);
         }
  lbl=fscanf(plainfile," fnam: %400s ",buff);
  if(lbl!=1)
  {
      throwout(_("Fehler, konnte Klardateinamen nicht lesen\nAbbruch ohne Namensaenderung!"),5);
      if(!plain_fil.Close())
              {
                 throwout(_("Warning, could not close temp header file"),3);
              }
          // remove helper file
      if(wxFileExists(temphname)) wxRemoveFile(temphname);
      return false;
  }
  else
  {
      if((pnampt!=NULL)&&(*pnampt!= _("")))
      {
         if(!wxRenameFile(temppname,*pnampt))
         {
             //rename plainfile
             throwout(_("error, could not rename plain file"),5);
         }
         if(!plain_fil.Close())
              {
                 throwout(_("Warning, could not close temp header file"),3);
              }
        if(!wxRemoveFile(temphname))
         {
             //remove plain header
             throwout(_("error, could not remove plain header"),5);
         }
      }
      else
      {
         hs=wxString::FromUTF8(buff);
         //prepend path and sep
         pathcomp=((wxFileName) temppname).GetPath(wxPATH_GET_VOLUME|wxPATH_GET_SEPARATOR);
//dirty patch: changing >< to underscore in hs
  i2=0;
  i2 += hs.Replace(_("<"), _("_"));
  i2 += hs.Replace(_(">"), _("_"));
  if(i2>0) throwout(_("Unsichere Zeichen im Dateinamen \n >,< ersetzt durch _"),5);
//end dirty patch
         *pnampt= pathcomp+ hs;

         if(wxFileExists(*pnampt))
         {
           throwout(_("Achtung, Datei mit Originalnamen existiert bereits\nwird umbenannt in *_old und fahre fort"));
           if(wxFileExists(*pnampt+_("_old"))) wxRemoveFile(*pnampt+_("_old"));
           wxRenameFile(*pnampt, *pnampt+_("_old"));
         }
         if(!wxRenameFile(temppname,*pnampt))
         {
             //rename plainfile
             throwout(_("error, could not rename plain file"),5);
         }

         if(!plain_fil.Close())
              {
                 throwout(_("Warning, could not close temp header file"),3);
              }
         if(!wxRemoveFile(temphname))
         {
             //remove plain header
             throwout(_("error, could not remove plain header"),5);
         }
      }
  }
  return true;
}
/*****************************************************/
bool decipher_file_cntmode(ulong32 deadhead, char* fnam, char* key, int keylen, int k_steps, int blolen, int mode, wxString algo, unsigned char shoprog)
//k_steps: number of owf-steps,   mode:0-clear number, 1number of lonulock steps for number
//
{
  long long filen=0, blockzahl, j;
  FILE *plainfile, *cipherfile;
  char *sfnam, *iv, *ic,*ip,hc,*thrfkey;
  int i,ichar, currpos,namlen,lastmax=0,fl1;
  unsigned int algoflag;
  longnumber ivno;
  bool goodp=true;

      // Extra for Threefish cipher context data
  ThreefishKey_t keyCtx;
  //some primitive error checking
  namlen=strlen(fnam);
  if(namlen > 398) return false;
  if(blolen<1) return false;
  wxString fnwx(fnam, wxConvFile);
  if(!wxFile::Exists(fnwx))
  {
      throwout(_("Chiffratdatei existiert nicht!!\nAbbruch."),2);
      return false;
  }
  //determine Length
  wxFile tmpfl(fnwx);
  filen=tmpfl.Length();
  tmpfl.Close();
  if((cipherfile = fopen(fnam, "rb"))== NULL) return false;
  // jetzt plainfile öffnen ohne anhängsel "_1c" etc und setze algoflag
    //create new sfnam "shadow fnam" buffer
    sfnam=(char *)malloc(strlen(fnam)+2);
    strcpy(sfnam,fnam);
    if(fnwx.Right(3)== _("_1c")) memcpy(sfnam+namlen-3,"\0",1);
    else if(fnwx.Right(3)== _("_1d")) memcpy(sfnam+namlen-3,"\0",1);
    else if(fnwx.Right(3)== _("_1b")) memcpy(sfnam+namlen-3,"\0",1);
    else if(fnwx.Right(3)== _("_tf")) memcpy(sfnam+namlen-3,"\0",1);
    else if(fnwx.Right(3)== _("_fl")) memcpy(sfnam+namlen-3,"\0",1);
    else if(fnwx.Right(3)== _("_fx")) memcpy(sfnam+namlen-3,"\0",1);
    else if(fnwx.Right(3)== _("_ff")) memcpy(sfnam+namlen-3,"\0",1);
    else if( (fnwx.Right(5)== _(".ciph"))||(fnwx.Right(5)== _("_ciph"))  ) memcpy(sfnam+namlen-5,"\0",1);
    else{
           throwout(_("Warnung!\nKeine vernuenftige Namenserweiterung gefunden\nschneide letzte drei chars ab."),3);
            memcpy(sfnam+namlen-3,"\0",1);
        }

  if(algo ==_("F_cnt_1c"))   { algoflag=9;}
  else if(algo ==_("F_cnt_1d"))   { algoflag=10;}
  else if(algo ==_("F_cnt_1b"))   { algoflag=11;}
  else if(algo ==_("threefish"))   { algoflag=17;}
  else if(algo ==_("flight"))   { algoflag=15;}
  else if(algo ==_("flightx"))   { algoflag=16;}
  else if(algo ==_("chimera"))   { algoflag=20;}
    /*else if(algo ==_("Fleas_3"))   {strncat(sfnam,"_f3",3); algoflag=3;}
     else if(algo ==_("aes"))   {strncat(sfnam,"_ae",3);blolen=8;algoflag=12;}*/
       else{throwout(_("unknown algorithm! \n aborting (2)"));free(sfnam);return false;}
  if((plainfile = fopen(sfnam, "wb"))== NULL)
  {
    free(sfnam);
    return false;
  }
  free(sfnam);

//initialize Progress bar
  wxProgressDialog pbar(_("Entschluesselungs-Fortschritt"),_("Der Prozessor arbeitet...."),100);
  //determine blockzahl
  blockzahl=(filen-deadhead)/blolen;
  if(blockzahl >0)blockzahl--;
  //create IV from random then lock with key
  iv=(char *) malloc(blolen);
  ic=(char *) malloc(blolen);
  ip=(char *) malloc(blolen);
  //read locked IV
  fl1=fseek(cipherfile,deadhead, SEEK_SET); //dead header überspringen
  for(i=0;i<blolen;i++)
  {
      ichar=fgetc(cipherfile) ;
      if(ichar==EOF)
      {
          free(iv);free(ic);free(ip);
          throwout(_("Fehler, konnte IV nicht setzen."),5);
          return false;
      }
      *(ic+i)=(char) ichar;
  }
  if(algo ==_("threefish"))   //set key and do special preparationes for threefish
  {
      if (blolen!= 128) { throwout(_("blocklength error  in threefish call"), 10); free(iv); free(ip); free(ic); return false;}
      thrfkey=develop_o_malloc(keylen,key,2,2,144);
      threefishSetKey(&keyCtx, Threefish1024, (uint64_t *)thrfkey, (uint64_t *)(thrfkey+128));  //use first 128 byte of key as key, next 16 as tweak
      if(thrfkey != NULL) free(thrfkey); //free the threefish key stuff
  }
  wxString primalgo;
  if(algoflag==9) primalgo=_("Fleas_lc"); //obsolete
  else if(algoflag==10) primalgo=_("Fleas_ld"); //obsolete
  else if(algoflag==11) primalgo=_("Fleas_lb"); //obsolete
  else if(algoflag==17) primalgo=_("Fleas_lc"); //obsolete
  else if(algoflag==20) primalgo=_("skein"); //obsolete

  if(mode>0)
  {
      if(algoflag==17)
      {
          threefishDecryptBlockBytes(&keyCtx, (uint8_t *)ic,  (uint8_t *)iv);
          //dofeistel(blolen, keylen, ic,iv, key, -4, mode, 0, NULL, primalgo); //special for threefish
      }
      else dofeistel_n(blolen, keylen, ic,iv, key, -4, mode, 0, NULL, algoflag);//-4Steps, mode-fold owf, no cpa blocker
  }
  else memcpy(iv,ic,blolen);


  //put IV into longnumber
  ivno.resizelonu(blolen+2);
  memcpy(ivno.ad, iv, blolen);
  ivno.setsize(true);
  if(algoflag==17) ivno.inc(); //Threefish: do not reuse first cipher block for xoring
  //loop complete blocks with progbar update
  for(j=0;j<blockzahl;j++)
  {
       if((shoprog>0)&&(blockzahl>1))
       {
         //prog bar propagation in main loop
         currpos=(int)( (100*j)/(blockzahl-1));
         if((currpos >lastmax)&&goodp)
         {
            goodp=pbar.Update(currpos);
            lastmax=currpos;
         }
       }
      //read in ciphertext block
      for(i=0;i<blolen;i++)
      {
          *(ic + i)= (char) fgetc(cipherfile);
      }
      //get key-developed number block from ivno
      memcpy(ip,(char *)(ivno.ad),blolen);
      if(algoflag==17)
      {
          threefishEncryptBlockBytes(&keyCtx, (uint8_t *)ip, (uint8_t *) ip);
      }
      else k_develop_f4(blolen, ip,0,NULL,keylen,key,k_steps,algoflag); //9 ->Fleas_lc 10->Fleas_ld
      //xor with plaintext block
      //write to plain file
      for(i=0;i<blolen;i++)
      {

          hc=*(ip + i);
          hc ^= *(ic+i);
          fputc( hc,plainfile) ;
      }
      //increment ivno
      ivno.inc();
  }
  //get rest block from plainfile
  //get key-developed number block from ivno
  //xor bytewise and write to cipherfile
  memcpy(ip,(char *)(ivno.ad),blolen);
  if(algoflag==17)
  {
     threefishEncryptBlockBytes(&keyCtx, (uint8_t *)ip, (uint8_t *) ip);
  }
  else k_develop_f4(blolen, ip,0,NULL,keylen,key,k_steps,algoflag); //9 ->Fleas_lc 10->Fleas_ld
  i=0;
  do
  {
      ichar=fgetc(cipherfile);
      if(ichar==EOF) break;
      hc=(char) ichar;
      hc ^= *(ip + i);
      fputc( hc,plainfile) ;
      i++;
  } while(ichar != EOF);
  //close files
  fclose(cipherfile);
  fclose(plainfile);
    //free buffers
    free(iv); free(ip); free(ic);
  return true;
}
/*****************************************************/
/*****************************************************/
bool decipher_file_fleafish(ulong32 deadhead, char* fnam, char* key, int keylen, int k_steps, int blolen, int mode, wxString algo, unsigned char shoprog)
//k_steps: number of owf-steps,   mode:0-clear number, 1number of lonulock steps for number
//
{
  long long filen=0, blockzahl, j;
  FILE *plainfile, *cipherfile;
  char *sfnam, *iv, *ic, *ffic,*ip,hc, *thrfkey, *flightxkey, *flightxblock;
  int i,ichar, currpos,namlen,lastmax=0;
  longnumber ivno, flivno;
  bool goodp=true, laengenfehler=false;;
  ThreefishKey_t keyCtx;


  //some primitive error checking
  if (algo!= _("chimera")) { throwout(_("error 1 in fleafish call"), 10); return false;}
  if (blolen!= 512) { throwout(_("blocklength error  in fleafish call"), 10); return false;}

  namlen=strlen(fnam);
  if(namlen > 401) return false;
  wxString fnwx(fnam, wxConvFile);
  if(!wxFile::Exists(fnwx))
  {
      throwout(_("Cipherfile does not exist!!\nAborting."),2);
      return false;
  }
  //determine Length
  wxFile tmpfl(fnwx);
  filen=tmpfl.Length();
  tmpfl.Close();
  if((cipherfile = fopen(fnam, "rb"))== NULL) return false;
  // jetzt plainfile öffnen ohne anhängsel "_1c" etc und setze algoflag
    //create new sfnam "shadow fnam" buffer
    sfnam=(char *)malloc(strlen(fnam)+2);
    strcpy(sfnam,fnam);
    if(fnwx.Right(3)== _("_ff")) memcpy(sfnam+namlen-3,"\0",1);
    else if( (fnwx.Right(5)== _(".ciph"))||(fnwx.Right(5)== _("_ciph"))  ) memcpy(sfnam+namlen-5,"\0",1);
       else{throwout(_("unknown filename marker! \n aborting (2)"));free(sfnam);return false;}

  if((plainfile = fopen(sfnam, "wb"))== NULL)
  {
    free(sfnam);
    return false;
  }
  free(sfnam);
// prepare key structure
  // reform key to 128 byte
  thrfkey=develop_o_malloc(keylen,key,2,2,144); //use independent key for threefish
  if(thrfkey==NULL) {throwout(_(" failed malloc in dec fleafish(1)"));return false;}
  threefishSetKey(&keyCtx, Threefish1024, (uint64_t *)thrfkey, (uint64_t *)(thrfkey+128));  //use first 128 byte of key as key, next 16 as tweak
  flightxkey=develop_o_malloc(keylen,key,3,2,512); //decouple flightxkey from threefkey precursor
  if(flightxkey==NULL) {free(thrfkey); throwout(_(" failed malloc in enc fleafish(2)"));return false;}
//initialize Progress bar
  wxProgressDialog pbar(_("Entschluesselungs-Fortschritt"),_("Computer arbeitet!"),100);
  //determine blockzahl
  blockzahl=(filen-deadhead)/blolen;
  if(blockzahl >0)blockzahl--;
  //create IV from random then unlock with key
  iv=(char *) malloc(blolen);
    if(iv==NULL)
      { free(thrfkey);free(flightxkey); return false;}
  ic=(char *) malloc(blolen);
      if(ic==NULL)
      { free(iv); free(thrfkey);free(flightxkey); return false;}
  ffic=(char *) malloc(blolen);
      if(ffic==NULL)
      { free(ic); free(iv); free(thrfkey);free(flightxkey); return false;}
  ip=(char *) malloc(blolen);
      if(ip==NULL)
      { free(ffic); free(ic); free(iv); free(thrfkey);free(flightxkey); return false;}
  flightxblock= (char *) malloc(blolen+512);
      if(flightxblock==NULL)
      { free(ip); free(ffic); free(ic); free(iv); free(thrfkey);free(flightxkey); return false;}


  //read locked IV
  fseek(cipherfile,deadhead, SEEK_SET); //dead header überspringen
  for(i=0;i<blolen;i++) //holt IV
  {
      ichar=fgetc(cipherfile) ;
      if(ichar==EOF)
      {
          free(iv);free(ic);free(ip);free(ffic);free(flightxblock);free(thrfkey);
          throwout(_("Fehler, konnte IV nicht lesen"),5);
          return false;
      }
      *(ic+i)=(char) ichar;
  }
  if(mode>0) //decrypt IV
  {
      dofeistel(blolen, 512, ic,iv, flightxkey, -4, mode, 0, NULL, _("flightx"));//4Steps, mode-fold owf, no cpa blocker
  }
  else //put as is into lonu and throw a warning
  {
      memcpy(iv,ic,blolen);
      throwout(_("Warnung: Chimera entschlue.-Aufruf ohne IV entschluesslg ist nicht gut!"),3);
  }

  ivno.resizelonu(128+2);
  flivno.resizelonu(blolen+2);
  memcpy(ivno.ad, iv, 128);
  memcpy(flivno.ad, iv, blolen);

  ivno.setsize(true);
  ivno.inc();
  flivno.setsize(true);
  flivno.inc();

  //loop complete blocks with progbar update
  for(j=0;j<blockzahl;j++)
  {
       if((shoprog>0)&&(blockzahl>1))
       {
         //prog bar propagation in main loop
         currpos=(int)( (100*j)/(blockzahl-1));
         if((currpos >lastmax)&&goodp)
         {
            goodp=pbar.Update(currpos);
            lastmax=currpos;
         }
       }
      //read in ciphertext block
      for(i=0;i<blolen;i++)
      {
          *(ic + i)= (char) fgetc(cipherfile);
      }
      threefishEncryptBlockBytes(&keyCtx, (uint8_t *)(ivno.ad), (uint8_t *) ip);
      ivno.inc();
      threefishEncryptBlockBytes(&keyCtx, (uint8_t *)(ivno.ad), (uint8_t *) (ip+128));
      ivno.inc();
      threefishEncryptBlockBytes(&keyCtx, (uint8_t *)(ivno.ad), (uint8_t *) (ip+256));
      ivno.inc();
      threefishEncryptBlockBytes(&keyCtx, (uint8_t *)(ivno.ad), (uint8_t *) (ip+384));
      ivno.inc();
      //now develop flightx-half
      //fflivno.ad nach ffic mit flightxkey
      memcpy(flightxblock,flightxkey,512); //copy
      memcpy(flightxblock+512,flivno.ad,512); //copy
      develop_flightx_onearm( flightxblock, blolen+512, 1);
      flivno.inc();
      //pick center section
      memcpy(ffic,flightxblock+(blolen+512)/2,512);
      //xor with plaintext block
      //write to plain file
      for(i=0;i<blolen;i++)
      {

          hc=*(ip + i);  //recovered xor pad
          hc ^= *(ic+i);  //cipher byte
          hc ^= *(ffic+i);  //recovered xor from fleas arm
          fputc( hc,plainfile) ;
      }
  }
  //get rest block from plainfile
  //get key-developed number block from ivno
  //xor bytewise and write to cipherfile
  threefishEncryptBlockBytes(&keyCtx, (uint8_t *) ivno.ad, (uint8_t *) ip);
      ivno.inc();
  threefishEncryptBlockBytes(&keyCtx, (uint8_t *)(ivno.ad), (uint8_t *) (ip+128));
      ivno.inc();
  threefishEncryptBlockBytes(&keyCtx, (uint8_t *)(ivno.ad), (uint8_t *) (ip+256));
      ivno.inc();
  threefishEncryptBlockBytes(&keyCtx, (uint8_t *)(ivno.ad), (uint8_t *) (ip+384));
      ivno.inc();
      memcpy(flightxblock,flightxkey,512); //copy
      memcpy(flightxblock+512,flivno.ad,512); //copy
      develop_flightx_onearm( flightxblock, blolen+512, 1);
      flivno.inc();
      //pick center section
      memcpy(ffic,flightxblock+(blolen+512)/2,512);


  i=0;
  do
  {
      ichar=fgetc(cipherfile);
      if(ichar==EOF) break;
      hc=(char) ichar;
      hc ^= *(ip + i); //threefish pad
      hc ^= *(ffic+i);  //flea-arm pad
      fputc( hc,plainfile) ;
      i++;
      if(i>blolen)
      {
          if(!laengenfehler)
          {
			  throwout(_("ups, dieser Zustand in decipher_fleafish soll nicht sein.\nPasswd falsch oder Integritaetsverletzung?"),4);
			  laengenfehler=true;
		  }
          ivno.inc();
          threefishEncryptBlockBytes(&keyCtx, (uint8_t *)(ivno.ad), (uint8_t *) ip);
          i=0;
      }
  } while(ichar != EOF);
  //close files
  fclose(cipherfile);
  fclose(plainfile);
    //free buffers
    free(iv); free(ip); free(ic), free(thrfkey); free(ffic); free(flightxkey); free(flightxblock);
  return true;
}
/*****************************************************/

/*****************************************************/
bool decipher_file_threefish(ulong32 deadhead, char* fnam, char* key, int keylen, int k_steps, int blolen, int mode, wxString algo, unsigned char shoprog)
//k_steps: number of owf-steps,   mode:0-clear number, 1number of lonulock steps for number
//
{
  long long filen=0, blockzahl, j;
  FILE *plainfile, *cipherfile;
  char *sfnam, *iv, *ic,*ip,hc, *thrfkey;
  int i,ichar, currpos,namlen,lastmax=0;
  longnumber ivno;
  bool goodp=true, laengenfehler=false;;
  ThreefishKey_t keyCtx;


  //some primitive error checking
  if (algo!= _("threefish")) { throwout(_("error 1 in threefish call"), 10); return false;}
  if (blolen!= 128) { throwout(_("blocklength error  in threefish call"), 10); return false;}

  namlen=strlen(fnam);
  if(namlen > 398) return false;
  wxString fnwx(fnam, wxConvFile);
  if(!wxFile::Exists(fnwx))
  {
      throwout(_("Chiffratdatei existiert nicht!!\nAbbruch."),2);
      return false;
  }
  //determine Length
  wxFile tmpfl(fnwx);
  filen=tmpfl.Length();
  tmpfl.Close();
  if((cipherfile = fopen(fnam, "rb"))== NULL) return false;
  // jetzt plainfile öffnen ohne anhängsel "_1c" etc und setze algoflag
    //create new sfnam "shadow fnam" buffer
    sfnam=(char *)malloc(strlen(fnam)+2);
    strcpy(sfnam,fnam);
    if(fnwx.Right(3)== _("_tf")) memcpy(sfnam+namlen-3,"\0",1);
    else if( (fnwx.Right(5)== _(".ciph"))||(fnwx.Right(5)== _("_ciph"))  ) memcpy(sfnam+namlen-5,"\0",1);
       else{throwout(_("unknown filename marker! \n aborting (2)"));free(sfnam);return false;}

  if((plainfile = fopen(sfnam, "wb"))== NULL)
  {
    free(sfnam);
    return false;
  }
  free(sfnam);
// prepare key structure
  // reform key to 128 byte
  thrfkey=develop_o_malloc(keylen,key,2,2,144);

  threefishSetKey(&keyCtx, Threefish1024, (uint64_t *)thrfkey, (uint64_t *)(thrfkey+128));  //use first 128 byte of key as key, next 16 as tweak

//initialize Progress bar
  wxProgressDialog pbar(_("Entschluesselungs-Fortschritt"),_("Computer arbeitet :-)"),100);
  //determine blockzahl
  blockzahl=(filen-deadhead)/blolen;
  if(blockzahl >0)blockzahl--;
  //create IV from random then lock with key
  iv=(char *) malloc(blolen);
  ic=(char *) malloc(blolen);
  ip=(char *) malloc(blolen);
  //read locked IV
  fseek(cipherfile,deadhead, SEEK_SET); //dead header überspringen
  for(i=0;i<blolen;i++) //holt IV
  {
      ichar=fgetc(cipherfile) ;
      if(ichar==EOF)
      {
          free(iv);free(ic);free(ip);
          throwout(_("Fehler, konnte IV nicht lesen"),5);
          free(thrfkey);
          return false;
      }
      *(ic+i)=(char) ichar;
  }
  //wxString primalgo;
  //primalgo=_("Fleas_lc");
  if(mode>0)
  {
      //dofeistel(blolen, 128, ic,iv, thrfkey, -4, mode, 0, NULL, primalgo);//-4Steps, mode-fold owf, no cpa blocker
      threefishDecryptBlockBytes(&keyCtx, (uint8_t *)ic, (uint8_t *) iv);
  }
  else memcpy(iv,ic,blolen);  //recovered IV now
  //put IV into longnumber
  ivno.resizelonu(blolen+2);
  memcpy(ivno.ad, iv, blolen);
  ivno.setsize(true);
  ivno.inc();
  //loop complete blocks with progbar update
  for(j=0;j<blockzahl;j++)
  {
       if((shoprog>0)&&(blockzahl>1))
       {
         //prog bar propagation in main loop
         currpos=(int)( (100*j)/(blockzahl-1));
         if((currpos >lastmax)&&goodp)
         {
            goodp=pbar.Update(currpos);
            lastmax=currpos;
         }
       }
      //read in ciphertext block
      for(i=0;i<blolen;i++)
      {
          *(ic + i)= (char) fgetc(cipherfile);
      }
      threefishEncryptBlockBytes(&keyCtx, (uint8_t *)(ivno.ad), (uint8_t *) ip);
      //xor with plaintext block
      //write to plain file
      for(i=0;i<blolen;i++)
      {

          hc=*(ip + i);
          hc ^= *(ic+i);
          fputc( hc,plainfile) ;
      }
      //increment ivno
      ivno.inc();
  }
  //get rest block from plainfile
  //get key-developed number block from ivno
  //xor bytewise and write to cipherfile
  threefishEncryptBlockBytes(&keyCtx, (uint8_t *)(ivno.ad), (uint8_t *) ip);
  i=0;
  do
  {
      ichar=fgetc(cipherfile);
      if(ichar==EOF) break;
      hc=(char) ichar;
      hc ^= *(ip + i);
      fputc( hc,plainfile) ;
      i++;
      if(i>blolen)
      {
          if(!laengenfehler)
          {
			  throwout(_("ups, dieser Zustand in decipher_threefish soll nicht sein.\nPasswd falsch oder Integritaetsverletzung?"),4);
			  laengenfehler=true;
		  }
          ivno.inc();
          threefishEncryptBlockBytes(&keyCtx, (uint8_t *)(ivno.ad), (uint8_t *) ip);
          i=0;
      }
  } while(ichar != EOF);
  //close files
  fclose(cipherfile);
  fclose(plainfile);
    //free buffers
    free(iv); free(ip); free(ic), free(thrfkey);
  return true;
}
/*****************************************************/

bool decipher_file(char* fnam, char* key, int keylen, char feistelsteps, int ksteps, int blolen, int mode, wxString algo, unsigned char shoprog)
// entschlüsselt ein File mit key
// wahre Blocklänge 2*blolen
// alte filelänge wird vorangestellt len: long
{
  long long filen=0, blockzahl, j,newfilen,blinds,bytcount=0;
  FILE *plainfile, *cipherfile;
  char *bff1, *bff_o, *cpa_bl, *bff_last, *key2;
  int restlen,i,namlen, cpa_blen,hlp,algoflag;
  int currpos, lastmax=0;
  bool goodp=true;
  unsigned int algonum;
  aes_context aesc;



  if( (algo==_("F_cnt_1c"))||(algo==_("F_cnt_1d"))||(algo==_("F_cnt_1b")) )
  {
      return( decipher_file_cntmode_mt(0,fnam, key, keylen, ksteps, blolen, 2, algo, shoprog) );
  }
  if( (algo==_("threefish"))||(algo==_("flight"))||(algo==_("flightx")) )
  {
      return( decipher_file_cntmode(0,fnam, key, keylen, ksteps, blolen, 2, algo, shoprog) );
  }

  if((blolen<26) && (algo !=_("aes"))) return false;// kein Platz für die Filelänge!
  namlen=strlen(fnam);
  if((cipherfile = fopen(fnam, "rb"))== NULL) return false;

    algonum= get_code_4_algo(algo);

  bff1 = (char *)malloc(2*blolen); bff_o = (char *)malloc(2*blolen);
  if(mode == 1) bff_last = (char *)malloc(2*blolen);
  //algocode bestimmen
  if(algo ==_("Fleas_1_8"))   { algoflag=0;}
    else if(algo ==_("Fleas_3"))   {algoflag=3;}
     else if(algo ==_("Fleas_4"))   { algoflag=4;}
     else if(algo ==_("Fleas_5"))   { algoflag=5;}
     else if(algo ==_("Fleas_x2"))   { algoflag=6;}
     else if(algo ==_("Fleas_x5"))   { algoflag=7;}
     else if(algo ==_("Fleas_o2"))   { algoflag=8;}
     else if(algo ==_("Fleas_o5"))   { algoflag=9;}
     else if(algo ==_("Fleas_l"))   { algoflag=10;}
     else if(algo ==_("Fleas_ls"))   {  algoflag=11;}
     else if(algo ==_("aes"))       { blolen=8; algoflag=12;}
     else if(algo ==_("Fleas_l3"))   {  algoflag=13;}
     else if(algo ==_("Fleas_lc"))   {  algoflag=14;}
     else if(algo ==_("Fleas_ld"))   {  algoflag=15;}
     else if(algo ==_("Fleas_lb"))   {  algoflag=16;}
       else{
            throwout(_("unknown algorithm! \n aborting (3)"));
            free(bff1); free(bff_last); free(bff_o);return false;}

//initialize Progress bar
  wxProgressDialog pbar(_("Entschluesselungs-Fortschritt"),_("Computer arbeitet hart :-)"),100);


  //calculate and produce cpa_attack blocker
  cpa_blen=blolen; //cpa_blen is matching blolen
  /** blolen must be 8 for aes!!  **/
  if((algo == _("aes"))&&(blolen != 8))
  {
      throwout(_("fatal error in aes blocksize!")); return false;
  }
  // prepare (possibly derived) key2
  key2= (char *) malloc(keylen);
  memcpy(key2,key,keylen);
  switch( algoflag) //different behaviour concerning cpa_blocking
  {
    case 4 :
      // build derived key for use with Fleas_4
      develop_c(key2,keylen,2);
      // build cpa_blocker with different algorithm
      cpa_bl= develop_b_malloc(keylen, key,4, cpa_blen);
      break;

    case 5 :
      // build derived key for use with Fleas_4
      develop_x(key2,keylen,2,3);
      // build cpa_blocker with different algorithm
      cpa_bl= develop_b_malloc(keylen, key,4, cpa_blen);
      break;

    case 6 :
      // build derived key for use with Fleas_x2
      develop_x1(key2,keylen,2,3);
      // build cpa_blocker with different algorithm
      cpa_bl= develop_x_malloc(keylen, key,4,2, cpa_blen);
      break;

    case 7 :
      // build derived key for use with Fleas_x5
      develop_x1(key2,keylen,2,3);
      // build cpa_blocker with different algorithm
      cpa_bl= develop_x_malloc(keylen, key,4,5, cpa_blen);
      break;

    case 8 :
      // build derived key for use with Fleas_o2
      develop_o(key2,keylen,2,3,0);
      // build cpa_blocker with different algorithm
      cpa_bl= develop_o_malloc(keylen, key,3,2, cpa_blen,0);
      break;

    case 9 :
      // build derived key for use with Fleas_o5
      develop_o(key2,keylen,2,3,2);
      // build cpa_blocker with different algorithm
      cpa_bl= develop_o_malloc(keylen, key,3,5, cpa_blen,2);
      break;


    case 10 :
      // build derived key for use with Fleas_l
      develop_l(key2,keylen,2);  //not used later on! only uses cpa_bl
      // build cpa_blocker with different algorithm
      cpa_bl= develop_l_malloc(keylen, key, 3,cpa_blen);
      break;

    case 11 :
      // build derived key for use with Fleas_ls
      develop_l(key2,keylen,2);
      // build cpa_blocker with different algorithm
      cpa_bl= develop_l_malloc(keylen, key, 3,cpa_blen);
      break;

    case 12 :
      // build derived key for use with aes
      develop_o(key2,keylen,2,3,2); //not used
      // build cpa_blocker with different algorithm
      cpa_bl= develop_o_malloc(keylen, key, 3,5,6*blolen,2); //cpablen, erste 16 für cpablock, 2.&3. 16 für key in 2 rounds
      //initiate aes-structure
      break;

    case 13 :
      // build derived key for use with Fleas_l3
      develop_ls(key2,keylen,3,3,1);
      // build cpa_blocker with different algorithm
      cpa_bl= develop_l_malloc(keylen, key, 3,cpa_blen);
      break;

    case 14 :
      // build derived key for use with Fleas_lc
      develop_ln(key2,keylen,3,3,1);
      // build cpa_blocker with different algorithm
      cpa_bl= develop_l_malloc(keylen, key, 3,cpa_blen);
      break;

    case 15 :
      // build derived key for use with Fleas_ld
      develop_ln2_mt(key2,keylen,3,3,1);
      // build cpa_blocker with different algorithm
      cpa_bl= develop_l_malloc(keylen, key, 4,cpa_blen);
      break;

    case 16 :
      // build derived key for use with Fleas_ld
      develop_ln2_mt(key2,keylen,3,3,1);
      // build cpa_blocker with different algorithm
      cpa_bl= develop_l_malloc(keylen, key, 2,cpa_blen);
      break;

    default:
      cpa_bl=bytmixxmalloc(keylen,key,6,cpa_blen);
      break;
  }

  for( i=0;i<2*blolen;i++)
  {
     *(bff1 + i)= (char) fgetc(cipherfile);
  }
  //erster Block geholt
  if((algo !=_("aes"))&&(mode == 1)) memcpy(bff_last,bff1,2*blolen);  //wenn CBC, lastblock fuellen
  if(algo==_("aes"))
  {
      //set encipher structure
        if (aes_setkey_dec( &aesc, (const unsigned char *)(cpa_bl+16), 256 )!= 0)
          throwout(_("error in setting aes key context"));
      //use aes in ecb mode
        if(aes_crypt_ecb( &aesc, AES_DECRYPT,(const unsigned char*)bff1,(unsigned char*)bff_o)!=0) throwout(_("error in aes transformation"));
      //xor cpa_blocker to input block
      if(mode==1) for(i=0;i<16;i++) *(bff_o+i) ^=  *(cpa_bl +i);
  }
  else  if(!dofeistel_n(2*blolen, keylen,bff1, bff_o, key2, -feistelsteps, ksteps,cpa_blen, cpa_bl,algonum))
      { printf("Error in dofeistel");}
  //erster Block ist rueckgewandelt

#if defined(__WXMSW__)
  hlp=sscanf(bff_o,"si3:%Ld ",&filen);
#else
  hlp=sscanf(bff_o,"si3:%lld ",&filen);
#endif
  if(hlp !=1) //filelänge lesen
  { //alles mist
      throwout(_("Konnte nicht dechiffrieren"));
      free(bff1); free(bff_o); free(cpa_bl); free(bff_last); free(key2);
      fclose(cipherfile);
      return false;
  }
  //now open deciphered file
  if(algo==_("Fleas_l")) memcpy(fnam+namlen-2,"\0",1); //nur ein extensionZeichen
  else memcpy(fnam+namlen-3,"\0",1); // um "_?3 bereinigter filename fuer den Klarfile
  if((plainfile = fopen(fnam, "wb"))== NULL)
  { free(bff1); free(bff_o); free(cpa_bl); free(bff_last); free(key2);return false;}
  // jetzt rausschreiben plainfile
  /**   Echtlänge des ersten Blocks bilden **/
  // zunächst blockzahl
  blockzahl = (filen + 50) / (2*(long long)blolen);
  if( ((filen + 50) % (2*(long long)blolen)) !=0) { blockzahl++; }
  // sollte vorliegende Blockzahl des Cipherfile sein
  newfilen= blockzahl*2*(long long)blolen;
  blinds=newfilen-filen;
  restlen=(int)(filen-(blockzahl-1)*2*(long long)blolen);
  // Echtlänge des ersten Block rausschreiben
  for( i= 0  /*2*blolen - restlen  */;i<2*blolen;i++)
  {
      bytcount++;
      if(bytcount > blinds)
      {
          fputc((char) *(bff_o + i),plainfile) ;
      }
  }
  //jetzt das restliche file
  for(j=1;j<blockzahl;j++)
  {
       if((shoprog>0)&&(blockzahl>1))
       {
         //prog bar propagation in main loop
         currpos=(int)( (100*j)/(blockzahl-1));
         if((currpos >lastmax)&&goodp)
         {
            goodp=pbar.Update(currpos);
            lastmax=currpos;
         }
       }

      for(i=0;i<2*blolen;i++)
      {
          *(bff1 + i)= (char) fgetc(cipherfile);
      }
      if(algo==_("aes"))
      {
        //use aes in ecb mode, CBC conversion already done
        if(aes_crypt_ecb( &aesc, AES_DECRYPT,(const unsigned char*)bff1,(unsigned char*)bff_o)!=0) throwout(_("error in aes transformation"));
       }
       else if(!dofeistel_n(2*blolen, keylen,bff1, bff_o, key2, -feistelsteps,ksteps,cpa_blen, cpa_bl, algonum))
          { printf("Error in dofeistel");}
      //if in CBC mode, xor with prev Cipherblock
      if(mode==1)
      {
        for(i=0;i<2*blolen;i++)
        {
          *(bff_o + i) ^= *(bff_last + i);
        }
      }

      for(i=0;i<2*blolen;i++)
      {
          bytcount++;
          if(bytcount > blinds)
                fputc((char) *(bff_o + i),plainfile);
      }
            //save last Cipherblock for next loop
      if(mode == 1) memcpy(bff_last,bff1,2*blolen);

  }
  free(bff1); free(bff_o); free(cpa_bl); free(key2);
  if(mode == 1) free(bff_last);

  fclose(plainfile);
  fclose(cipherfile);
  return true;
}
/*********************************************/
bool dec_fil_4ecc(ulong32 deadhead, wxString cnam, wxString* pnampt, char* key, int keylen, char feistelsteps, int ksteps, int blolen, int mode, wxString algo, unsigned char shoprog)
// entschlüsselt ein File mit key
// wahre Blocklänge 2*blolen
// alte filelänge wird vorangestellt len: long
{
  char *hlp1,*hlp2;
  bool bv;

  if( (algo==_("F_cnt_1c"))||(algo==_("F_cnt_1d"))||(algo==_("F_cnt_1b")) )//catch new countermode
  {
      return( decipher_file_cntmode_mt4ecc(deadhead, cnam, pnampt, key, keylen, ksteps, blolen, 2, algo, shoprog) );
  }
  if( (algo==_("flight"))||(algo==_("flightx"))|| (algo==_("threefish")) || (algo == _("chimera")))//catch new countermode
  {
      return( decipher_file_cntmode_4ecc(deadhead, cnam, pnampt, key, keylen, ksteps, blolen, 2, algo, shoprog) );
  }
  hlp1= (char *) malloc(4*cnam.Len()+2);
  strncpy(hlp1, (const char*)cnam.mb_str(wxConvUTF8),cnam.Len());
  if((pnampt==NULL)||(*pnampt==_("")))
  {
      hlp2= (char *) malloc(10);
      strncpy(hlp2,"dummy",5);
  }
  else  //refer to old dec_ciph?
  {
          hlp2= (char *) malloc(4*pnampt->Len()+2);
          strncpy(hlp2,(const char*)cnam.mb_str(wxConvUTF8),pnampt->Len());
  }
  bv=  dec_fil(deadhead, hlp1, hlp2, key, keylen, feistelsteps, ksteps, blolen, mode, algo, shoprog);
  free(hlp1); free(hlp2);
  if(!bv) //deciphering failed
  {
      throwout(_("Dechiffrierung gescheitert!\nLoesche kaputtes Zwischenprodukt."),1);
      if(wxFile::Exists(_("dummy"))) wxRemoveFile(_("dummy"));
  }
  return bv;
}
/*********************************************/
bool dec_fil(ulong32 deadhead, char* fnam, char* pnam, char* key, int keylen, char feistelsteps, int ksteps, int blolen, int mode, wxString algo, unsigned char shoprog)
// entschlüsselt ein File mit key
// wahre Blocklänge 2*blolen
// alte filelänge wird vorangestellt len: long
{
  long long filen=0, blockzahl, j,newfilen,blinds,bytcount=0;
  FILE *plainfile, *cipherfile;
  char *bff1, *bff_o, *cpa_bl, *bff_last, *key2;
  int restlen,i,namlen, cpa_blen,hlp,algoflag;
  int currpos, lastmax=0;
  bool goodp=true;
  aes_context aesc;



  if( (algo==_("F_cnt_1c"))||(algo==_("F_cnt_1d"))||(algo==_("F_cnt_1b")) )//catch new countermode
  {
      return( decipher_file_cntmode_mt(deadhead, fnam, pnam, key, keylen, ksteps, blolen, 2, algo, shoprog) );
  }
  if( (algo==_("flight"))||(algo==_("flightx")) )//catch new countermode
  {
      return( decipher_file_cntmode(deadhead, fnam, key, keylen, ksteps, blolen, 2, algo, shoprog) );
  }
  if( (algo==_("threefish")) )//catch new countermode
  {
      return( decipher_file_threefish(deadhead, fnam, key, keylen, ksteps, blolen, 2, algo, shoprog) );
  }
  if( (algo==_("chimera")) )//catch new countermode
  {
      return( decipher_file_fleafish(deadhead, fnam, key, keylen, ksteps, blolen, 2, algo, shoprog) );
  }

  if((blolen<26) && (algo !=_("aes"))) return false;// kein Platz für die Filelänge!
  namlen=strlen(fnam);
  if((cipherfile = fopen(fnam, "rb"))== NULL) return false;
  fseek(cipherfile,deadhead, SEEK_SET); //dead header überspringen



  bff1 = (char *)malloc(2*blolen); bff_o = (char *)malloc(2*blolen);
  if(mode == 1) bff_last = (char *)malloc(2*blolen);
  //algocode bestimmen
  if(algo ==_("Fleas_1_8"))   { algoflag=0;}
    else if(algo ==_("Fleas_3"))   {algoflag=3;}
     else if(algo ==_("Fleas_4"))   { algoflag=4;}
     else if(algo ==_("Fleas_5"))   { algoflag=5;}
     else if(algo ==_("Fleas_x2"))   { algoflag=6;}
     else if(algo ==_("Fleas_x5"))   { algoflag=7;}
     else if(algo ==_("Fleas_o2"))   { algoflag=8;}
     else if(algo ==_("Fleas_o5"))   { algoflag=9;}
     else if(algo ==_("Fleas_l"))   { algoflag=10;}
     else if(algo ==_("Fleas_ls"))   {  algoflag=11;}
     else if(algo ==_("aes"))       { blolen=8; algoflag=12;}
     else if(algo ==_("Fleas_l3"))   {  algoflag=13;}
     else if(algo ==_("Fleas_lc"))   {  algoflag=14;}
     else if(algo ==_("Fleas_ld"))   {  algoflag=15;}
     else if(algo ==_("Fleas_lb"))   {  algoflag=16;} //only locally defined, elsewhere 16 corresponds to flightx
       else{
            throwout(_("unknown algorithm! \n aborting (4)"));
            free(bff1); free(bff_last); free(bff_o);return false;}

//initialize Progress bar
  wxProgressDialog pbar(("Entschluesselungs-Fortschritt"), ("Prozessor arbeitet wie wild..."),100);


  //calculate and produce cpa_attack blocker
  cpa_blen=blolen; //cpa_blen is matching blolen
  /** blolen must be 8 for aes!!  **/
  if((algo == _("aes"))&&(blolen != 8))
  {
      throwout(_("fatal error in aes blocksize!")); return false;
  }
  // prepare (possibly derived) key2
  key2= (char *) malloc(keylen);
  memcpy(key2,key,keylen);
  switch( algoflag) //different behaviour concerning cpa_blocking
  {
    case 4 :
      // build derived key for use with Fleas_4
      develop_c(key2,keylen,2);
      // build cpa_blocker with different algorithm
      cpa_bl= develop_b_malloc(keylen, key,4, cpa_blen);
      break;

    case 5 :
      // build derived key for use with Fleas_4
      develop_x(key2,keylen,2,3);
      // build cpa_blocker with different algorithm
      cpa_bl= develop_b_malloc(keylen, key,4, cpa_blen);
      break;

    case 6 :
      // build derived key for use with Fleas_x2
      develop_x1(key2,keylen,2,3);
      // build cpa_blocker with different algorithm
      cpa_bl= develop_x_malloc(keylen, key,4,2, cpa_blen);
      break;

    case 7 :
      // build derived key for use with Fleas_x5
      develop_x1(key2,keylen,2,3);
      // build cpa_blocker with different algorithm
      cpa_bl= develop_x_malloc(keylen, key,4,5, cpa_blen);
      break;

    case 8 :
      // build derived key for use with Fleas_o2
      develop_o(key2,keylen,2,3,0);
      // build cpa_blocker with different algorithm
      cpa_bl= develop_o_malloc(keylen, key,3,2, cpa_blen,0);
      break;

    case 9 :
      // build derived key for use with Fleas_o5
      develop_o(key2,keylen,2,3,2);
      // build cpa_blocker with different algorithm
      cpa_bl= develop_o_malloc(keylen, key,3,5, cpa_blen,2);
      break;


    case 10 :
      // build derived key for use with Fleas_l
      develop_l(key2,keylen,2);  //not used later on! only uses cpa_bl
      // build cpa_blocker with different algorithm
      cpa_bl= develop_l_malloc(keylen, key, 3,cpa_blen);
      break;

    case 11 :
      // build derived key for use with Fleas_ls
      develop_l(key2,keylen,2);
      // build cpa_blocker with different algorithm
      cpa_bl= develop_l_malloc(keylen, key, 3,cpa_blen);
      break;

    case 12 :
      // build derived key for use with aes
      develop_o(key2,keylen,2,3,2); //not used
      // build cpa_blocker with different algorithm
      cpa_bl= develop_o_malloc(keylen, key, 3,5,6*blolen,2); //cpablen, erste 16 für cpablock, 2.&3. 16 für key in 2 rounds
      //initiate aes-structure
      break;

    case 13 :
      // build derived key for use with Fleas_l3
      develop_ls(key2,keylen,3,3,1);
      // build cpa_blocker with different algorithm
      cpa_bl= develop_l_malloc(keylen, key, 3,cpa_blen);
      break;

    case 14 :
      // build derived key for use with Fleas_lc
      develop_ln(key2,keylen,3,3,1);
      // build cpa_blocker with different algorithm
      cpa_bl= develop_l_malloc(keylen, key, 3,cpa_blen);
      break;

    case 15 :
      // build derived key for use with Fleas_ld (also for fleas_ld)
      develop_ln2_mt(key2,keylen,3,3,1);
      // build cpa_blocker with different algorithm
      cpa_bl= develop_l_malloc(keylen, key, 4,cpa_blen);
      break;

    case 16 :
      // build derived key for use with Fleas_lb (also for fleas_ld)
      develop_ln2_mt(key2,keylen,3,3,1);
      // build cpa_blocker with different algorithm
      cpa_bl= develop_l_malloc(keylen, key, 2,cpa_blen);
      break;

    default:
      cpa_bl=bytmixxmalloc(keylen,key,6,cpa_blen);
      break;


  }

  for( i=0;i<2*blolen;i++)
  {
     *(bff1 + i)= (char) fgetc(cipherfile);
  }
  //erster Block geholt
  if((algo !=_("aes"))&&(mode == 1)) memcpy(bff_last,bff1,2*blolen);  //wenn CBC, lastblock fuellen
  if(algo==_("aes"))
  {
      //set encipher structure
        if (aes_setkey_dec( &aesc, (const unsigned char *)(cpa_bl+16), 256 )!= 0)
          throwout(_("error in setting aes key context"));
      //use aes in ecb mode
        if(aes_crypt_ecb( &aesc, AES_DECRYPT,(const unsigned char*)bff1,(unsigned char*)bff_o)!=0) throwout(_("error in aes transformation"));
      //xor cpa_blocker to input block
      if(mode==1) for(i=0;i<16;i++) *(bff_o+i) ^=  *(cpa_bl +i);
  }
  else  if(!dofeistel(2*blolen, keylen,bff1, bff_o, key2, -feistelsteps, ksteps,cpa_blen, cpa_bl,algo))
      { printf("Error in dofeistel");}
  //erster Block ist rueckgewandelt

#if defined(__WXMSW__)
  hlp=sscanf(bff_o,"si3:%Ld ",&filen);
#else
  hlp=sscanf(bff_o,"si3:%lld ",&filen);
#endif
  if(hlp !=1) //filelänge lesen
  { //alles mist
      throwout(_("Konnte nicht dechiffrieren"));
      free(bff1); free(bff_o); free(cpa_bl); free(bff_last); free(key2);
      fclose(cipherfile);
      return false;
  }
  //now open deciphered file
  if((plainfile = fopen(pnam, "wb"))== NULL)
  { free(bff1); free(bff_o); free(cpa_bl); free(bff_last); free(key2);fclose(cipherfile);return false;}
  // jetzt rausschreiben plainfile
  /**   Echtlänge des ersten Blocks bilden **/
  // zunächst blockzahl
  blockzahl = (filen + 50) / (2*(long long)blolen);
  if( ((filen + 50) % (2*(long long)blolen)) !=0) { blockzahl++; }
  // sollte vorliegende Blockzahl des Cipherfile sein
  newfilen= blockzahl*2*(long long)blolen;
  blinds=newfilen-filen;
  restlen=(int)(filen-(blockzahl-1)*2*(long long)blolen);
  // Echtlänge des ersten Block rausschreiben
  for( i= 0  /*2*blolen - restlen  */;i<2*blolen;i++)
  {
      bytcount++;
      if(bytcount > blinds)
      {
          fputc((char) *(bff_o + i),plainfile) ;
      }
  }
  //jetzt das restliche file
  for(j=1;j<blockzahl;j++)
  {
       if((shoprog>0)&&(blockzahl>1))
       {
         //prog bar propagation in main loop
         currpos=(int)( (100*j)/(blockzahl-1));
         if((currpos >lastmax)&&goodp)
         {
            goodp=pbar.Update(currpos);
            lastmax=currpos;
         }
       }

      for(i=0;i<2*blolen;i++)
      {
          *(bff1 + i)= (char) fgetc(cipherfile);
      }
      if(algo==_("aes"))
      {
        //use aes in ecb mode, CBC conversion already done
        if(aes_crypt_ecb( &aesc, AES_DECRYPT,(const unsigned char*)bff1,(unsigned char*)bff_o)!=0) throwout(_("error in aes transformation"));
       }
       else if(!dofeistel(2*blolen, keylen,bff1, bff_o, key2, -feistelsteps,ksteps,cpa_blen, cpa_bl, algo))
          { printf("Error in dofeistel");}
      //if in CBC mode, xor with prev Cipherblock
      if(mode==1)
      {
        for(i=0;i<2*blolen;i++)
        {
          *(bff_o + i) ^= *(bff_last + i);
        }
      }

      for(i=0;i<2*blolen;i++)
      {
          bytcount++;
          if(bytcount > blinds)
                fputc((char) *(bff_o + i),plainfile);
      }
            //save last Cipherblock for next loop
      if(mode == 1) memcpy(bff_last,bff1,2*blolen);

  }
  free(bff1); free(bff_o); free(cpa_bl); free(key2);
  if(mode == 1) free(bff_last);

  fclose(plainfile);
  fclose(cipherfile);
  return true;
}
/*********************************************/
/****************************************/
bool rsa_encipher_file(char* fnam, rsakey* rsak, char feistelsteps, int k_steps, int blolen)
// verschlüsselt ein File mit dem RSA Verfahren
// wahre Blocklänge 2*blolen, blolen muss größer 25 sein
// alte filelänge long len wird vorangestellt
{
  long filen=0, blockzahl, j;
  FILE *plainfile, *cipherfile;
  char *bff1, *bff_o;
  int restlen,i;
  char *fleakey;
  longnumber fk_c,hlp;
  bool flag;

// banales abfangen
  if((rsak->type != 2)&&(rsak->type != 3)){
      throwout(_("warning!\n unable to rsa-decipher \n with incomplete keyset!"));
      if(rsak->type ==0){
          throwout(_("warning , untested keyset!"));}
  }
//zunächst den Zufallskey bilden
  fleakey=(char *) malloc(rsak->blosiz);
  globlonu.randomize(1);
  for(i=0;i<(int) rsak->blosiz;i++)
  {
      *(fleakey+i)=*(globlonu.ad + i%globlonu.length); // vorübg. raus
     // *(fleakey+i)='c';//vorueb. rein
      if(i==(int) globlonu.length) globlonu.randomize(1);
  }//jetzt ist der Zufallskey gesetzt
  // zufallskey mit pubkey verschlüsseln
  //ersten block nach hlp holen
  hlp.resizelonu(rsak->blosiz,0);//hilfsvariable auf rsa-Blockgroesse
  memcpy(hlp.ad, fleakey,rsak->blosiz);
  if(rsak->type == 2) hlp.pownum_q(&(rsak->e), &(rsak->n));
  else if(rsak->type == 3)
  {
      lonucrt xg(&hlp,&(rsak->p),&(rsak->q));
      flag=xg.powcrt(&(rsak->e));
      xg.exportnum(&hlp);
  }
  hlp.shrinktofit();

  if(blolen<26) return false;// kein Platz für die Filelänge!

  if(strlen(fnam) > 198) return false;
  if((plainfile = fopen(fnam, "rb"))== NULL) return false;
  // jetzt cipherfile öffnen mit anhängsel "_c"
  strncat(fnam,"_rsa",4);
  if((cipherfile = fopen(fnam, "wb"))== NULL) return false;
  fprintf(cipherfile,"rsa: %d :",(int)hlp.length);
  for(i=0;i<(int)hlp.length;i++)
  {
      fputc((char) *(hlp.ad + i),cipherfile);
  }

  while (!feof(plainfile)) {   // bestimme die Filelänge
      fgetc (plainfile);
      filen++;
      }
      rewind(plainfile);
  //bestimme jetzt die Blockzahl
  blockzahl = (filen + 50) / (2*blolen); if( ((filen + 50) % (2*blolen)) !=0) { blockzahl++; }
  bff1 = (char *)malloc(2*blolen); bff_o = (char *)malloc(2*blolen);
  srand((unsigned int) *(bff_o)); // randomvariable mit unbestimmtem seeden
  sprintf(bff1,"siz:%ld ",filen); // hat jetzt die echtlänge an den Anfang geschrieben
  // jetzt restlänge bestimmen
  restlen=filen-(blockzahl-1)*2*blolen;
  //differenz mit Zufallszahlen auffüllen

  for( i=strlen(bff1)+1;i<2*blolen;i++){
      if( i>=(2*blolen)-restlen ) *(bff1 + i)= (char) fgetc(plainfile);
      else *(bff1 + i)= (char) (rand() % 256); // mit zufallszahl auffüllen
  } // füllt den Rest mit dem Filerest auf, dazwischen wirds mit zufallszahlen aufgefüllt und wird bei decipher auch weggeschmissen
  // jetzt rausschreiben in cipherfile
  /********* muss noch verschl. werden ************/
  if(!dofeistel(2*blolen, rsak->blosiz,bff1, bff_o, fleakey, feistelsteps,k_steps))
      { printf("Error in dofeistel");}
      //hlp=bff1; bff1=bff_o; bff_o=hlp;

// Feistel Netzwerk Funktion macht neuen Block aus altem Block + key
//flag=1 fur encipher, -1 für decipher


   for( i=0;i<2*blolen;i++){
     fputc((char) *(bff_o + i),cipherfile) ;
  } // füllt den Rest mit dem Filerest auf, dazwischen bleibts unbestimmt und wird bei decipher auch weggeschmissen
  //jetzt das restliche file
  for(j=1;j<blockzahl;j++)
  {
      for(i=0;i<2*blolen;i++)
      {
          *(bff1 + i)= (char) fgetc(plainfile);
      }
      // jetzt muss enciphert werden
          if(!dofeistel(2*blolen, rsak->blosiz,bff1, bff_o, fleakey, feistelsteps,k_steps))
          { printf("Error in dofeistel");}
    //rausschreiben
      for(i=0;i<2*blolen;i++)
      {
         fputc((char) *(bff_o + i),cipherfile) ;
      }
  }
  free(bff1); free(bff_o);
  fclose(plainfile);
  fclose(cipherfile);
  free(fleakey);
  return true;
}
/****************************************/
/****************************************/
bool rsa_decipher_file(char* fnam, rsakey *rsak,  char feistelsteps, int ksteps, int blolen)
// entschlüsselt ein File mit key
// wahre Blocklänge 2*blolen
// alte filelänge wird vorangestellt len: long


{
  long filen=0, blockzahl, j;
  FILE *plainfile, *cipherfile;
  char *bff1, *bff_o;
  int restlen,i,namlen,rsablolen,f1;
  char *fleakey;
  longnumber fk_c,hlp;



  if(blolen<26) return false;// kein Platz für die Filelänge!
  if(rsak->type < 2) {
      throwout(_("cannot decipher, unknown private key!"));
      return false;
  }
  namlen=strlen(fnam);
  if((cipherfile = fopen(fnam, "rb"))== NULL) return false;
  // jetzt cipherfile öffnen mit anhängsel "_rsa"
  memcpy(fnam+namlen-3,"p\0",2);
  if((plainfile = fopen(fnam, "wb"))== NULL) return false;
  //zunächst rsa-Info einlesen, auswerten und abspalten
  f1=fscanf(cipherfile,"rsa: %d :",&rsablolen);
  //if(rsablolen != (int)rsak->blosiz) throwout(_("attention!!!\n discrepancies in blocksize"));
  fleakey=(char *) malloc(rsak->blosiz);
  hlp.resizelonu(rsablolen,0);//hilfsvariable auf rsa-Blockgroesse
  for(i=0;i<rsablolen;i++)
  {
      *(hlp.ad + i)=fgetc(cipherfile);
  }
  if(rsak->type ==2) hlp.pownum_q(&(rsak->d),&(rsak->n));
  else if(rsak->type == 3)
  {
      lonucrt xg(&hlp,&(rsak->p),&(rsak->q));
      xg.powcrt(&(rsak->d));
      xg.exportnum(&hlp);
  }

  memcpy(fleakey,hlp.ad,rsak->blosiz);

  bff1 = (char *)malloc(2*blolen); bff_o = (char *)malloc(2*blolen);
  // sprintf(bff1,"siz:%ld",filen); // hat jetzt die echtlänge an den Anfang geschrieben
  // jetzt restlänge bestimmen

  for( i=0;i<2*blolen;i++){
     // if( i>=(2*blolen)-restlen )
     *(bff1 + i)= (char) fgetc(cipherfile);
  }
  // hier Einzelblock dechiffrieren
      if(!dofeistel(2*blolen, rsak->blosiz,bff1, bff_o, fleakey, feistelsteps, ksteps))
      { printf("Error in dofeistel");}

  sscanf(bff_o,"siz:%ld",&filen); //filelänge lesen
  //
  /**   Echtlänge des ersten Blocks bilden **/
  // zunächst blockzahl
  blockzahl = (filen + 50) / (2*blolen); if( ((filen + 50) % (2*blolen)) !=0) { blockzahl++; }
  restlen=filen-(blockzahl-1)*2*blolen;
  // Echtlänge des ersten Block rausschreiben
   for( i=2*blolen - restlen;i<2*blolen;i++){
     fputc((char) *(bff_o + i),plainfile) ;
  }
  //jetzt das restliche file
  for(j=1;j<blockzahl;j++)
  {
      for(i=0;i<2*blolen;i++)
      {
          *(bff1 + i)= (char) fgetc(cipherfile);
      }
      // jetzt muss deciphert werden
      if(!dofeistel(2*blolen, rsak->blosiz,bff1, bff_o, fleakey, feistelsteps,ksteps))
          { printf("Error in dofeistel");}

      for(i=0;i<2*blolen;i++)
      {
         fputc((char) *(bff_o + i),plainfile) ;
      }
  }
  free(bff1); free(bff_o);
  fclose(plainfile);
  fclose(cipherfile);
  free(fleakey);
  return true;
}
/*********************************************/
bool getsignature(wxString fnam, longnumber* signo, int blocksize)
{
    ulong32 origlen,blockzahl,indx;
    unsigned int startbyte,readno,i,rounds=3;
    char *plainblock, *ciphblock, *keyblock,*hlp;

    wxFFile docfile(fnam,  _("rb"));
    // throwout(_("open command sent") );
    if( !docfile.IsOpened() ) {throwout(_("Document \n not opened error")); return false; }
    origlen=docfile.Length(); //Laenge des File bestimmen
    blockzahl= (origlen/(ulong32)blocksize) ; //blockzahl bestimmen
    if(origlen%blocksize !=0) blockzahl++; // nur wenns nicht aufgeht, ein block mehr
    startbyte=(unsigned int)((ulong32)blocksize*blockzahl -  origlen);
    plainblock=(char *)malloc(blocksize);
    ciphblock=(char *)malloc(blocksize);
    keyblock=(char *)malloc(blocksize);

    if(blockzahl <2)
     {
       throwout(_("Warnung!!\n Datei ist zu klein!!\npadding ist notwendig"),4);
       readno=docfile.Read(plainblock,(unsigned int)origlen);
       if(readno != origlen) throwout(_("Error reading File 8-O"));
       for(indx=(ulong32)readno;indx<(ulong32)blocksize;indx++)
          *(plainblock+indx)= (*(plainblock+(indx%readno)))^((char) indx);//plainblock wird vervollständigt
       for(i=0;i<rounds+1;i++) //im letzten block eine runde mehr
       {
       if(!bytmixx(blocksize,plainblock,ciphblock,1))
              throwout(_("Unknown Error in Hashing"));
         hlp=plainblock; // plain und cipherblock vertauschen, so dass jetzt cipher weiterverarb wird
         plainblock=ciphblock;
         ciphblock=hlp;
       }
     }
     else
     {
// ersten Block verarbeiten, reentrant auffüllen, so dass nur noch glatte bloecke bleiben
       readno=docfile.Read(plainblock,blocksize - startbyte);
       if(readno != (blocksize - startbyte)) throwout(_("Error reading File 8-O"));
       for(indx=(ulong32)readno;indx<(ulong32)blocksize;indx++)
          *(plainblock+indx)=(*(plainblock+(indx%readno)))^((char) indx) ;//wird mit xortem mit indx aufgefüllt
       for(i=0;i<rounds;i++)
       {
       //  erster teilaufgefüllter block
       if(!bytmixx(blocksize,plainblock,ciphblock,1))
              throwout(_("Unknown Error in Hashing"));
         hlp=plainblock;// plain und cipherblock vertauschen, so dass jetzt cipher weiterverarb wird
         plainblock=ciphblock;
         ciphblock=hlp;
       }
       memcpy(keyblock,plainblock,blocksize); //output nun auch nach "keyblock"
       //jetzt mit den normalen glatten blöcken weiter
       for(indx=1;indx<blockzahl;indx++)
       {
         readno=docfile.Read(plainblock,blocksize);
         if(readno != (unsigned int)blocksize) throwout(_("Error reading File 8-O"));
         //im letzten block eine runde mehr, loest das "extension problem"
         if(indx+1 == blockzahl) rounds++;
         for(i=0;i<rounds;i++)
         {
           //if(!keyedbytejump4(blono,plainblock,ciphblock,blono,keyblock))
           if(!kbytmixx(blocksize,plainblock,ciphblock,blocksize,keyblock,1))
                throwout(_("Unknown Error in Hashing"));
           hlp=plainblock;     //switch adresses
           plainblock=ciphblock;
           ciphblock=hlp;
         }// immer wieder cipher und plain durchspielen und wechseln
         for(i=0;i<(unsigned int)blocksize;i++)// letzte cipher im "plainblock"
         {
              *(keyblock +i) = *(plainblock +i);
         }

       }// zum nächsaten block
     }
     docfile.Close();
     signo->fillwith(plainblock,blocksize); //ist aber die cipher!
     if (plainblock != NULL)  free(plainblock);
     if (ciphblock != NULL)  free(ciphblock);
     if (keyblock != NULL)  free(keyblock);
     return true;
}
/****************************************/
bool storrsig(wxString fnam,longnumber* psig, rsakey* pkey, int sigsiz)
{
    wxString sigval;
    ulong32 strinlen;
    char *buff;


    //fnam += _(".rsg");

    wxFFile rsgfile(fnam,  _("rb"));
    if( !rsgfile.IsOpened() ) {throwout(_("sig_text \n not opened error")); return false; }

    psig->writechar(&sigval);
     sigval= _("rsg: ") +sigval;
     strinlen= sigval.Length();
       buff = (char *)malloc(2*strinlen+2);
       strcpy( buff, (const char*) sigval.mb_str(wxConvUTF8) );
       rsgfile.Write(buff,strinlen);
       free(buff);
       rsgfile.Close();
    return true;
}
/*******************************************/
  /*************************************************************/
    int prep_rsa(longnumber *pln,int rsa_len, int padsiz)
  //prepariert formatinformation(=size) vorne dran, setzt die letzten 2 bytes jedes blockes mit zufallszahlen und füllt ab size mit Zufallszahlen auf
  // gib Blockzahl zurueck(-1 bei error)
  {
      ulong32 cnt,newlen,oldlen,i,done;
      unsigned int headsiz;
      int blono=0,blocks;
      longnumber tmp;


      headsiz=sizeof(ulong32);
      if( (unsigned int) padsiz > (rsa_len -headsiz -1))
      {
          throwout(_("too much padding error! \n nothing left for codin!"));
          return -1;
      }
      pln->shrinktofit();
      oldlen= pln->size;
      newlen = pln->size + headsiz;
      blocks= newlen/(rsa_len-padsiz);
      if((newlen % (rsa_len-padsiz)) !=0) blocks ++;
      newlen = blocks*rsa_len;
      tmp.resizelonu(newlen,0);
      // schreib alte size an vorderste 4 byte
      cnt=0; done=0;
      memcpy(tmp.ad,&oldlen,headsiz); //laenge dran!,  memcpy(to, from,count)
      cnt += headsiz;
      do{
          if(cnt%(ulong32)rsa_len == (ulong32)(rsa_len -padsiz))
          // setze ans ende jedes blockes padsiz zufallsbytes
          {
              blono++;
              for(i=cnt;i<cnt+padsiz;i++)
              {
                 *(tmp.ad + i)= (unsigned char) rand();
                 //*(tmp.ad + i)= 'x';
              }
              cnt += padsiz;
          }
          else
          {
            *(tmp.ad+cnt) = *(pln->ad+done);
            cnt++; done++;
          }
      } while (done < pln->size);
      if(cnt< tmp.length)
      {
        while( cnt < tmp.length)
            //fuelle auf ganze blockzahl mit zufallsbytes auf
        {
           *(tmp.ad+cnt) = (unsigned char) rand();
           cnt++;
        }
        blono++; //blockzahl steht jetzt
      }
      //gib blockzahl zurueck,-1 bei error
      pln->copynum(&tmp);
      return blono;
  }
  /*******************************************************************/
  ulong32 clean_rsa(longnumber *pln,int rsa_len, int padsiz)
  //prepariert longnumber zurueck und returned den alten wert "size"
  {
      ulong32 cnt,done,oldlen;
      unsigned int headsiz;
      int blono=0;
      longnumber tmp;

 //Konsistenzcheck
    if(pln->length%rsa_len != 0)
    {
        throwout(_("Inconsisten Blocklength \n -> aborting clean_rsa()"));
        return 0;
    }
    cnt=0;done=0;
    headsiz=sizeof(ulong32);
    //bestimme size(erste 4 byte)
    memcpy(&oldlen,pln->ad,headsiz);
    cnt += headsiz;
    tmp.resizelonu(oldlen,0);
      //bestimme blockzahl und fülle jeweils die gueltigen bytes in neue lonu ein
    if(done<oldlen)
    {
      while(done<oldlen)
      {
        if(cnt%(ulong32)rsa_len == (ulong32)(rsa_len -padsiz))
        {
            cnt += padsiz;
            blono++;
        }
        else
        {
           *(tmp.ad+done)= *(pln->ad + cnt);
           done++; cnt++;
        }
      }
    blono++;
    }
      //shrinktofit und kopiere nach this
      tmp.shrinktofit();
      pln->copynum(&tmp);
        // gib size zurueck
    return oldlen;
  }
/*******************************************************************/
  bool enciph_d_rsa(longnumber *pln,rsakey *ky)
  //prepariert und vermuckelt alle bloecke, aendert lonu in rsa_cipher
  {
      int i,blono,blsiz,idx;
      longnumber tmp(8),scr(8);
      ulong32 newlen,offs;

        // ruf prep_rsa auf
      ky->n.shrinktofit(); //wer weiss, ob der key vernünftig "normalisiert" ist..
      ky->n.setsize(true);
      blsiz=ky->n.size - 1;
      i=prep_rsa(pln,blsiz);//padding done!

      if((pln->length % blsiz) !=0)
      {
          throwout(_("inconsistent length in enciph\n out of prep_rsa!"));
          return false;
      }

      blono = pln->length/blsiz; //blockzahl
      newlen =blono * (blsiz+1);
      tmp.resizelonu(newlen,0);
      scr.resizelonu((ulong32)blsiz+1,0);
      for(i=0;i<blono;i++)
      {
          idx=i*(blsiz+1);
          memcpy(scr.ad, pln->ad + i*(blsiz),blsiz); //quelle noch im kleineren Raster
          scr.trusiz=false;
          *(scr.ad + blsiz) = 0; //höchstes byte auf null im groesseren Zielraster
          scr.setsize(true);
          //jetzt potenzieren mit privkey
          if(ky->type == 3)
          {
              scr.q_pownum(&(ky->d),&(ky->p),&(ky->q));
              //scr.shrinktofit();
          }
          else if(ky->type == 2)
                {
                    scr.pownum_q(&(ky->d),&(ky->n));
                    //scr.shrinktofit();
                }
                else {
                    throwout(_("signature needs private key!!?!"));
                    return false;
                }
           offs=(ulong32)i * (ulong32)(blsiz+1);
           memcpy((tmp.ad + offs ), scr.ad ,blsiz+1);
      }
      tmp.setsize(true);
      pln->copynum(&tmp);
      return true;
       //gib true zurueck, wenns geklappt hat
  }
  /*******************************************************************/
  bool deciph_e_rsa(longnumber *pln,rsakey *ky)
  //rueckmuckelt alle Bloecke und putzt auf Ursprungsformat, icl. shrinktofit
  {
      int i,blono,blsiz,nblsiz;
      longnumber tmp(2),scr(4);
      ulong32 offs1,newlen;

        // ruf prep_rsa auf
        blono=0;//dummykommando
      ky->n.shrinktofit(); //wer weiss, ob der key vernünftig "normalisiert" ist..
      ky->n.setsize(true);

      blsiz=ky->n.size;
    //check ob glatte blockzahl, sonst error

      if(((pln->length % blsiz) !=0)&&((pln->length % blsiz) !=1)) //vielleicht das zweite weg???
      {
          throwout(_("inconsistent length in deciph\n out Blockmodulus!"));
          return false;
      }
      //glatte bloecke!!
      blono= (int) (pln->length / blsiz);
      scr.resizelonu((ulong32)blsiz,0);
      newlen=(ulong32)blono*(blsiz-1);
      tmp.resizelonu(newlen,0);
      nblsiz=blsiz-1; //neu dazugemuellt!
      for(i=0;i<blono;i++)
      {
          memcpy(scr.ad, pln->ad + i*blsiz,blsiz);
          //jetzt potenzieren mit pubkey
      //decpher each block with pubkey
          if(ky->type == 3)
          {
              scr.q_pownum(&(ky->e),&(ky->p),&(ky->q));
              //scr.shrinktofit();
          }
          else
                {
                    scr.pownum_q(&(ky->e),&(ky->n));
                    //scr.shrinktofit();
                }
                /* hier sitzt ein Fehler wo??? */
          // auf blsiz kappen und hibyte auf 0 testen, error ausgeben, wenn nicht!
          //scr.resizelonu(blsiz);
          if(*(scr.ad + blsiz-1) != 0)
          {
              throwout(_("error in deciph_e_rsa \n hibyte not zero"));
              return false;
          }

          //nblsiz=  scr.size;  /* hier sitzt ein Fehler wo??? */
          nblsiz = blsiz -1;
          /*  if(nblsiz > blsiz-1)
          {
              throwout(_("Incorrect rsa deciphering of \n longnumber \n aborting!!"));
              return false;
          }   */
          offs1=(ulong32)i * (ulong32)(nblsiz);
          memcpy((tmp.ad + offs1 ), scr.ad  ,nblsiz);
      }
          // ruf clean_rsa auf
      clean_rsa(&tmp,nblsiz);
      pln->copynum(&tmp);
      return true;
       //gib true zurueck, wenns geklappt hat
  }

/*******************************************************************/
bool lockfile(wxString fnam, longnumber *keyadress, wxString algo, unsigned char shoprog)
{
    wxString exnam;

    //check: for legal cipher
      if(algo ==_("Fleas_1_8"))   exnam =_("_c3");
     else if(algo ==_("Fleas_3"))    exnam =_("_f3");
     else if(algo ==_("Fleas_4"))     exnam =_("_f4");
     else if(algo ==_("Fleas_5"))   exnam =_("_f5");
     else if(algo ==_("Fleas_x2"))  exnam =_("_x2");
     else if(algo ==_("Fleas_x5"))   exnam =_("_x5");
     else if(algo ==_("Fleas_o2"))   exnam =_("_o2");
     else if(algo ==_("Fleas_o5"))   exnam =_("_o5");
     else if(algo ==_("Fleas_l"))   exnam =_("_l");
     else if(algo ==_("Fleas_ls"))   exnam =_("_ls");
     else if(algo ==_("Fleas_l3"))   exnam =_("_l3");
     else if(algo ==_("Fleas_lc"))   exnam =_("_lc");
     else if(algo ==_("Fleas_ld"))   exnam =_("_ld");
     else if(algo ==_("Fleas_lb"))   exnam =_("_lb");
     else if(algo ==_("aes"))   exnam =_("_ae");
     else if(algo ==_("F_cnt_1c"))   exnam =_("_1c");
     else if(algo ==_("F_cnt_1d"))   exnam =_("_1d");
     else if(algo ==_("F_cnt_1b"))   exnam =_("_1b");
     else if(algo ==_("flight"))   exnam =_("_fl");
     else if(algo ==_("flightx"))   exnam =_("_fx");
     else if(algo ==_("threefish"))   exnam =_("_tf");
     else if(algo ==_("chimera"))   exnam =_("_ff");
     else{throwout(_("unknown algorithm! \n aborting (1)"));return false;}

     if(!(wxFile::Exists(fnam + exnam)))
        return( lockinplace(fnam, keyadress, algo, shoprog)  );
     else //already exists and hopefully has not been changed
     {
         //just delete plain file
             if(!wxRemoveFile(fnam))
             {
                 throwout(_("Achtung! \n Klardatei vielleicht nicht geloescht!\nmuss vllcht. manuell entfernt werden!"));
                  return false;
             }
     }
     return true;
}
/**************************************************************/
bool lockinplace(wxString fnam, longnumber *pkey, wxString algo, unsigned char shoprog)
//verschlüsselt file mit longnumber und loescht original
{

    char fnambuff[3202];
    int fr=4, kr=2, blsiz=65; //mode=1 means cipher block chaining -> ok

    wxString syscommand;

    if(fnam.Len()>800)
    {
        throwout(_("Alarm! Dateiname hat excessive Laenge\n Schneide ab!\nDu solltest an eine boesartige Attacke denken!"));
        fnam=fnam.Left(800);
    }
    strcpy( fnambuff, (const char*)fnam.mb_str(wxConvLocal) );// schreibt den Filenamen in fnambuff rein

    pkey->setsize(true);
    if(algo ==_("Fleas_4")) {fr=5;kr=3;blsiz=512;}
    else if(algo ==_("Fleas_5")) {fr=5;kr=3;blsiz=512;}
    else if(algo ==_("Fleas_x2")) {fr=5;kr=3;blsiz=512;}
    else if(algo ==_("Fleas_x5")) {fr=5;kr=3;blsiz=512;}
    else if(algo ==_("Fleas_o2")) {fr=4;kr=2;blsiz=512;}
    else if(algo ==_("Fleas_o5")) {fr=4;kr=2;blsiz=512;}
    else if(algo ==_("Fleas_l")) {fr=4;kr=2;blsiz=512;}
    else if(algo ==_("Fleas_ls")) {fr=4;kr=2;blsiz=480;}
    else if(algo ==_("Fleas_l3")) {fr=4;kr=3;blsiz=480;}
    else if(algo ==_("Fleas_lc")) {fr=4;kr=3;blsiz=480;}
    else if(algo ==_("Fleas_ld")) {fr=4;kr=3;blsiz=480;}
    else if(algo ==_("Fleas_lb")) {fr=4;kr=3;blsiz=480;}
    else if(algo ==_("aes")) {fr=0;kr=1;blsiz=16;}
    else if(algo ==_("F_cnt_1c")) {fr=4;kr=3;blsiz=480;}
    else if(algo ==_("F_cnt_1d")) {fr=4;kr=3;blsiz=480;}
    else if(algo ==_("F_cnt_1b")) {fr=4;kr=3;blsiz=480;}
    else if(algo ==_("flight")) {fr=4;kr=3;blsiz=480;}
    else if(algo ==_("flightx")) {fr=4;kr=3;blsiz=480;}
    else if(algo ==_("threefish")) {fr=0;kr=0;blsiz=128;}
    else if(algo ==_("chimera")) {fr=4;kr=3;blsiz=512;}
     if(!encipher_file(fnambuff, (char *)(pkey->ad), pkey->size, fr, kr, blsiz,1,algo,shoprog))
    {
        wxMessageBox(_("ALARM!!!, Fehler bei der chiffrierung!!! \nManuelle Entfernung von sensiblen Dateifragmenten erforderlich!!"), _("RED ALERT:"));
        return false;
    }
    if(!wxRemoveFile(fnam))
    {
        throwout(_("Alarm! \nOriginale Klardatei moegl. nicht entfernt!\nBitte manuell Löschen!"));
        return false;
    }
    return true;
}
/***************************************************************/
bool unlockfile(wxString fnam, longnumber *keyadress, wxString algo, unsigned char shoprog)
//entschlüsselt file mit longnumber und loescht original nicht
{
    return unlockinplace(fnam, keyadress, algo, shoprog, false);
}
/***************************************************************/
bool unlockinplace(wxString fnam, longnumber *pkey, wxString algo, unsigned char shoprog, bool del_ciph)
//entschlüsselt file mit longnumber und loescht original
{
    wxString syscommand;
    char fnambuff[3202];
    int fr=4, kr=2, blsiz=65; //defaults for 1_8 and 3

    if(algo== _("Fleas_3")) fnam += _("_f3");
    else if(algo== _("Fleas_1_8")) fnam += _("_c3");
    else if(algo== _("Fleas_4")) fnam += _("_f4");
    else if(algo== _("Fleas_5")) fnam += _("_f5");
    else if(algo== _("Fleas_x2")) fnam += _("_x2");
    else if(algo== _("Fleas_x5")) fnam += _("_x5");
    else if(algo== _("Fleas_o2")) fnam += _("_o2");
    else if(algo== _("Fleas_o5")) fnam += _("_o5");
    else if(algo== _("Fleas_l")) fnam += _("_l");
    else if(algo== _("Fleas_ls")) fnam += _("_ls");
    else if(algo ==_("Fleas_l3")) fnam += _("_l3");
    else if(algo ==_("Fleas_lb")) fnam += _("_lb");
    else if(algo ==_("Fleas_lc")) fnam += _("_lc");
    else if(algo ==_("Fleas_ld")) fnam += _("_ld");
    else if(algo== _("aes")) fnam += _("_ae");
    else if(algo ==_("F_cnt_1c")) fnam += _("_1c");
    else if(algo ==_("F_cnt_1d")) fnam += _("_1d");
    else if(algo ==_("F_cnt_1b")) fnam += _("_1b");
    else if(algo ==_("flight")) fnam += _("_fl");
    else if(algo ==_("flightx")) fnam += _("_fx");
    else if(algo ==_("threefish")) fnam += _("_tf");
    else if(algo ==_("chimera")) fnam += _("_ff");

    else
    {
        throwout(_("Alert! Unknown Algorithm, \nCannot unlock!"));
        return false;
    }

    if(fnam.Len()>800)
    {
        throwout(_("Alarm! Dateiname hat excessive Laenge\n Schneide ab!\nBoesartige Attacke??"));
        fnam=fnam.Left(800);
    }
    strcpy( fnambuff, (const char*)fnam.mb_str(wxConvLocal) );// schreibt den Filenamen in fnambuff rein
    pkey->setsize(true);
    if(algo ==_("Fleas_4")) {fr=5;kr=3;blsiz=512;}
    else if(algo ==_("Fleas_5")) {fr=5;kr=3;blsiz=512;}
    else if(algo ==_("Fleas_x2")) {fr=5;kr=3;blsiz=512;}
    else if(algo ==_("Fleas_x5")) {fr=5;kr=3;blsiz=512;}
    else if(algo ==_("Fleas_o2")) {fr=4;kr=2;blsiz=512;} //new set for supersecure version
    else if(algo ==_("Fleas_o5")) {fr=4;kr=2;blsiz=512;} //new set for supersecure version
    else if(algo ==_("Fleas_l")) {fr=4;kr=2;blsiz=512;} //new set for supersecure version
    else if(algo ==_("Fleas_ls")) {fr=4;kr=2;blsiz=480;} //new set for supersecure version
    else if(algo ==_("Fleas_l3")) {fr=4;kr=3;blsiz=480;} //new set for supersecure version
    else if(algo ==_("Fleas_lb")) {fr=4;kr=3;blsiz=480;} //new set for supersecure version
    else if(algo ==_("Fleas_lc")) {fr=4;kr=3;blsiz=480;} //new set for supersecure version
    else if(algo ==_("Fleas_ld")) {fr=4;kr=3;blsiz=480;}
    else if(algo ==_("aes")) {fr=0;kr=1;blsiz=16;} //new set aes
    else if(algo ==_("F_cnt_1c")) {fr=4;kr=3;blsiz=480;} //new set for counter mode version
    else if(algo ==_("F_cnt_1d")) {fr=4;kr=3;blsiz=480;} //new set for counter mode version
    else if(algo ==_("F_cnt_1b")) {fr=4;kr=3;blsiz=480;} //new set for counter mode version
    else if(algo ==_("flight")) {fr=4;kr=3;blsiz=480;} //new set for counter mode version
    else if(algo ==_("flightx")) {fr=4;kr=3;blsiz=480;} //new set for counter mode version
    else if(algo ==_("threefish")) {fr=0;kr=0;blsiz=128;} //new set for counter mode version
    else if(algo ==_("chimera")) {fr=4;kr=3;blsiz=512;} //new set for counter mode version
    if(!decipher_file(&fnambuff[0], (char *)(pkey->ad), pkey->size, fr, kr, blsiz,1,algo,shoprog))
    {
        wxMessageBox(_("error deciphering, \n cannot proceed! \n cipherfile corrupted?"), _("Grave Problem:"));
        return false;
    }
    if(del_ciph) //only if delete cipher is true
    {

      if(!wxRemoveFile(fnam))
      {
        throwout(_("Alert! \n old cipher possibly not deleted!\n clean manually!"));
        return false;
      }
    }
    return true;
}
/***************************************************/
bool lockfile_eax(wxString fnam, longnumber *keyadress, wxString algo, unsigned char shoprog)
//verschlüsselt file mit longnumber wenn noch nicht vorhanden und loescht original
{
    if(!(wxFile::Exists(fnam +_("_cph"))))
    {
        if(!lockinplace_eax(fnam, keyadress, algo, shoprog));
        throwout(_("Couldn't lock file!!!\n must inspect leftover plainfile manually, \nsave secrets and clean manually 8-O"));
    }
    if(!wxRemoveFile(fnam))
    {
        throwout(_("Alert! \n original possibly not deleted!\n must clean manually!"));
        return false;
    }
    return true;
}
/**************************************************************/
bool lockinplace_eax(wxString fnam, longnumber *pkey, wxString algo, unsigned char shoprog)
//verschlüsselt file mit longnumber und loescht original
{
    wxString cnam;
    int rsef;


    if(!wxFileExists(fnam))
    {
        throwout(_("cannot encipher!\nPlaintext not found!"),3);
        return false;
    }
    //construct cnam
    //find original extension
    cnam= fnam +_("_cph");

    rsef=eax_enc_file(fnam, cnam, (char *)(pkey->ad), (int)(pkey->size), algo, shoprog);
    if(rsef != 0)

    {
        throwout(_("Error in locking file 8-O"));
        return false;
    }


    if(!wxRemoveFile(fnam))
    {
        throwout(_("Alert! \n original possibly not deleted!\n must clean manually!"));
        return false;
    }
    return true;
}
/***************************************************************/
bool unlockfile_eax(wxString fnam, longnumber *pkey, unsigned char shoprog, bool strict)
//entschlüsselt file mit longnumber und loescht original nicht
{
    return( unlockinplace_eax(fnam, pkey, shoprog, strict, false)  );
}
/***************************************************************/
bool unlockinplace_eax(wxString fnam, longnumber *pkey, unsigned char shoprog, bool strict, bool del_ciph)
//entschlüsselt file mit longnumber und loescht original falls wie ueblich del_ciph=true
{
    wxString cnam,hlp;
    int rsdf;
    bool lockres;


    if(wxFileExists(fnam))
    {
        hlp=_("Uups!?!  Plaintext already exists 8-O\n");
        hlp +=   fnam + _(" will be overwritten ");
        throwout( hlp,5);
    }
    //construct cnam
    //find original extension
    cnam= fnam +_("_cph");
    if(!wxFileExists(cnam))
    {
        throwout(_("This message may only appear upon first call after update from prior aca_sig_b40 - else it is an intruder alert!:\nProper authenticated Cipher of public key file does not exist.\nwill search for legacy cipher and try conversion."));
        lockres=unlockinplace(fnam, pkey, _("Fleas_1_8"),0);
        if(lockres)
        {
            throwout(_("Conversion successful :-)"),3);
            return true;
        }
        else
        {
           throwout(_("cannot unlock, fatal error."));
           return false;
        }

    }
    rsdf= eax_dec_file(cnam, fnam,  (char *)(pkey->ad), pkey->size, shoprog, strict);
    if(rsdf != 0)
    {
        throwout(_("Error in unlocking file 8-O"));
        return false;
    }

    if(del_ciph)
    {
      if(!wxRemoveFile(cnam))
      {
        throwout(_("Alert! \n old cipher possibly not deleted!\n clean manually!"));
        return false;
      }
    }
    return true;
}
/*****************stupid windows needs extra treatment Argh!!! ***********************************************/
unsigned int skein1024_from_file( wxString *pfnam, char* skbuff, int deadhead, unsigned int chunksize, bool shobar )
{
    unsigned char c, *data_block;
    unsigned int readno,i,currpos,lastmax=0;;
    int retval;
    bool isdone=false;
    int sk_ret;
    wxString sourcename;
    unsigned long long origlen,chunkzahl;
    unsigned long chunks;
    bool goodp=true;

    SkeinCtx_t ctx;

    //check for existence
    if(!wxFile::Exists(*pfnam))
    {
        if(shobar) throwout(_("Quelldatei fuer Hash existiert nicht!\nAbbruch."),3);
        return 5;
    }
    //get source file length
     wxFFile sourcefile(*pfnam,  _("rb"));
     if((sourcefile.Length()==0)||((unsigned long long)sourcefile.Length() <= (unsigned long long)deadhead) )
     {
         if(shobar) throwout(_("Warnung:\nBerechne Hash von leerer(oder fastleerer Datei? \ngebe dummy zurueck!"),2);
             //init hash
             sk_ret= skeinCtxPrepare(&ctx, Skein1024);
             if(sk_ret !=0) { throwout(_("skein error 1"),3); return 1; }
             sk_ret= skeinInit(&ctx, 1024);
             if(sk_ret !=0) { throwout(_("skein error 2"),3); return 2; }
             sk_ret=skeinFinal(&ctx, (uint8_t*) skbuff);
             if(sk_ret !=0) { throwout(_("skein finalizing error"),5); return 5; }
         return 3;
     }
    origlen=(unsigned long long)sourcefile.Length() - (unsigned long long) deadhead;
    chunkzahl= (origlen/(unsigned long long)chunksize);
     //initialize Progress bar
    wxProgressDialog *pbar;
    if(shobar)
     {
          //wxProgressDialog pbar(_("Hashing Progress"),_("Hashing time may be the bottleneck :-)"),100);
          pbar= new wxProgressDialog(_("Hashing Fortschritt"),_("Skein Hashing......"),100);
     }
    if(!sourcefile.IsOpened())
        {
            throwout(_("could not open file"));
            if((shobar)&&(pbar!=NULL)) delete(pbar);
            return 1;
        }
    //advance deadhead
    if(deadhead > 0)
    for(i=0;i<(unsigned int)deadhead;i++)
     {
       retval=sourcefile.Read(&c,1);
       if (retval!=1)
       {
           throwout(_("could not read enough bytes from file!"));
           if((shobar)&&(pbar!=NULL)) delete(pbar);
           return 2;
       }
     }
    //init hash
    sk_ret= skeinCtxPrepare(&ctx, Skein1024);
    if(sk_ret !=0)
    {
       if(shobar) throwout(_("skein error 1"),3);
       //if((shobar)&&(pbar!=NULL)) delete(pbar);
       return 1;
    }
    sk_ret= skeinInit(&ctx, 1024);
    if(sk_ret !=0)
    {
        if(shobar) throwout(_("skein error 2"),3);
        if((shobar)&&(pbar!=NULL)) delete(pbar);
        return 2;
    }
    //set structure for bar
    // to do now..............................
    data_block= (unsigned char *)malloc(chunksize);
    chunks=0;
    do
    {
    //read next block
    readno=0;
       readno=sourcefile.Read(data_block,chunksize);
       if(readno==chunksize) //block was full
       {
          sk_ret=skeinUpdate(&ctx, (const uint8_t *)data_block, chunksize);
          if(sk_ret !=0)
            {
                if(shobar) throwout(_("skein error 3"),3);
                if(data_block != NULL) free(data_block);
                //if((shobar)&&(pbar!=NULL)) delete(pbar);
                return 3;
            }
       }
       else //EOF was encountered
       {
          isdone=true;
          sk_ret=skeinUpdate(&ctx, (const uint8_t *)data_block, readno);
          if(sk_ret !=0)
            {
                if(shobar) throwout(_("skein error 4"),3);
                if(data_block != NULL) free(data_block);
                if((shobar)&&(pbar!=NULL)) delete(pbar);
                return 4;
            }
       }
       if((shobar)&&(chunkzahl>10))
           {
                currpos=(int)( (100*chunks)/(chunkzahl-1));
                if((currpos >lastmax)&&goodp)
                {
                    goodp=pbar->Update(currpos);
                    lastmax=currpos;
                }
            }
       chunks++;
    } while(!isdone);
    if((shobar)&&(pbar!=NULL)) delete(pbar);
    if(data_block != NULL) free(data_block);
    sk_ret=skeinFinal(&ctx, (uint8_t*) skbuff);
    if(sk_ret !=0) { throwout(_("skein finalizing error"),5); return 5; }
    return 0;
}
/***************************************************/

/****************************************************************/
unsigned int skein1024_mac_from_file( char* fnambuff, char* skbuff, unsigned int keylen, char *key, int deadhead, unsigned int chunksize )
{
    FILE *setfp;
    unsigned char c, *data_block;
    unsigned int readno,i;
    int retval;
    bool isdone=false;
    int sk_ret;

    SkeinCtx_t ctx;

    //open file
    setfp=fopen(fnambuff, "rb");
    if(setfp==NULL) { throwout(_("could not open file")); return 1;}
    //advance deadhead
    if(deadhead > 0)
    for(i=0;i<(unsigned int)deadhead;i++)
     {
       retval=fgetc(setfp);
       if (retval== EOF)
       {
           throwout(_("could not read enough bytes from file!"));
           fclose(setfp);
           return 2;
       }
     }
    //init hash
    sk_ret= skeinCtxPrepare(&ctx, Skein1024);
    if(sk_ret !=0) { throwout(_("skein error 1"),3); return 1; }
    sk_ret= skeinMacInit(&ctx, (uint8_t *)key, keylen, 1024);
    if(sk_ret !=0) { throwout(_("skein error 2"),3); return 2; }
    // to do now..............................
    data_block= (unsigned char *)malloc(chunksize);
    do
    {
    //read next block
    readno=0;
       do
       {
           retval=fgetc(setfp);
           c=(unsigned char) retval;
           if(retval==EOF)
           {
               isdone=true;
           }
           else
            {
                *(data_block+readno)= c;
                readno++;
            }
       }while(!isdone && readno < chunksize);
       if(readno==chunksize) //block was full
       {
          sk_ret=skeinUpdate(&ctx, (const uint8_t *)data_block, chunksize);
          if(sk_ret !=0) { throwout(_("skein error 3"),3); if(data_block != NULL) free(data_block);return 3; }
       }
       else //EOF was encountered
       {
          sk_ret=skeinUpdate(&ctx, (const uint8_t *)data_block, readno);
          if(sk_ret !=0) { throwout(_("skein error 4"),3); if(data_block != NULL) free(data_block);return 4; }
      }
    } while(!isdone);
    if(data_block != NULL) free(data_block);
    sk_ret=skeinFinal(&ctx, (uint8_t*) skbuff);
    if(sk_ret !=0) { throwout(_("skein finalizing error"),5); return 5; }
    fclose(setfp);
    return 0;
}
/***************************************************/
/****************************************************************/
/* calculate hash from file -> needs rh's header file  --old version*/
unsigned int JH_from_file(unsigned int rhlen, char* fnambuff, char* rhbuff, int deadhead, unsigned int chunksize )
{
    FILE *setfp;
    unsigned char c, *data_block;
    unsigned int readno,i;
    int retval;
    hashState state;
    bool isdone=false;

    //open file
    setfp=fopen(fnambuff, "rb");
    if(setfp==NULL) { throwout(_("could not open file")); return 1;}
    //advance deadhead
    if(deadhead > 0)
    for(i=0;i<(unsigned int)deadhead;i++)
     {
       retval=fgetc(setfp);
       if (retval== EOF)
       {
           throwout(_("could not read enough bytes from file!"));
           fclose(setfp);
           return 2;
       }
     }
    //init hash
    if (!( rhlen == 256 || rhlen == 512 ))
       {
           throwout(_("wrong rh-hashlength!"));
           fclose(setfp);
           return 2;
       }
    data_block= (unsigned char *)malloc(chunksize);
    Init(&state, rhlen);
    do
    {
    //read next block
    readno=0;
       do
       {
           retval=fgetc(setfp);
           c=(unsigned char) retval;
           if(retval==EOF)
           {
               isdone=true;
           }
           else
            {
                *(data_block+readno)= c;
                readno++;
            }
       }while(!isdone && readno < chunksize);
       if(readno==chunksize) //block was full
       {
          Update(&state, data_block, chunksize<<3);
       }
       else //EOF was encountered
       {
           Update(&state, data_block, readno<<3);
       }
    } while(!isdone);
    Final(&state, (BitSequence *)rhbuff);
    fclose(setfp);
    if(data_block != NULL) free(data_block);
    return 0;
}
/***************************************************/
bool get_lonu_JH_from_filename(longnumber *ph, wxString *pfnam, int rhlen_b, bool mss, int deadhead)
{
   char rhbuff[66]; //soll rhlen/8 +2 sein
   char fnambuff[3202];
   int res;
   unsigned int rhlen;


   if(rhlen_b <=0) throwout(_("Bullshit error! \n negative bitlength of JH Hash required ?!?"));
   rhlen=(unsigned int) rhlen_b;
   if(rhlen==256) ph->resizelonu(33,0); //das wird der buffer für den Returnwert
   else ph->resizelonu(65,0); //das wird der buffer für den Returnwert
   //pfnam in const char * wandeln
    if((*pfnam).Len()>800)
    {
        {if(mss) throwout(_("Alarm! Dateiname hat excessive Laenge\n Schneide ab!\nDu solltest an eine boesartige Attacke denken!"));}
        *pfnam=(*pfnam).Left(800);
    }
   strcpy( fnambuff, (const char*)(*pfnam).mb_str(wxConvLocal) );// schreibt den Filenamen in fnambuff rein
   if( !wxFileExists(*pfnam))
   {
     {if(mss) throwout(_("Datei: ")+ *pfnam +_(" nicht lesbar"));}
     return false;
   }

   res= JH_from_file(rhlen, fnambuff, rhbuff, deadhead );
   //0 if successful, 1 if fopen failed, or 2 if fread failed
   if(res==1) {if(mss) throwout(_(" opening file to sign failed")); return false;}
   if(res==2) {if(mss) throwout(_(" reading file to sign failed")); return false;}
   //transfer result to lonu
   ph->wipezero();
   memcpy(ph->ad,rhbuff,rhlen>>3);
   ph->setsize(true);
   return true;
}

/***************************************************/
/***************************************************/
bool get_lonu_skein1024_from_filename(longnumber *ph, wxString *pfnam, bool mss, int deadhead)
{
   char fnambuff[3202], resbuff[1024];
   int res;


   ph->resizelonu(129,0); //das wird der buffer für den Returnwert
   //pfnam in const char * wandeln
    if((*pfnam).Len()>800)
    {
        {if(mss) throwout(_("Alarm! Dateiname hat excessive Laenge\n Schneide ab!\nDu solltest an eine boesartige Attacke denken!"));}
        *pfnam=(*pfnam).Left(800);
    }
   strcpy( fnambuff, (const char*)(*pfnam).mb_str(wxConvUTF8) );// schreibt den Filenamen in fnambuff rein
   if( !wxFileExists(*pfnam))
   {
     {if(mss) throwout(_("Datei: ")+ *pfnam +_(" nicht lesbar"));}
     return false;
   }
   res= skein1024_from_file( pfnam, resbuff, deadhead, 4000,mss);
   //0 if successful, 1 if fopen failed, or 2 if fread failed
   if(res==1) {if(mss) throwout(_(" opening file to sign failed")); return false;}
   if(res==2) {if(mss) throwout(_(" reading file to sign failed")); return false;}
   //transfer result to lonu
   ph->wipezero();
   memcpy(ph->ad,resbuff,128);
   ph->setsize(true);
   return true;
}
/***************************************************/
/***************************************************/
bool get_lonu_skein1024_mac_from_filename(longnumber *ph, wxString *pfnam, unsigned int keylen, char *key, bool mss, int deadhead)
{
   char fnambuff[3202], resbuff[1024];
   int res;


   ph->resizelonu(129,0); //das wird der buffer für den Returnwert
   //pfnam in const char * wandeln
    if((*pfnam).Len()>800)
    {
        {if(mss) throwout(_("Alarm! Dateiname hat excessive Laenge\n Schneide ab!\nDu solltest an eine boesartige Attacke denken!"));}
        *pfnam=(*pfnam).Left(800);
    }
   strcpy( fnambuff, (const char*)(*pfnam).mb_str(wxConvLocal) );// schreibt den Filenamen in fnambuff rein
   if( !wxFileExists(*pfnam))
   {
     {if(mss) throwout(_("Datei: ")+ *pfnam +_(" nicht lesbar."));}
     return false;
   }

   res= skein1024_mac_from_file( fnambuff, resbuff, keylen, key, deadhead );
   //0 if successful, 1 if fopen failed, or 2 if fread failed
   if(res==1) {if(mss) throwout(_(" opening file to sign failed")); return false;}
   if(res==2) {if(mss) throwout(_(" reading file to sign failed")); return false;}
   //transfer result to lonu
   ph->wipezero();
   memcpy(ph->ad,resbuff,128);
   ph->setsize(true);
   return true;
}
/****************************************************************/
/* calculate JH HMAC type from file -> needs rh's header file*/
unsigned int JHMAC_from_file(longnumber *pk, unsigned int rhlen, char* fnambuff, char* rhbuff, int deadhead, unsigned int chunksize )
{
    FILE *setfp;
    unsigned char c, *data_block, *inblo, *outblo;
    unsigned int readno,i,keysize,rhbyte;
    int retval;
    hashState state;
    bool isdone=false;

    //open file
    pk->setsize(true);
    keysize= (unsigned int) pk->size;
    //make key into two standard blocks
    rhbyte=rhlen>>3;
    outblo=(unsigned char *) (develop_l_malloc(keysize,(char *)(pk->ad),3,rhbyte));
    if(outblo==NULL){throwout(_("Error:cannot allocate outblo in JHMAC")); return 3;}
    inblo=  (unsigned char*) develop_l_malloc(rhbyte,(char *)outblo,3,rhbyte);
    if(inblo==NULL){throwout(_("Error:cannot allocate inblo in JHMAC")); free(outblo); return 3;}
    setfp=fopen(fnambuff, "rb");
    if(setfp==NULL) { throwout(_("could not open file")); free(outblo); free(inblo); return 1;}
    //advance deadhead
    if(deadhead > 0)
    for(i=0;i<(unsigned int)deadhead;i++)
     {
       retval=fgetc(setfp);
       if (retval== EOF)
       {
           throwout(_("could not read enough bytes from file!"));
           fclose(setfp);
           free(outblo); free(inblo);
           return 2;
       }
     }
    //init hash
    if (!( rhlen == 256 || rhlen == 512 ))
       {
           throwout(_("wrong rh-hashlength!"));
           fclose(setfp);
           free(outblo); free(inblo);
           return 2;
       }
    data_block= (unsigned char *)malloc(chunksize);
    Init(&state, rhlen);
    //feed first with inblock
    Update(&state, inblo, rhlen);
    do
    {
    //read next block
    readno=0;
       do
       {
           retval=fgetc(setfp);
           c=(unsigned char) retval;
           if(retval==EOF)
           {
               isdone=true;
           }
           else
            {
                *(data_block+readno)= c;
                readno++;
            }
       }while(!isdone && readno < chunksize);
       if(readno==chunksize) //block was full
       {
          Update(&state, data_block, chunksize<<3);
       }
       else //EOF was encountered
       {
           Update(&state, data_block, readno<<3);
       }
    } while(!isdone);
    Final(&state, (BitSequence *)rhbuff);
    //final treat with rhbuff, rhbuff has length rhlen>>3 in bytes
    if(!co_develop_ln2( rhbuff,(int)rhbyte,(char *)outblo,(int) rhbyte,3,3,1))
    {
        throwout(_("Error in codevelop_ln2_at JHMAC"));
        free(outblo); free(inblo);free(data_block);
        return 4;
    }
    fclose(setfp);
    free(outblo); free(inblo);
    if(data_block != NULL) free(data_block);
    return 0;
}
/***************************************************/
bool get_lonu_JHMAC_from_filename(longnumber *ph, longnumber *pk, wxString *pfnam, int rhlen_b, bool mss, int deadhead)
{
   char rhbuff[66]; //soll rhlen/8 +2 sein
   char fnambuff[3202];
   int res;
   unsigned int rhlen;


   if(rhlen_b <=0) throwout(_("Bullshit error! \n negative bitlength of JH Hash required ?!?"));
   rhlen=(unsigned int) rhlen_b;
   if(rhlen==256) ph->resizelonu(33,0); //das wird der buffer für den Returnwert
   else ph->resizelonu(65,0); //das wird der buffer für den Returnwert
   //pfnam in const char * wandeln
    if((*pfnam).Len()>800)
    {
        {if(mss) throwout(_("Alarm! Dateiname hat excessive Laenge\n Schneide ab!\nBoesartige Attacke??"));}
        *pfnam=(*pfnam).Left(800);
    }
   strcpy( fnambuff, (const char*)(*pfnam).mb_str(wxConvLocal) );// schreibt den Filenamen in fnambuff rein
   if( !wxFileExists(*pfnam))
   {
     {if(mss) throwout(_("Datei: ")+ *pfnam +_(" nicht lesbar."));}
     return false;
   }

   res= JHMAC_from_file(pk, rhlen, fnambuff, rhbuff, deadhead );
   //0 if successful, 1 if fopen failed, or 2 if fread failed
   if(res==1) {if(mss) throwout(_(" opening file to sign failed")); return false;}
   if(res==2) {if(mss) throwout(_(" reading file to sign failed")); return false;}
   //transfer result to lonu
   ph->wipezero();
   memcpy(ph->ad,rhbuff,rhlen>>3);
   ph->setsize(true);
   return true;
}

/***************************************************/
bool get_lonu_sha2_from_filename(longnumber *ph, wxString *pfnam, int shalen, bool mss, int deadhead)
{
   unsigned char shabuff[64];
   char fnambuff[3202];
   int res;


   if(shalen==256) ph->resizelonu(33,0); //das wird der buffer für den Returnwert
   else ph->resizelonu(65,0); //das wird der buffer für den Returnwert
   //pfnam in const char * wandeln
    if((*pfnam).Len()>800)
    {
        {if(mss) throwout(_("Alarm! Dateiname hat excessive Laenge\n Schneide ab!\nBoesartige Attacke??"));}
        *pfnam=(*pfnam).Left(800);
    }
   strcpy( fnambuff, (const char*)(*pfnam).mb_str(wxConvLocal) );// schreibt den Filenamen in fnambuff rein
   if( !wxFileExists(*pfnam))
   {
     {if(mss) throwout(_("Datei: ")+ *pfnam +_(" nicht lesbar"));}
     return false;
   }


   if(shalen==256) res= sha2_file_ddhd( fnambuff, shabuff, 0, deadhead );
   else res= sha4_file_ddhd( fnambuff, shabuff, 0, deadhead );
   //0 if successful, 1 if fopen failed, or 2 if fread failed
   if(res==1) {if(mss) throwout(_(" opening file to sign failed")); return false;}
   if(res==2) {if(mss) throwout(_(" reading file to sign failed")); return false;}
   //transfer result to lonu
   ph->wipezero();
   memcpy(ph->ad,shabuff,shalen/8);
   ph->setsize(true);
   return true;
}
/****************************************************************/

/***************************************************/
bool get_lonu_sha2_HMAC_from_filename(longnumber *ph, wxString *pfnam, longnumber *pky, int shalen, bool mss, int deadhead)
{
   unsigned char shabuff[64];
   char fnambuff[1602];
   int res;


   if(shalen==256) ph->resizelonu(34,0); //das wird der buffer für den Returnwert
   else ph->resizelonu(66,0); //das wird der buffer für den Returnwert
   ph->wipezero();
   //pfnam in const char * wandeln
    if((*pfnam).Len()>800)
    {
        {if(mss) throwout(_("Alarm! Dateiname hat excessive Laenge\n Schneide ab!\nBoesartige Attacke??"));}
        *pfnam=(*pfnam).Left(800);
    }
   strcpy( fnambuff, (const char*)(*pfnam).mb_str(wxConvLocal) );// schreibt den Filenamen in fnambuff rein
   if( !wxFileExists(*pfnam))
   {
     {if(mss) throwout(_("Datei: ")+ *pfnam +_(" nicht lesbar"));}
     return false;
   }


   if(shalen==256)
   {
       res= sha2_file_ddhd( fnambuff, shabuff, 0 ,deadhead);
       //now hmac result
       sha2_hmac((const unsigned char *)(pky->ad), pky->length, shabuff, shalen>>3 , ph->ad, 0);
   }
   else {
        res= sha4_file_ddhd( fnambuff, shabuff, 0, deadhead );
        //now hmac result
        sha4_hmac((const unsigned char *)(pky->ad), pky->length, shabuff, shalen>>3 , ph->ad, 0);

   }
   //0 if successful, 1 if fopen failed, or 2 if fread failed
   if(res==1) {if(mss) throwout(_(" opening file to sign failed(1)")); return false;}
   if(res==2) {if(mss) throwout(_(" reading file to sign failed(2)")); return false;}
   ph->setsize(true);
   return true;
}
/****************************************************************/
bool get_lonu_hash_from_filename(longnumber *ph, wxString *pfnam, unsigned int blono, bool mss, int deadhead)
//hashwert bestimmen und binär nach *ph schreiben
{
      wxString hashval,wfl,pdn;
      unsigned long long indx,blockzahl,origlen;
      unsigned int readno,i,startbyte;
      unsigned int rounds=2;
      char *plainblock, *ciphblock, *keyblock, *hlp, dummy;
      longnumber lzahl;

     //hashwert bestimmen
     wxFFile plaintext(*pfnam,  _("rb"));
     if( !plaintext.IsOpened() ) {if(mss) throwout(_("plainfile \n not opened error")); return false; }
    //accomodating "deadhead"
         if( (unsigned long long) plaintext.Length() < (unsigned long) deadhead)
         {
             throwout(_("error!\nplainfile too short."));
             plaintext.Close();
             return false;
         }
     for(i=0;(int)i<deadhead;i++)
     {
        readno=plaintext.Read(&dummy,1);
        if(readno != 1)
        {
            throwout(_("cipher file read error\ncannot continue"));
            plaintext.Close();
            return false;
        }
     }
     origlen=plaintext.Length() - (ulong32) deadhead;

     if(origlen==0)
     {
         {if(mss) throwout(_("Warnung:\nKann keinen Hash aus leerer oder fast leerer Datei berechnen\ngebe dummy zurueck!"),10);}
         hashval=_("Empty File. Inserting dummy value for empty file. This is a very short value compared to the regular value!");
         (*ph).stortext(&hashval);
         plaintext.Close();
         return false;
     }


     blockzahl= (origlen/(ulong32)blono);
     if(origlen % (ulong32)blono !=0) { blockzahl++; }

     startbyte=(unsigned int)((ulong32)blono*blockzahl -  origlen); //Ueberstand am Anfang ist irgendwie zu fuellen
     plainblock=(char *)malloc(blono);
     ciphblock=(char *)malloc(blono);
     keyblock=(char *)malloc(blono);

     if(blockzahl <2)
     {
       {if(mss) throwout(_("Warnung!!\n Datei ist zu klein!!\nPadding ist notwendignecessary"));}
       readno=plaintext.Read(plainblock,(unsigned int)origlen);
       if(readno != origlen) throwout(_("Error reading File 8-O"));
       for(indx=readno;indx<blono;indx++)
          *(plainblock+indx)= (*(plainblock+(indx%readno)))^((char) indx);
       for(i=0;i<rounds+1;i++)
       {
        if(!bytmixx_d((int)blono,plainblock,ciphblock,1))
              {if(mss) throwout(_("Unknown Error in Hashing"));}
         hlp=plainblock; // plain und cipherblock vertauschen, so dass jetzt cipher weiterverarb wird
         plainblock=ciphblock;
         ciphblock=hlp;
       }
     }
     else
     {
// ersten Block verarbeiten, reentrant auffüllen, so dass nur noch glatte bloecke bleibeb
       readno=plaintext.Read(plainblock,blono - startbyte); //restzahl vom ersten block einlesen
       if(readno != (blono - startbyte)) {if(mss) throwout(_("Error reading File 8-O"));}
       for(indx=readno;indx<blono;indx++)
          *(plainblock+indx)=(*(plainblock+(indx%readno)))^((char) indx) ;//wird mit xortem mit indx aufgefüllt
       for(i=0;i<rounds;i++)
       {
       //  erster teilaufgefüllter block
       if(!bytmixx_d((int)blono,plainblock,ciphblock,1)) // durchmixern und nach ciphblock schreiben
              {if(mss) throwout(_("Unknown Error in Hashing"));}
         hlp=plainblock;// plain und cipherblock vertauschen, so dass jetzt cipher weiterverarb wird
         plainblock=ciphblock;
         ciphblock=hlp;
       }
       for(i=0;i<blono;i++)
       {
           *(keyblock +i) = *(plainblock +i); //resultat der ersten vermixerung wird key beim zweiten block
       }
       for(indx=1;indx<blockzahl;indx++)
       {
         readno=plaintext.Read(plainblock,blono); //einlesen von naechstem plainblock
         if(readno != blono) {if(mss) throwout(_("Error reading File 8-O"));}
         if(indx+1 ==blockzahl) //im letzten block eine runde mehr, loest das "extension problem"
         { rounds++;}
         for(i=0;i<rounds;i++)
         {
           if(!kbytmixx((int)blono,plainblock,ciphblock,blono,keyblock,1))
                {if(mss) throwout(_("Unknown Error in Hashing"));}
           hlp=plainblock;     //switch adresses
           plainblock=ciphblock;
           ciphblock=hlp;
         }// immer wieder cipher und plain durchspielen und wechseln
         for(i=0;i<blono;i++)// letzte cipher im "plainblock"
         {
              *(keyblock +i) = *(plainblock +i);
         }
         //if(indx+1< blockzahl) memcpy(keyblock, plainblock, blono);

       }// zum nächsaten block
     }
     plaintext.Close();
     //jetzt in die out_lonu schreiben
     ph->fillwith(plainblock,blono);
     //memcpy(ph->ad,plainblock,blono); //in die outzahl kopieren
     ph->setsize(true);
     ph->shrinktofit();
     free(plainblock); free(ciphblock); free(keyblock);
     return true;
}

/**************************************************/
bool get_lonu_hash_from_filename_x(longnumber *ph, wxString *pfnam, unsigned int blono, wxString algo, bool shobar, int deadhead)
//hashwert bestimmen und binär nach *ph schreiben
{
      wxString hashval,wfl,pdn,h1;
      unsigned long long indx,blockzahl,origlen;
      unsigned int readno,i,j,startbyte;
      unsigned int rounds=2;
      char *plainblock, *ciphblock, *keyblock, *hlbl, dummy;
      longnumber lzahl;
      //for progress bar:
      int currpos, algonum, lastmax=0;
      bool goodp=true;

    //check for existence
    if(!wxFile::Exists(*pfnam))
    {
        if(shobar) throwout(_("file to hash does not exist!\nDo Nothing."),3);
        return false;
    }

     //hashwert bestimmen
     wxFFile plaintext(*pfnam,  _("rb"));
     if((plaintext.Length()==0)||((unsigned long long)plaintext.Length() <= (unsigned long long)deadhead) )
     {
         if(shobar) throwout(_("Warnung:\nKann nicht den Hash einer leeren oder fast leeren Datei berechnen\ngebe dummy zurueck!"),10);
         h1=_("Empty File. Inserting dummy value for empty file. This is a very short value compared to the regular value!");
         (*ph).stortext(&h1);
         return false;
     }
     //if old hash Algo, refer to other routines
     if(algo==_("Fleas_1_8"))
     {
          return get_lonu_hash_from_filename(ph, pfnam,128, deadhead);
     }
     else if(algo==_("Fleas_3"))
     {
          return get_lonu_hash_from_filename_3(ph, pfnam,128, deadhead);
     }
     else if(algo==_("sha2"))
     {
          return get_lonu_sha2_from_filename(ph, pfnam,256,true,deadhead);
     }
     else if(algo==_("sha4"))
     {
          return get_lonu_sha2_from_filename(ph, pfnam, 512, true, deadhead);
     }
     else if(algo==_("JH"))
     {
          return get_lonu_JH_from_filename(ph, pfnam, 512, true, deadhead);
     }
     //set storage structure
     memblo mb(blono,4);

     if( !plaintext.IsOpened() ) {if(shobar) throwout(_("plainfile \n not opened error")); return false; }

         //accomodating "deadhead"
         if( (unsigned long long) plaintext.Length() < (unsigned long) deadhead)
         {
             throwout(_("error!\nplainfile too short."));
             plaintext.Close();
             return false;
         }
     for(i=0;(int)i<deadhead;i++)
     {
        readno=plaintext.Read(&dummy,1);
        if(readno != 1)
        {
            throwout(_("cipher file read error\ncannot continue"));
            plaintext.Close();
            return false;
        }
     }
     origlen=(unsigned long long)plaintext.Length() - (unsigned long) deadhead;


     blockzahl= (origlen/(unsigned long long)blono);
     if(origlen % (unsigned long long)blono !=0) { blockzahl++; }

     startbyte=(unsigned int)((unsigned long long )blono*blockzahl -  origlen); //Ueberstand am Anfang ist irgendwie zu fuellen
     plainblock=(char *)malloc(blono);
     ciphblock=(char *)malloc(blono);
     keyblock=(char *)malloc(blono);
     hlbl=(char *)malloc(blono);

     //initialize Progress bar
     wxProgressDialog *pbar;
     if(shobar)
     {
          //wxProgressDialog pbar(_("Hashing Progress"),_("Hashing time may be the bottleneck :-)"),100);
          pbar= new wxProgressDialog(_("Hashing Fortschritt"),_("Hashing-Zeit kann der Flaschenhals sein :-)"),100);
     }

       //Fallunterscheidung für switch
        algonum=9; //default
       if(algo==_("Fleas_x2")) algonum=2;
       else if(algo==_("Fleas_x5")) algonum=3;
       else if(algo==_("Fleas_o2")) algonum=4;
       else if(algo==_("Fleas_o5")) algonum=5;
       else if( algo==_("Fleas_l") ) algonum=6;
       else if( algo==_("Fleas_lx") ) algonum=7;
       else if( algo==_("Fleas_lx3") ) algonum=8;
       else if( algo==_("Fleas_lc") ) algonum=9;
       else if( algo==_("Fleas_ld") ) algonum=10;
       else if( algo==_("Fleas_lb") ) algonum=11;
       else if( algo==_("Fleas_c") ) algonum=12;
       else if( algo==_("Fleas_d") ) algonum=13;
       else if( algo==_("Fleas_b") ) algonum=14;
       else{ if(shobar) throwout(_("Error, unknown hash\ndefaulting to Fleas_lc"),3);}




     if(blockzahl <2)
     {
       if(shobar) throwout(_("Warnung!!\nDatei ist sehr klein!\npadding notwendig"));
       readno=plaintext.Read(plainblock,(unsigned int)origlen);
       if(readno != origlen){ if(shobar)  throwout(_("Error reading File 8-O"));}
       for(indx=readno;indx<blono;indx++)
          *(plainblock+indx)= (*(plainblock+(indx%readno)))^((char) indx); //periodisch aufgefüllt
    //Fallunterscheidung einziger Block
       switch(algonum)
       {

         case 2:
          for(i=0;i<rounds+1;i++)
          {
            if(!develop_x1(plainblock,blono,rounds,2))
              if(shobar) throwout(_("Err1 in get_lonu_hash_.._x(2)"));
          }
          break;

        case 3:
          for(i=0;i<rounds+1;i++)
          {
            if(!develop_x1(plainblock,blono,rounds,5))
              if(shobar) throwout(_("Err2 in get_lonu_hash_.._x(5)"));
          }
          break;


        case 4:
          for(i=0;i<rounds+1;i++)
          {
            if(!develop_o(plainblock,blono,rounds,2,0))
              if(shobar) throwout(_("Err2 in get_lonu_hash_.._o(2)"));
          }
          break;

        case 5:
          for(i=0;i<rounds+1;i++)
          {
            if(!develop_o(plainblock,blono,rounds,5,2))
              if(shobar) throwout(_("Err2 in get_lonu_hash_.._o(5)"));
          }
          break;

        case 6:
          for(i=0;i<rounds+1;i++)
          {
            if(!develop_l(plainblock,blono,rounds))
             if(shobar)  throwout(_("Err2 in get_lonu_hash_.._lean"));
          }
          break;

        case 7:
          //to prevent "early cancellation attack" mirror-xor output of first round with input
          //keyblock is allocated, still free and can be used as temp storage
          memcpy(keyblock,plainblock,blono);
          for(i=0;i<rounds+1;i++)
          {
            if(!co_develop_l(plainblock,blono,(char *)&startbyte,sizeof(unsigned int),rounds))
              if(shobar) throwout(_("Err2 in get_lonu_hash_.._lean_x"));
            for(j=0;j<blono;j++) //mirrored xoring step
            {
               *(plainblock+j) ^= *(keyblock+blono-j-1);
            }
            develop_l(plainblock,blono,1);
          }
          break;

       case 8:
          //to prevent "early cancellation attack" mirror-xor output of first round with input
          //keyblock is allocated, still free and can be used as temp storage
          memcpy(keyblock,plainblock,blono);
          for(i=0;i<rounds+1;i++)
          {
            if(!co_develop_ls(plainblock,blono,(char *)&startbyte,sizeof(unsigned int),rounds+1,3))
              if(shobar) throwout(_("Err2 in get_lonu_hash_.._lean_x3"));
            for(j=0;j<blono;j++) //mirrored xoring step
            {
               *(plainblock+j) ^= *(keyblock+blono-j-1);
            }
            develop_l(plainblock,blono,1);
          }
          break;

        case 12:
        case 9:
          //to prevent "early cancellation attack" mirror-xor output of first round with input
          //keyblock is allocated, still free and can be used as temp storage
          memcpy(keyblock,plainblock,blono);
          for(i=0;i<rounds+1;i++)
          {
            if(!co_develop_ln2(plainblock,blono,(char *)&startbyte,sizeof(unsigned int),rounds+1,3))
              if(shobar) throwout(_("Err2 in get_lonu_hash_.._c//lc"));
            for(j=0;j<blono;j++) //mirrored xoring step
            {
               *(plainblock+j) ^= *(keyblock+blono-j-1);
            }
            develop_l(plainblock,blono,1);
          }
          break;

        case 13:
        case 10:
          //to prevent "early cancellation attack" mirror-xor output of first round with input
          //keyblock is allocated, still free and can be used as temp storage
          memcpy(keyblock,plainblock,blono);
          for(i=0;i<rounds+1;i++)
          {
            if(!co_develop_ln2(plainblock,blono,(char *)&startbyte,sizeof(unsigned int),rounds+1,4))
              if(shobar) throwout(_("Err2 in get_lonu_hash_.._d//ld"));
            for(j=0;j<blono;j++) //mirrored xoring step
            {
               *(plainblock+j) ^= *(keyblock+blono-j-1);
            }
            develop_l(plainblock,blono,1);
          }
          break;

         case 14:
         case 11:
          //to prevent "early cancellation attack" mirror-xor output of first round with input
          //keyblock is allocated, still free and can be used as temp storage
          memcpy(keyblock,plainblock,blono);
          for(i=0;i<rounds+1;i++)
          {
            if(!co_develop_ln2(plainblock,blono,(char *)&startbyte,sizeof(unsigned int),rounds+1,2))
              if(shobar) throwout(_("Err2 in get_lonu_hash_.._lean_b"));
            for(j=0;j<blono;j++) //mirrored xoring step
            {
               *(plainblock+j) ^= *(keyblock+blono-j-1);
            }
            develop_ln2_mt(plainblock,blono,1,2,1,1);
          }
          break;
       default:
         if(shobar) throwout(_("Fatal Error, Hash algorithm not known"));
         return false;
    //Ende Fallunterscheidung einziger Block
       }
     }
     else //more than one block
     //put in switch instead of ifs
     {
// ersten Block verarbeiten, reentrant auffüllen, so dass nur noch glatte bloecke bleibeb
       readno=plaintext.Read(plainblock,blono - startbyte); //restzahl vom ersten block einlesen
       if(readno != (blono - startbyte)) {if(shobar) throwout(_("Error reading File 8-O"));}
       for(indx=readno;indx<blono;indx++)
          *(plainblock+indx)=(*(plainblock+(indx%readno)))^((char) indx) ;//wird mit xortem mit indx aufgefüllt
       switch(algonum)
       {
    //Fallunterscheidung erster Block
        case 2: //Fleas x2
          for(i=0;i<rounds+1;i++)
          {
            if(!develop_x1(plainblock,blono,rounds,2))
            {
                if(shobar) throwout(_("Err1 in get_lonu_hash_.._x(2)"));
            }
          }
       break;

       case 3:  //Fleas x5
          for(i=0;i<rounds+1;i++)
          {
            if(!develop_x1(plainblock,blono,rounds,5))
            {
                if(shobar) throwout(_("Err2 in get_lonu_hash_.._x(5)"));
            }
          }
       break;

       case 4:  //fleas o2
          for(i=0;i<rounds+1;i++)
          {
            if(!develop_o(plainblock,blono,rounds,2,0))
            {
               if(shobar) throwout(_("Err2 in get_lonu_hash_.._o(2)"));
            }
          }
       break;

       case 5: //Fleas o5
          for(i=0;i<rounds+1;i++)
          {
            if(!develop_o(plainblock,blono,rounds,5,2))
            {
              if(shobar) throwout(_("Err2 in get_lonu_hash_.._o(5)"));
            }
          }
        break;
        case 6:   //Fleas_l
          for(i=0;i<rounds+1;i++)
          {
            if(!develop_l(plainblock,blono,rounds))
              {if(shobar) throwout(_("Err2 in get_lonu_hash_.._lean"));}
          }
       break;

       case 7:  //Fleas_lx
          //to prevent "early cancellation attack" mirror-xor output of first round with input
          //keyblock is allocated, still free and can be used as temp storage
          memcpy(keyblock,plainblock,blono);
          for(i=0;i<rounds+1;i++)
          {
            if(!co_develop_l(plainblock,blono,(char *)&startbyte,sizeof(unsigned int),rounds))
              throwout(_("Err2 in get_lonu_hash_.._lean_x"));
            for(j=0;j<blono;j++) //mirrored xoring step
            {
               *(plainblock+j) ^= *(keyblock+blono-j-1);
            }
            develop_l(plainblock,blono,2);
          }
       break;

       case 8: //Fleas_lx3
          //to prevent "early cancellation attack" mirror-xor output of first round with input
          //keyblock is allocated, still free and can be used as temp storage
          memcpy(keyblock,plainblock,blono);
          for(i=0;i<rounds+1;i++)
          {
            if(!co_develop_ls(plainblock,blono,(char *)&startbyte,sizeof(unsigned int),rounds+1,3))
              {if(shobar) throwout(_("Err2 in get_lonu_hash_.._lean_x3"));}
            for(j=0;j<blono;j++) //mirrored xoring step
            {
               *(plainblock+j) ^= *(keyblock+blono-j-1);
            }
            develop_l(plainblock,blono,2);
          }
       break;
       case 12:
       case 9: //Fleas_lc
          //to prevent "early cancellation attack" mirror-xor output of first round with input
          //keyblock is allocated, still free and can be used as temp storage
          memcpy(keyblock,plainblock,blono);
          for(i=0;i<rounds+1;i++)
          {
            if(!co_develop_ln(plainblock,blono,(char *)&startbyte,sizeof(unsigned int),rounds+1,3))
              {if(shobar) throwout(_("Err2 in get_lonu_hash_.._c//lc"));}
            for(j=0;j<blono;j++) //mirrored xoring step
            {
               *(plainblock+j) ^= *(keyblock+blono-j-1);
            }
            develop_l(plainblock,blono,2);
          }
       break;
       case 13:
       case 10: //Fleas_ld
          //to prevent "early cancellation attack" mirror-xor output of first round with input
          //keyblock is allocated, still free and can be used as temp storage
          memcpy(keyblock,plainblock,blono);
          for(i=0;i<rounds+1;i++)
          {
            if(!co_develop_ln2(plainblock,blono,(char *)&startbyte,sizeof(unsigned int),rounds+1,4))
              {if(shobar) throwout(_("Err2 in get_lonu_hash_.._d//ld"));}
            for(j=0;j<blono;j++) //mirrored xoring step
            {
               *(plainblock+j) ^= *(keyblock+blono-j-1);
            }
            develop_l(plainblock,blono,2);
          }
       break;

        case 14:
        case 11: //Fleas_lb
          //to prevent "early cancellation attack" mirror-xor output of first round with input
          //keyblock is allocated, still free and can be used as temp storage
          memcpy(keyblock,plainblock,blono);
          for(i=0;i<rounds+1;i++)
          {
            if(!co_develop_ln2(plainblock,blono,(char *)&startbyte,sizeof(unsigned int),rounds+1,2))
              {if(shobar) throwout(_("Err2 in get_lonu_hash_.._lean_lb"));}
            for(j=0;j<blono;j++) //mirrored xoring step
            {
               *(plainblock+j) ^= *(keyblock+blono-j-1);
            }
            develop_ln2_mt(plainblock,blono,1,2,1,1);
          }
       break;

       default:
           {if(shobar) throwout(_("Fatal Error, Hash algorithm not known"));}
       }
    //ende Fallunterscheidung erster Block
    //jetzt weitere Bloecke
       memcpy(keyblock,plainblock,blono);
       for(indx=1;indx<blockzahl;indx++)
       {
           if((shobar)&&(blockzahl>1))
           {
                currpos=(int)( (100*indx)/(blockzahl-1));
                if((currpos >lastmax)&&goodp)
                {
                    goodp=pbar->Update(currpos);
                    lastmax=currpos;
                }
            }
         readno=plaintext.Read(plainblock,blono); //einlesen von naechstem plainblock
         if(readno != blono) {if(shobar) throwout(_("Error reading File 8-O"));}
         if(indx+1 ==blockzahl) //im letzten block eine runde mehr
         { rounds++;}
         switch(algonum)
         {

	       case 6: //wenn lean propagate keine "extension into key_derivate
                if(!k_develop_f4(blono,plainblock,blono,keyblock,0,keyblock,rounds,algonum,&mb))
                        throwout(_("Unknown Error 3 in Hashing Fleas_l"));
            break;

           case 7://wenn lean propagate keine "extension into key_derivate
            //to prevent "early cancellation attack" mirror-xor output of last block with input
            //keyblock is allocated, still free and can be used as temp storage
                if(!k_develop_f4(blono,plainblock,0,NULL,blono,keyblock,rounds,algonum))  //no cpa blocking! to keep chaining var untouched
                        {if(shobar) throwout(_("Unknown Error 3 in Hashing Fleas_lx"));}
                for(j=0;j<blono;j++) //mirrored Xoring step
                {
                    *(plainblock+j) ^= *(keyblock+blono-j-1);
                }
                //for dispersion purpose
                if(!develop_l(plainblock,blono,1))
                   {if(shobar) throwout(_("Unknown Error 3 in Hashing Fleas_lx"));}
            break;

            case 8:
            case 9:
            case 10:
                //to prevent "early cancellation attack" mirror-xor output of last block with input
                //keyblock is allocated, still free and can be used as temp storage
                if(!k_develop_f4(blono,plainblock,0,NULL,blono,keyblock,rounds,algonum)) //no cpa blocking! to keep chaining var untouched
                    {if(shobar) throwout(_("Unknown Error 3 in Hashing Fleas_(lx3/lc/ld)"));}
                    for(j=0;j<blono;j++) //mirrored xoring step
                    {
                        *(plainblock+j) ^= *(keyblock+blono-j-1);
                    }
           //for dispersion purpose
                    if(!develop_l(plainblock,blono,2))
                        {if(shobar) throwout(_("Unknown Error 3 in Hashing Fleas_(lx/lc/ld)"));}
            break;

            case 11:
                if(!k_develop_f4(blono,plainblock,0,NULL,blono,keyblock,rounds,algonum)) //no cpa blocking! to keep chaining var untouched
                    {if(shobar) throwout(_("Unknown Error 3 in Hashing Fleas_(lb)"));}
                    for(j=0;j<blono;j++) //mirrored xoring step
                    {
                        *(plainblock+j) ^= *(keyblock+blono-j-1);
                    }
           //for dispersion purpose
                    if(!develop_ln2_mt(plainblock,blono,2,1,1))
                        {if(shobar) throwout(_("Unknown Error 3 in Hashing Fleas_(lb)"));}
            break;
            case 12:
            case 13:
                //to prevent "early cancellation attack" mirror-xor output of last block with input
                memcpy(hlbl,plainblock,blono); //keep reference for longitudinal xor
                if(!develop_ln2_mt(plainblock,blono,rounds,algonum-9,1)) //algonum 12(c)-> 3 paths, 13(d)-> 4 paths
                    {if(shobar) throwout(_("Unknown Error 3a in Hashing Fleas_c//d"));}
                for(j=0;j<blono;j++) //mirrored xoring step
                {
                    *(plainblock+j) ^= *(hlbl+blono-j-1);
                }
                if(!k_develop_f4(blono,keyblock,0,NULL,blono,plainblock,rounds,algonum)) //no cpa blocking! to keep chaining var untouched
                    {if(shobar) throwout(_("Unknown Error 3b in Hashing Fleas_c//d"));}           //for dispersion purpose
            break;

            case 14:
                 //to prevent "early cancellation attack" mirror-xor output of last block with input
                memcpy(hlbl,plainblock,blono);
                if(!develop_ln2_mt(plainblock,blono,rounds,2,1)) //algonum 14(b)-> 2 paths
                    {if(shobar) throwout(_("Unknown Error 3c in Hashing Fleas_b"));}
                    for(j=0;j<blono;j++) //mirrored xoring step
                    {
                        *(plainblock+j) ^= *(hlbl+blono-j-1);
                    }
                if(!k_develop_f4(blono,keyblock,0,NULL,blono,plainblock,rounds,algonum)) //no cpa blocking! to keep chaining var untouched
                    {if(shobar) throwout(_("Unknown Error 3d in Hashing Fleas_b"));}
            break;

            default:
                    if(!k_develop_f4(blono,plainblock,blono,keyblock,blono,keyblock,rounds,algonum))
                            {if(shobar) throwout(_("Unknown Error 3 in Hashing Fleas_(x/o)5"));}
       }

       //Ende Fallunterscheidung restliche Blöcke
          if((algonum<12)||(algonum>14))
          {
              memcpy(keyblock,plainblock,blono); //for 12/13/14 result already in keyblock
          }

      }// zum naechsten block
     }
     plaintext.Close();
     //jetzt in die out_lonu schreiben
     if((algonum<12)||(algonum>14))
     {
        ph->fillwith(plainblock,blono); //ausser Fleas_b/c/d
     }
     else
     {
         ph->fillwith(keyblock,blono);  //fuer Fleas_b/c/d
     }
     ph->setsize(true);
     ph->shrinktofit();
     free(plainblock); free(ciphblock); free(keyblock); free(hlbl);
     if(shobar&&(pbar!=NULL)) delete pbar;
     return true;
}

/**************************************************/
bool get_lonu_hmac_from_filename_x(longnumber *ph, wxString *pfnam, longnumber *pky, unsigned int blono, wxString algo, bool shobar,int deadhead)
//hashwert bestimmen und binär nach *ph schreiben
{
      wxString hashval,wfl,pdn,h1;
      unsigned long long indx,blockzahl,origlen;
      unsigned int readno,i,j,startbyte;
      unsigned int rounds=2;
      char *plainblock, *ciphblock, *keyblock,*hlbl,dummy;
      longnumber lzahl, k1;
      //for progress bar:
      int currpos, algonum, lastmax=0;
      bool goodp=true,res_o;

    //check for existence
    if(!wxFile::Exists(*pfnam))
    {
        if(shobar) throwout(_("Datei zu Hashen ist nicht vorhanden!\nAbbruch."),3);
        return false;
    }
  /******* catch special case skein **********/
    if(algo==_("skein"))
    {
        if(get_lonu_skein1024_mac_from_filename(ph,pfnam,(unsigned int)pky->size,(char *)(pky->ad), false, deadhead ) ) //returns true on success
            return true;
        else {throwout(_("error in skein mac"),5); return false;}
    }
  /******* end special case, go on as usual ************/
     //hashwert bestimmen
     wxFFile plaintext(*pfnam,  _("rb"));
     if(plaintext.Length()<= (unsigned long)deadhead)
     {
         if(shobar) throwout(_("Warnung:\nKann den Hash einer leeren oder fast leeren Datei nicht bestimmen\ngebe dummy zurueck!"),10);
         h1=_("Empty File. Inserting dummy value for empty file. This is a very short value compared to the regular value!");
         (*ph).stortext(&h1);
         return false;
     }
     //prepare secondary key
     k1.copynum(pky);
     k1.setsize();
     develop_ln2((char *)(k1.ad), (int) (k1.size), 2, 3, 4  );



     //if old hash Algo, refer to other routines
     if(algo==_("Fleas_1_8"))
     {
          res_o = get_lonu_hash_from_filename(ph, pfnam,128, deadhead);
          ph->lonu_lock( &k1, algo);
          return res_o;
     }
     else if(algo==_("Fleas_3"))
       {
            res_o = get_lonu_hash_from_filename_3(ph, pfnam,128,deadhead);
            ph->lonu_lock( &k1, algo);
            return res_o;
       }
       else if(algo==_("sha2"))
         {
              res_o = get_lonu_sha2_HMAC_from_filename(ph, pfnam, &k1, 256, true, deadhead);
              return res_o;
         }
           else if(algo==_("sha4"))
           {
               //to redo
                return get_lonu_sha2_HMAC_from_filename(ph, pfnam, &k1, 512, true, deadhead);
           }
             else if(algo==_("JH"))
             {
               //to redo
                return get_lonu_JHMAC_from_filename(ph, &k1, pfnam, 512, true, deadhead);
             }
     //set storage structure
     memblo mb(blono,4);

     if( !plaintext.IsOpened() ) {if(shobar) throwout(_("plainfile \n not opened error")); return false; }
     origlen=plaintext.Length() - deadhead;


     for(i=0;i<deadhead;i++)
     {
        readno=plaintext.Read(&dummy,1);
        if(readno != 1)
        {
            throwout(_("cipher file read error\ncannot continue"));
            plaintext.Close();
            return false;
        }
     }


     blockzahl= (origlen/(unsigned long long)blono);
     if(origlen % (unsigned long long)blono !=0) { blockzahl++; }

     startbyte=(unsigned int)((unsigned long long )blono*blockzahl -  origlen); //Ueberstand am Anfang ist irgendwie zu fuellen
     plainblock=(char *)malloc(blono);
     ciphblock=(char *)malloc(blono);
     keyblock=(char *)malloc(blono);
     hlbl=(char *)malloc(blono);


     //initialize Progress bar
     wxProgressDialog *pbar;
     if(shobar)
     {
          //wxProgressDialog pbar(_("Hashing Progress"),_("Hashing time may be the bottleneck :-)"),100);
          pbar= new wxProgressDialog(_("Hashing Fortschritt"),_("Hashing-Zeit koennte der Flaschenhals sein :-)"),100);
     }

       //Fallunterscheidung für switch
        algonum=9; //default
       if(algo==_("Fleas_x2")) algonum=2;
       else if(algo==_("Fleas_x5")) algonum=3;
       else if(algo==_("Fleas_o2")) algonum=4;
       else if(algo==_("Fleas_o5")) algonum=5;
       else if( algo==_("Fleas_l") ) algonum=6;
       else if( algo==_("Fleas_lx") ) algonum=7;
       else if( algo==_("Fleas_lx3") ) algonum=8;
       else if( algo==_("Fleas_lc") ) algonum=9;
       else if( algo==_("Fleas_ld") ) algonum=10;
       else if( algo==_("Fleas_lb") ) algonum=11;
       else if( algo==_("Fleas_c") ) algonum=12;
       else if( algo==_("Fleas_d") ) algonum=13;
       else if( algo==_("Fleas_b") ) algonum=14;
       else{ if(shobar) throwout(_("Error, unknown hash\ndefaulting to Fleas_lc"),3);}




     if(blockzahl <2)  //nur ein block
     {
       if(shobar) throwout(_("Warnung!!\nDatei ist sehr klein!\nPadding notwendig"));
       readno=plaintext.Read(plainblock,(unsigned int)origlen);
       if(readno != origlen){ if(shobar)  throwout(_("Error reading File 8-O"));}
       for(indx=readno;indx<blono;indx++)
          *(plainblock+indx)= (*(plainblock+(indx%readno)))^((char) indx); //periodisch aufgefüllt
       // develop block with primary key
       co_develop_ln2(plainblock,blono, (char *)(pky->ad),(int)pky->length,2,3);
    //Fallunterscheidung einziger Block
       switch(algonum)
       {

         case 2:
          for(i=0;i<rounds+1;i++)
          {
            if(!develop_x1(plainblock,blono,rounds,2))
              if(shobar) throwout(_("Err1 in get_lonu_hash_.._x(2)"));
          }
          break;

        case 3:
          for(i=0;i<rounds+1;i++)
          {
            if(!develop_x1(plainblock,blono,rounds,5))
              if(shobar) throwout(_("Err2 in get_lonu_hash_.._x(5)"));
          }
          break;


        case 4:
          for(i=0;i<rounds+1;i++)
          {
            if(!develop_o(plainblock,blono,rounds,2,0))
              if(shobar) throwout(_("Err2 in get_lonu_hash_.._o(2)"));
          }
          break;

        case 5:
          for(i=0;i<rounds+1;i++)
          {
            if(!develop_o(plainblock,blono,rounds,5,2))
              if(shobar) throwout(_("Err2 in get_lonu_hash_.._o(5)"));
          }
          break;

        case 6:
          for(i=0;i<rounds+1;i++)
          {
            if(!develop_l(plainblock,blono,rounds))
             if(shobar)  throwout(_("Err2 in get_lonu_hash_.._lean"));
          }
          break;

        case 7:
          //to prevent "early cancellation attack" mirror-xor output of first round with input
          //keyblock is allocated, still free and can be used as temp storage
          memcpy(keyblock,plainblock,blono);
          for(i=0;i<rounds+1;i++)
          {
            if(!co_develop_l(plainblock,blono,(char *)&startbyte,sizeof(unsigned int),rounds))
              if(shobar) throwout(_("Err2 in get_lonu_hash_.._lean_x"));
            for(j=0;j<blono;j++) //mirrored xoring step
            {
               *(plainblock+j) ^= *(keyblock+blono-j-1);
            }
            develop_l(plainblock,blono,1);
          }
          break;

       case 8:
          //to prevent "early cancellation attack" mirror-xor output of first round with input
          //keyblock is allocated, still free and can be used as temp storage
          memcpy(keyblock,plainblock,blono);
          for(i=0;i<rounds+1;i++)
          {
            if(!co_develop_ls(plainblock,blono,(char *)&startbyte,sizeof(unsigned int),rounds+1,3))
              if(shobar) throwout(_("Err2 in get_lonu_hash_.._lean_x3"));
            for(j=0;j<blono;j++) //mirrored xoring step
            {
               *(plainblock+j) ^= *(keyblock+blono-j-1);
            }
            develop_l(plainblock,blono,1);
          }
          break;

        case 12:
        case 9:
          //to prevent "early cancellation attack" mirror-xor output of first round with input
          //keyblock is allocated, still free and can be used as temp storage
          memcpy(keyblock,plainblock,blono);
          for(i=0;i<rounds+1;i++)
          {
            if(!co_develop_ln2(plainblock,blono,(char *)&startbyte,sizeof(unsigned int),rounds+1,3))
              if(shobar) throwout(_("Err2 in get_lonu_hash_.._lean_x3"));
            for(j=0;j<blono;j++) //mirrored xoring step
            {
               *(plainblock+j) ^= *(keyblock+blono-j-1);
            }
            develop_l(plainblock,blono,1);
          }
          break;

        case 13:
        case 10:
          //to prevent "early cancellation attack" mirror-xor output of first round with input
          //keyblock is allocated, still free and can be used as temp storage
          memcpy(keyblock,plainblock,blono);
          for(i=0;i<rounds+1;i++)
          {
            if(!co_develop_ln2(plainblock,blono,(char *)&startbyte,sizeof(unsigned int),rounds+1,4))
              if(shobar) throwout(_("Err2 in get_lonu_hash_.._lean_x3"));
            for(j=0;j<blono;j++) //mirrored xoring step
            {
               *(plainblock+j) ^= *(keyblock+blono-j-1);
            }
            develop_l(plainblock,blono,1);
          }
          break;

        case 14:
        case 11:
          //to prevent "early cancellation attack" mirror-xor output of first round with input
          //keyblock is allocated, still free and can be used as temp storage
          memcpy(keyblock,plainblock,blono);
          for(i=0;i<rounds+1;i++)
          {
            if(!co_develop_ln2(plainblock,blono,(char *)&startbyte,sizeof(unsigned int),rounds+1,2))
              if(shobar) throwout(_("Err2 in get_lonu_hash_.._lean_b"));
            for(j=0;j<blono;j++) //mirrored xoring step
            {
               *(plainblock+j) ^= *(keyblock+blono-j-1);
            }
            develop_ln2_mt(plainblock,blono,1,2,1,1);
          }
          break;
       default:
         if(shobar) throwout(_("Fatal Error, Hash algorithm not known"));
         return false;
    //Ende Fallunterscheidung einziger Block
       }
     }
     else //more than one block
     //put in switch instead of ifs
     {
// ersten Block verarbeiten, reentrant auffüllen, so dass nur noch glatte bloecke bleiben
       readno=plaintext.Read(plainblock,blono - startbyte); //restzahl vom ersten block einlesen
       if(readno != (blono - startbyte)) {if(shobar) throwout(_("Error reading File 8-O"));}
       for(indx=readno;indx<blono;indx++)
          *(plainblock+indx)=(*(plainblock+(indx%readno)))^((char) indx) ;//wird mit xortem mit indx aufgefüllt
       // develop block with primary key
       co_develop_ln2(plainblock,blono, (char *)(pky->ad),(int)pky->length,2,3);

       switch(algonum)
       {
    //Fallunterscheidung erster Block
        case 2: //Fleas x2
          for(i=0;i<rounds+1;i++)
          {
            if(!develop_x1(plainblock,blono,rounds,2))
            {
                if(shobar) throwout(_("Err1 in get_lonu_hash_.._x(2)"));
            }
          }
       break;

       case 3:  //Fleas x5
          for(i=0;i<rounds+1;i++)
          {
            if(!develop_x1(plainblock,blono,rounds,5))
            {
                if(shobar) throwout(_("Err2 in get_lonu_hash_.._x(5)"));
            }
          }
       break;

       case 4:  //fleas o2
          for(i=0;i<rounds+1;i++)
          {
            if(!develop_o(plainblock,blono,rounds,2,0))
            {
               if(shobar) throwout(_("Err2 in get_lonu_hash_.._o(2)"));
            }
          }
       break;

       case 5: //Fleas o5
          for(i=0;i<rounds+1;i++)
          {
            if(!develop_o(plainblock,blono,rounds,5,2))
            {
              if(shobar) throwout(_("Err2 in get_lonu_hash_.._o(5)"));
            }
          }
        break;
        case 6:   //Fleas_l
          for(i=0;i<rounds+1;i++)
          {
            if(!develop_l(plainblock,blono,rounds))
              {if(shobar) throwout(_("Err2 in get_lonu_hash_.._lean"));}
          }
       break;

       case 7:  //Fleas_lx
          //to prevent "early cancellation attack" mirror-xor output of first round with input
          //keyblock is allocated, still free and can be used as temp storage
          memcpy(keyblock,plainblock,blono);
          for(i=0;i<rounds+1;i++)
          {
            if(!co_develop_l(plainblock,blono,(char *)&startbyte,sizeof(unsigned int),rounds))
              throwout(_("Err2 in get_lonu_hash_.._lean_x"));
            for(j=0;j<blono;j++) //mirrored xoring step
            {
               *(plainblock+j) ^= *(keyblock+blono-j-1);
            }
            develop_l(plainblock,blono,2);
          }
       break;

       case 8: //Fleas_lx3
          //to prevent "early cancellation attack" mirror-xor output of first round with input
          //keyblock is allocated, still free and can be used as temp storage
          memcpy(keyblock,plainblock,blono);
          for(i=0;i<rounds+1;i++)
          {
            if(!co_develop_ls(plainblock,blono,(char *)&startbyte,sizeof(unsigned int),rounds+1,3))
              {if(shobar) throwout(_("Err2 in get_lonu_hash_.._lean_x3"));}
            for(j=0;j<blono;j++) //mirrored xoring step
            {
               *(plainblock+j) ^= *(keyblock+blono-j-1);
            }
            develop_l(plainblock,blono,2);
          }
       break;

       case 12:
       case 9: //Fleas_lc
          //to prevent "early cancellation attack" mirror-xor output of first round with input
          //keyblock is allocated, still free and can be used as temp storage
          memcpy(keyblock,plainblock,blono);
          for(i=0;i<rounds+1;i++)
          {
            if(!co_develop_ln(plainblock,blono,(char *)&startbyte,sizeof(unsigned int),rounds+1,3))
              {if(shobar) throwout(_("Err2 in get_lonu_hash_.._lean_x3"));}
            for(j=0;j<blono;j++) //mirrored xoring step
            {
               *(plainblock+j) ^= *(keyblock+blono-j-1);
            }
            develop_l(plainblock,blono,2);
          }
       break;

       case 13:
       case 10: //Fleas_ld
          //to prevent "early cancellation attack" mirror-xor output of first round with input
          //keyblock is allocated, still free and can be used as temp storage
          memcpy(keyblock,plainblock,blono);
          for(i=0;i<rounds+1;i++)
          {
            if(!co_develop_ln2(plainblock,blono,(char *)&startbyte,sizeof(unsigned int),rounds+1,4))
              {if(shobar) throwout(_("Err2 in get_lonu_hash_.._lean_x3"));}
            for(j=0;j<blono;j++) //mirrored xoring step
            {
               *(plainblock+j) ^= *(keyblock+blono-j-1);
            }
            develop_l(plainblock,blono,2);
          }
       break;

       case 14:
       case 11: //Fleas_lb
          //to prevent "early cancellation attack" mirror-xor output of first round with input
          //keyblock is allocated, still free and can be used as temp storage
          memcpy(keyblock,plainblock,blono);
          for(i=0;i<rounds+1;i++)
          {
            if(!co_develop_ln2(plainblock,blono,(char *)&startbyte,sizeof(unsigned int),rounds+1,2))
              {if(shobar) throwout(_("Err2 in get_lonu_hash_.._lean_lb"));}
            for(j=0;j<blono;j++) //mirrored xoring step
            {
               *(plainblock+j) ^= *(keyblock+blono-j-1);
            }
            develop_ln2_mt(plainblock,blono,1,2,1,1);
          }
       break;

       default:
           {if(shobar) throwout(_("Fatal Error, Hash algorithm not known"));}
       }
    //ende Fallunterscheidung erster Block

       memcpy(keyblock,plainblock,blono);
       for(indx=1;indx<blockzahl;indx++)
       {
           if((shobar)&&(blockzahl>1))
           {
                currpos=(int)( (100*indx)/(blockzahl-1));
                if((currpos >lastmax)&&goodp)
                {
                    goodp=pbar->Update(currpos);
                    lastmax=currpos;
                }
            }
         readno=plaintext.Read(plainblock,blono); //einlesen von naechstem plainblock
         if(readno != blono) {if(shobar) throwout(_("Error reading File 8-O"));}
         if(indx+1 ==blockzahl) //im letzten block eine runde mehr, loest das "extension problem", solves extension problem for usage in MAC
         { rounds++;}
         switch(algonum)
         {

	       case 6: //wenn lean propagate keine "extension into key_derivate
                if(!k_develop_f4(blono,plainblock,blono,keyblock,0,keyblock,rounds,algonum,&mb))
                        throwout(_("Unknown Error 3 in Hashing Fleas_l"));
            break;

           case 7://wenn lean propagate keine "extension into key_derivate
            //to prevent "early cancellation attack" mirror-xor output of first round with input
            //keyblock is allocated, still free and can be used as temp storage
                if(!k_develop_f4(blono,plainblock,0,NULL,blono,keyblock,rounds,algonum))  //no cpa blocking! to keep chaining var untouched
                        {if(shobar) throwout(_("Unknown Error 3 in Hashing Fleas_lx"));}
                for(j=0;j<blono;j++) //mirrored Xoring step
                {
                    *(plainblock+j) ^= *(keyblock+blono-j-1);
                }
                //for dispersion purpose
                if(!develop_l(plainblock,blono,1))
                   {if(shobar) throwout(_("Unknown Error 3 in Hashing Fleas_lx"));}
            break;

            case 8:
            case 9:
            case 10:
                //to prevent "early cancellation attack" mirror-xor output of first round with input
                //keyblock is allocated, still free and can be used as temp storage
                if(!k_develop_f4(blono,plainblock,0,NULL,blono,keyblock,rounds,algonum)) //no cpa blocking! to keep chaining var untouched
                    {if(shobar) throwout(_("Unknown Error 3 in Hashing Fleas_(lx3/lc/ld)"));}
                    for(j=0;j<blono;j++) //mirrored xoring step
                    {
                        *(plainblock+j) ^= *(keyblock+blono-j-1);
                    }
           //for dispersion purpose
                    if(!develop_l(plainblock,blono,2))
                        {if(shobar) throwout(_("Unknown Error 3 in Hashing Fleas_(lx/lc/ld)"));}
            break;

            case 11:
                //to prevent "early cancellation attack" mirror-xor output of first round with input
                //keyblock is allocated, still free and can be used as temp storage
                if(!k_develop_f4(blono,plainblock,0,NULL,blono,keyblock,rounds,algonum)) //no cpa blocking! to keep chaining var untouched
                    {if(shobar) throwout(_("Unknown Error 3 in Hashing Fleas_(lb)"));}
                    for(j=0;j<blono;j++) //mirrored xoring step
                    {
                        *(plainblock+j) ^= *(keyblock+blono-j-1);
                    }
           //for dispersion purpose
                    if(!develop_ln2_mt(plainblock,blono,2,1,1))
                        {if(shobar) throwout(_("Unknown Error 3 in Hashing Fleas_(lb)"));}
            break;

            case 12:
            case 13:
                //to prevent "early cancellation attack" mirror-xor output of last block with input
                memcpy(hlbl,plainblock,blono); //keep reference for longitudinal xor
                if(!develop_ln2_mt(plainblock,blono,rounds,algonum-9,1)) //algonum 12(c)-> 3 paths, 13(d)-> 4 paths
                    {if(shobar) throwout(_("Unknown Error 3a in Hashing Fleas_c"));}
                for(j=0;j<blono;j++) //mirrored xoring step
                {
                    *(plainblock+j) ^= *(hlbl+blono-j-1);
                }
                if(!k_develop_f4(blono,keyblock,0,NULL,blono,plainblock,rounds,algonum)) //no cpa blocking! to keep chaining var untouched
                    {if(shobar) throwout(_("Unknown Error 3b in Hashing Fleas_d"));}           //for dispersion purpose
            break;
            case 14:
                //to prevent "early cancellation attack" mirror-xor output of last block with input
                memcpy(hlbl,plainblock,blono);
                if(!develop_ln2_mt(plainblock,blono,rounds,2,1)) //algonum 14(b)-> 2 paths
                    {if(shobar) throwout(_("Unknown Error 3c in Hashing Fleas_b"));}
                    for(j=0;j<blono;j++) //mirrored xoring step
                    {
                        *(plainblock+j) ^= *(hlbl+blono-j-1);
                    }
                if(!k_develop_f4(blono,keyblock,0,NULL,blono,plainblock,rounds,algonum)) //no cpa blocking! to keep chaining var untouched
                    {if(shobar) throwout(_("Unknown Error 3d in Hashing Fleas_b"));}
            break;


            default:
                    if(!k_develop_f4(blono,plainblock,blono,keyblock,blono,keyblock,rounds,algonum))
                            {if(shobar) throwout(_("Unknown Error 3 in Hashing Fleas_(x/o)5"));}
       }
       //Ende Fallunterscheidung restliche Blöcke
          if((algonum<12)||(algonum>14))
          {
              memcpy(keyblock,plainblock,blono); //for 12/13/14 result already in keyblock
          }
      }// zum naechsten block
     }
     plaintext.Close();
     //jetzt co_develop mit k1
     co_develop_ln2(plainblock,blono, (char *)(k1.ad),(int)k1.length,2,3);

     //jetzt in die out_lonu schreiben
     if((algonum<12)||(algonum>14))
     {
        ph->fillwith(plainblock,blono); //ausser Fleas_b/c/d ist das resultat im plainblock
     }
     else
     {
         ph->fillwith(keyblock,blono);  //fuer Fleas_b/c/d liegt die chaining variable in keyblock
     }
     ph->setsize(true);
     ph->shrinktofit();
     free(plainblock); free(ciphblock); free(keyblock); free(hlbl);
     if(shobar&&(pbar!=NULL)) delete pbar;
     return true;
}

/****************************************************************/
/****************************************************************/
bool get_lonu_hash_from_filename_5(longnumber *ph, wxString *pfnam, unsigned int blono,bool shobar, int deadhead)
//hashwert bestimmen und binär nach *ph schreiben
{
      wxString hashval,wfl,pdn;
      unsigned long long indx,blockzahl,origlen;
      unsigned int readno,i,startbyte;
      unsigned int rounds=2;
      char *plainblock, *ciphblock, *keyblock, dummy;
      longnumber lzahl;
      //for progress bar:
      int currpos,lastmax=0;
      bool goodp=true;

     //hashwert bestimmen
     wxFFile plaintext(*pfnam,  _("rb"));
     if( !plaintext.IsOpened() ) {throwout(_("plainfile \n not opened error")); return false; }
         //accomodating "deadhead"
         if( (unsigned long long) plaintext.Length() < (unsigned long) deadhead)
         {
             throwout(_("error!\nplainfile too short."));
             plaintext.Close();
             return false;
         }
     for(i=0;i<deadhead;i++)
     {
        readno=plaintext.Read(&dummy,1);
        if(readno != 1)
        {
            throwout(_("cipher file read error\ncannot continue"));
            plaintext.Close();
            return false;
        }
     }
     origlen=plaintext.Length() - (ulong32) deadhead;
     if(origlen==0)
     {
         {if(shobar) throwout(_("Warnung:\nKann Hash von leerer oder fast leerer Datei nicht berechnen\ngebe dummy zurueck!"),10);}
         hashval=_("Empty File. Inserting dummy value for empty file. This is a very short value compared to the regular value!");
         (*ph).stortext(&hashval);
         plaintext.Close();
         return false;
     }
     blockzahl= (origlen/(ulong32)blono);
     if(origlen % (ulong32)blono !=0) { blockzahl++; }

     startbyte=(unsigned int)((ulong32)blono*blockzahl -  origlen); //Ueberstand am Anfang ist irgendwie zu fuellen
     plainblock=(char *)malloc(blono);
     ciphblock=(char *)malloc(blono);
     keyblock=(char *)malloc(blono);

     //initialize Progress bar
     wxProgressDialog *pbar;
     if(shobar)
     {
        pbar= new wxProgressDialog(_("Hashing Fortschritt"),_("Hashing-Zeit ist der Flaschenhals :-)"),100);


     }

     if(blockzahl <2)
     {
       throwout(_("Warnung!!\nDatei zu klein!!\nPadding notwendig"));
       readno=plaintext.Read(plainblock,(unsigned int)origlen);
       if(readno != origlen) throwout(_("Error reading File 8-O"));
       for(indx=readno;indx<blono;indx++)
          *(plainblock+indx)= (*(plainblock+(indx%readno)))^((char) indx);
       for(i=0;i<rounds+1;i++)
       {
        if(!develop_x(plainblock,blono,rounds,5))
              {if(shobar) throwout(_("Unknown Error 1 in Hashing Fleas_5"));}
       }
     }
     else
     {
// ersten Block verarbeiten, reentrant auffüllen, so dass nur noch glatte bloecke bleibeb
       readno=plaintext.Read(plainblock,blono - startbyte); //restzahl vom ersten block einlesen
       if(readno != (blono - startbyte)) throwout(_("Error reading File 8-O"));
       for(indx=readno;indx<blono;indx++)
          *(plainblock+indx)=(*(plainblock+(indx%readno)))^((char) indx) ;//wird mit xortem mit indx aufgefüllt
       if(!develop_x(plainblock,blono,rounds,5))
              {if(shobar) throwout(_("Unknown Error 2 in Hashing Fleas_4"));}
       memcpy(keyblock,plainblock,blono);
       for(indx=1;indx<blockzahl;indx++)
       {
          if((shobar)&&(blockzahl>1))
          {
            currpos=(int)( (100*indx)/(blockzahl-1));
            if((currpos >lastmax)&&goodp)
            {
                goodp=pbar->Update(currpos);
                lastmax=currpos;
            }
         }
         readno=plaintext.Read(plainblock,blono); //einlesen von naechstem plainblock
         if(readno != blono) {if(shobar) throwout(_("Error reading File 8-O"));}
         if(indx+1 ==blockzahl) //im letzten block eine runde mehr, loest das "extension problem"
         { rounds++;}
           if(!k_develop_f4(blono,plainblock,blono,keyblock,blono,keyblock,rounds,1))
                {if(shobar) throwout(_("Unknown Error 3 in Hashing Fleas_5"));}
          memcpy(keyblock,plainblock,blono);
       }// zum naechsten block
     }
     plaintext.Close();
     //jetzt in die out_lonu schreiben
     ph->fillwith(plainblock,blono);
     //memcpy(ph->ad,plainblock,blono); //in die outzahl kopieren
     ph->setsize(true);
     ph->shrinktofit();
     free(plainblock); free(ciphblock); free(keyblock);
     if(shobar&&(pbar!=NULL)) delete pbar;
     return true;
}

/**************************************************/
/****************************************************************/
bool get_lonu_hash_from_filename_4(longnumber *ph, wxString *pfnam, unsigned int blono, bool mss, int deadhead)
//hashwert bestimmen und binär nach *ph schreiben
{
      wxString hashval,wfl,pdn;
      unsigned long long indx,blockzahl,origlen;
      unsigned int readno,i,startbyte;
      unsigned int rounds=2;
      char *plainblock, *ciphblock, *keyblock, dummy;
      longnumber lzahl;

     //hashwert bestimmen
     wxFFile plaintext(*pfnam,  _("rb"));
     if( !plaintext.IsOpened() ) {if(mss) throwout(_("plainfile \n not opened error")); return false; }
    //accomodating "deadhead"
         if( (unsigned long long) plaintext.Length() < (unsigned long) deadhead)
         {
             throwout(_("error!\nplainfile too short."));
             return false;
         }
     for(i=0;i<deadhead;i++)
     {
        readno=plaintext.Read(&dummy,1);
        if(readno != 1)
        {
            throwout(_("cipher file read error\ncannot continue"));
            plaintext.Close();
            return false;
        }
     }
     origlen=plaintext.Length() - (ulong32) deadhead;
     if(origlen==0)
     {
         {if(mss) throwout(_("Warnung:\nKann Hash von leerer oder fast leerer Datei nicht berechnen\ngebe dummy zurueck!"),10);}
         hashval=_("Empty File. Inserting dummy value for empty file. This is a very short value compared to the regular value!");
         (*ph).stortext(&hashval);
         plaintext.Close();
         return false;
     }
     blockzahl= (origlen/(ulong32)blono);
     if(origlen % (ulong32)blono !=0) { blockzahl++; }

     startbyte=(unsigned int)((ulong32)blono*blockzahl -  origlen); //Ueberstand am Anfang ist irgendwie zu fuellen
     plainblock=(char *)malloc(blono);
     ciphblock=(char *)malloc(blono);
     keyblock=(char *)malloc(blono);

     if(blockzahl <2)
     {
       {if(mss) throwout(_("Warnung!!\n Datei ist zu klein!\nPadding notwendig"),3);}
       readno=plaintext.Read(plainblock,(unsigned int)origlen);
       if(readno != origlen) {if(mss) throwout(_("Error reading File 8-O"));}
       for(indx=readno;indx<blono;indx++)
          *(plainblock+indx)= (*(plainblock+(indx%readno)))^((char) indx);
       for(i=0;i<rounds+1;i++)
       {
        if(!develop_e(plainblock,blono,rounds))
             {if(mss)  throwout(_("Unknown Error 1 in Hashing Fleas_4"));}
       }
     }
     else
     {
// ersten Block verarbeiten, reentrant auffüllen, so dass nur noch glatte bloecke bleibeb
       readno=plaintext.Read(plainblock,blono - startbyte); //restzahl vom ersten block einlesen
       if(readno != (blono - startbyte)) {if(mss) throwout(_("Error reading File 8-O"));}
       for(indx=readno;indx<blono;indx++)
          *(plainblock+indx)=(*(plainblock+(indx%readno)))^((char) indx) ;//wird mit xortem mit indx aufgefüllt
       if(!develop_e(plainblock,blono,rounds))
              {if(mss) throwout(_("Unknown Error 2 in Hashing Fleas_4"));}
       memcpy(keyblock,plainblock,blono);
       for(indx=1;indx<blockzahl;indx++)
       {
         readno=plaintext.Read(plainblock,blono); //einlesen von naechstem plainblock
         if(readno != blono) {if(mss) throwout(_("Error reading File 8-O"));}
         if(indx+1 ==blockzahl) //im letzten block eine runde mehr, loest das "extension problem"
         { rounds++;}
           if(!k_develop_f4(blono,plainblock,blono,keyblock,blono,keyblock,rounds,0))
                {if(mss) throwout(_("Unknown Error 3 in Hashing Fleas_4"));}
          memcpy(keyblock,plainblock,blono);
       }// zum naechsten block
     }
     plaintext.Close();
     //jetzt in die out_lonu schreiben
     ph->fillwith(plainblock,blono);
     //memcpy(ph->ad,plainblock,blono); //in die outzahl kopieren
     ph->setsize(true);
     ph->shrinktofit();
     free(plainblock); free(ciphblock); free(keyblock);
     return true;
}

/**************************************************/
/****************************************************************/
bool get_lonu_hash_from_filename_3(longnumber *ph, wxString *pfnam, unsigned int blono,bool mss, int deadhead )
//hashwert bestimmen und binär nach *ph schreiben
{
      wxString hashval,wfl,pdn;
      unsigned long long  indx,blockzahl,origlen;
      unsigned int readno,i,startbyte;
      unsigned int rounds=2;
      char *plainblock, *ciphblock, *keyblock, dummy;
      longnumber lzahl;

     //hashwert bestimmen
     wxFFile plaintext(*pfnam,  _("rb"));
     if( !plaintext.IsOpened() ) {if(mss) throwout(_("plainfile \n not opened error")); return false; }
         //accomodating "deadhead"
         if( (unsigned long long) plaintext.Length() < (unsigned long) deadhead)
         {
             throwout(_("error!\nplainfile too short."));
             plaintext.Close();
             return false;
         }
     for(i=0;i<deadhead;i++)
     {
        readno=plaintext.Read(&dummy,1);
        if(readno != 1)
        {
            throwout(_("cipher file read error\ncannot continue"));
            plaintext.Close();
            return false;
        }
     }
     origlen=plaintext.Length() - (ulong32) deadhead;

     if(origlen==0)
     {
         {if(mss) throwout(_("Warnung:\nKann Hash von leerer oder fast leerer Datei nicht berechnen\ngebe dummy zurueck!"),10);}
         hashval=_("Empty File. Inserting dummy value for empty file. This is a very short value compared to the regular value!");
         (*ph).stortext(&hashval);
         plaintext.Close();
         return false;
     }
     blockzahl= (origlen/(ulong32)blono);
     if(origlen % (ulong32)blono !=0) { blockzahl++; }

     startbyte=(unsigned int)((ulong32)blono*blockzahl -  origlen); //Ueberstand am Anfang ist irgendwie zu fuellen
     plainblock=(char *)malloc(blono);
     ciphblock=(char *)malloc(blono);
     keyblock=(char *)malloc(blono);

     if(blockzahl <2)
     {
       {if(mss) throwout(_("Warnung!!\n Datei ist zu klein!\nPadding notwendig"),3);}
       readno=plaintext.Read(plainblock,(unsigned int)origlen);
       if(readno != origlen) throwout(_("Error reading File 8-O"));
       for(indx=readno;indx<blono;indx++)
          *(plainblock+indx)= (*(plainblock+(indx%readno)))^((char) indx);
       for(i=0;i<rounds+1;i++)
       {
        if(!develop_b(plainblock,blono,rounds))
              {if(mss) throwout(_("Unknown Error 1 in Hashing Fleas_3"));}
       }
     }
     else
     {
// ersten Block verarbeiten, reentrant auffüllen, so dass nur noch glatte bloecke bleibeb
       readno=plaintext.Read(plainblock,blono - startbyte); //restzahl vom ersten block einlesen
       if(readno != (blono - startbyte)) {if(mss) throwout(_("Error reading File 8-O"));}
       for(indx=readno;indx<blono;indx++)
          *(plainblock+indx)=(*(plainblock+(indx%readno)))^((char) indx) ;//wird mit xortem mit indx aufgefüllt
       if(!develop_b(plainblock,blono,rounds))
              {if(mss) throwout(_("Unknown Error 2 in Hashing Fleas_3"));}
       memcpy(keyblock,plainblock,blono);
       for(indx=1;indx<blockzahl;indx++)
       {
         readno=plaintext.Read(plainblock,blono); //einlesen von naechstem plainblock
         if(readno != blono) {if(mss) throwout(_("Error reading File 8-O"));}
         if(indx+1 ==blockzahl) //im letzten block eine runde mehr, loest das "extension problem"
         { rounds++;}
           if(!k_develop_b(plainblock,blono,keyblock,blono,rounds))
                {if(mss) throwout(_("Unknown Error 3 in Hashing Fleas_3"));}
         memcpy(keyblock,plainblock,blono);
       }// zum nächsaten block
     }
     plaintext.Close();
     //jetzt in die out_lonu schreiben
     ph->fillwith(plainblock,blono);
     //memcpy(ph->ad,plainblock,blono); //in die outzahl kopieren
     ph->setsize(true);
     ph->shrinktofit();
     free(plainblock); free(ciphblock); free(keyblock);
     return true;
}

/**************************************************/
bool get_hash_from_filename(wxString *hstr, wxString *pfnam, int preflag, bool mss)
//hashwert bestimmen und nach *hstr schreiben
//am 7.1.11 rundenzahl von 1 auf 2 geaendert
{
      wxString hashval,wfl,pdn;
      unsigned long long indx,blockzahl,origlen;
      unsigned int readno,i,startbyte;
      unsigned int blono=65, rounds=3;
      char *plainblock, *ciphblock, *keyblock, *hlp;
      longnumber lzahl;

     //hashwert bestimmen
     wxFFile plaintext(*pfnam,  _("rb"));
     if( !plaintext.IsOpened() ) {if(mss) throwout(_("plainfile \n not opened error")); return false; }
     origlen=plaintext.Length();
     if(origlen==0)
     {
         {if(mss) throwout(_("Warnung:\nKann Hash von leerer oder fast leerer Datei nicht berechnen\ngebe dummy zurueck!"),10);}
         *hstr=_("Empty File. Inserting dummy value for empty file. This is a very short value compared to the regular value!");
         plaintext.Close();
         return false;
     }
     blockzahl= (origlen/(ulong32)blono);
     if(origlen % (ulong32)blono !=0) { blockzahl++; }

     startbyte=(unsigned int)((ulong32)blono*blockzahl -  origlen);
     plainblock=(char *)malloc(blono);
     ciphblock=(char *)malloc(blono);
     keyblock=(char *)malloc(blono);

     if(blockzahl <2)
     {
       {if(mss) throwout(_("Warnung!!\n Datei ist zu klein!\nPadding notwendig"));}
       readno=plaintext.Read(plainblock,(unsigned int)origlen);
       if(readno != origlen) {if(mss) throwout(_("Error reading File 8-O"));}
       for(indx=readno;indx<blono;indx++)
          *(plainblock+indx)= (*(plainblock+(indx%readno)))^((char) indx);
       for(i=0;i<rounds+1;i++)
       {
 //        if(!bytejump4(blono,plainblock,ciphblock))
       if(!bytmixx(blono,plainblock,ciphblock,2))
              {if(mss) throwout(_("Unknown Error in Hashing"));}
         hlp=plainblock; // plain und cipherblock vertauschen, so dass jetzt cipher weiterverarb wird
         plainblock=ciphblock;
         ciphblock=hlp;
       }
     }
     else
     {
// ersten Block verarbeiten, reentrant auffüllen, so dass nur noch glatte bloecke bleibeb
       readno=plaintext.Read(plainblock,blono - startbyte);
       if(readno != (blono - startbyte)) {if(mss) throwout(_("Error reading File 8-O"));}
       for(indx=readno;indx<blono;indx++)
          *(plainblock+indx)=(*(plainblock+(indx%readno)))^((char) indx) ;//wird mit xortem mit indx aufgefüllt
       for(i=0;i<rounds;i++)
       {
       //  erster teilaufgefüllter block
       if(!bytmixx(blono,plainblock,ciphblock,2))
              {if(mss) throwout(_("Unknown Error in Hashing"));}
         hlp=plainblock;// plain und cipherblock vertauschen, so dass jetzt cipher weiterverarb wird
         plainblock=ciphblock;
         ciphblock=hlp;
       }
       for(i=0;i<blono;i++)
       {
           *(keyblock +i) = *(plainblock +i);
       }
       for(indx=1;indx<blockzahl;indx++)
       {
         readno=plaintext.Read(plainblock,blono);
         if(readno != blono) {if(mss) throwout(_("Error reading File 8-O"));}
         if(indx+1 ==blockzahl) //im letzten block eine runde mehr, loest das "extension problem"
         { rounds++;}
         for(i=0;i<rounds;i++)
         {
           if(!kbytmixx(blono,plainblock,ciphblock,blono,keyblock,2))
                {if(mss) throwout(_("Unknown Error in Hashing"));}
           hlp=plainblock;     //switch adresses
           plainblock=ciphblock;
           ciphblock=hlp;
         }// immer wieder cipher und plain durchspielen und wechseln
         for(i=0;i<blono;i++)// letzte cipher im "plainblock"
         {
              *(keyblock +i) = *(plainblock +i);
         }

       }// zum naechsten block
     }
     plaintext.Close();
     lzahl.fillwith(plainblock,blono); //ist aber die cipher!
     lzahl.writechar(hstr);
     if(preflag==1) *hstr= _("s_65: ") + *hstr;
     free(plainblock); free(ciphblock); free(keyblock);
     return true;
}
/********************************************************************/
bool mac_from_file(longnumber *ph, wxString *pfnam, longnumber *p_lonukey, wxString hash_algo,wxString ciph_algo )
//does what it says, calculates a MAC from a file and deposits it in *ph
{
    int blono=512;
    //check for file existence
    if(!(wxFile::Exists(*pfnam)))
    {
        throwout(_("Datei zur MAC-Berechnung existiert nicht!\n Abbruch."),3);
        return false;
    }
    //if so, calculate hash value
    if(hash_algo==_("Fleas_o2")) blono=256;
    if( !(get_lonu_hash_from_filename_x(ph,pfnam,blono,hash_algo)) )
    {
        throwout(_("Could not calculate hash!\n Aborting."),3);
        return false;
    }
    //encipher hash value
    if(!(ph->lonu_lock(p_lonukey,ciph_algo)))
    {
        throwout(_("Could not lock hash for MAC\n Aborting."),3);
        return false;
    }
    //deposit in *ph
    ph->setsize(true);
    ph->shrinktofit();
    //if all went well return true
    return true;
}
/***************************************************************/
bool make_mac_file(longnumber *pkey, wxString *pfnam, wxString h_algo, wxString c_algo)
{
    longnumber mymac;
    wxString outfile,mytxt,hashnum;
    char *buff;
    ulong32 strinlen;

    //calculate mac
    if(!(mac_from_file(&mymac, pfnam, pkey, h_algo, c_algo )))
    {
        throwout(_("Error \nCould not make MAC\nAborting"),3);
        return false;
    }
    //write mac
    outfile=*pfnam + _(".mac");
    wxFFile macfil(outfile,  _("wb"));

    if( !macfil.IsOpened() ) {throwout(_("MAC File\nnot opened error")); return false; }
    mymac.writehex(&hashnum);
    mytxt= _("MAC: ") +hashnum;
    mytxt += _(" \nh_algo: ") + h_algo;
    mytxt += _(" \nc_algo: ") + c_algo;
    mytxt += _(" \n");

    strinlen= mytxt.Length();
    buff = (char *)malloc(2*strinlen+2);
    strcpy( buff, (const char*) mytxt.mb_str(wxConvUTF8) );
    macfil.Write(buff,strinlen);
    free(buff);
    return true;
}
/***************************************************************/
bool check_mac(longnumber *pky, wxString fnam, wxString macnam, bool stumm)
{
    FILE *setfp;
    char buff[4002]; //large, needed for long MACs
    longnumber mac_ff,mac_calc;
    wxString h_algo, c_algo, hashval;
    int fl1;

    //if macnam is "default", replace by fnam + .mac
    if(macnam==_("default")) macnam= fnam +_(".mac");
    //open mac file
    if(!wxFile::Exists(macnam))
    {
        if(!stumm) throwout(_("Mac-File nicht gefunden.\nMaster_key authorization?..."),2);
        return false;
    }

    //read params

    setfp=fopen((const char*)((macnam).mb_str(wxConvLocal)), "r");
    fl1=fscanf(setfp," MAC: %4000s",buff);
    if(fl1==1)
       {
           hashval=wxString::FromUTF8(buff);
       }
       else {if(!stumm) throwout(_("Could not read docpath parameter!"),2); return false;}

    fl1=fscanf(setfp," h_algo: %4000s",buff);
    if(fl1==1)
       {
           h_algo=wxString::FromUTF8(buff);
       }
       else {if(!stumm) throwout(_("Could not read hash name!"),2); return false;}

    fl1=fscanf(setfp," c_algo: %4000s",buff);
    if(fl1==1)
       {
           c_algo=wxString::FromUTF8(buff);
       }
       else {if(!stumm) throwout(_("Could not read cipher name!"),2); return false;}
    //convert lonu
    mac_ff.storhex(&hashval);
    //calculate mac
    if( !(mac_from_file(&mac_calc, &fnam, pky, h_algo,c_algo )))
    {
        if(!stumm) throwout(_("could not calculate MAC from file\nAborting."),2);
        return false;
    }
    //compare with stated mac
    if(mac_ff.compare(&mac_calc) != 0)
    {
        if(!stumm) throwout(_("Falscher MAC."),2);
        return false;
    }
    else return true;
}
/*****************************************************************/
bool ispowof2(int blolen)
{
    if((blolen < 2)||(blolen&1)) return false;
    do{
        blolen>>=1;
        if (blolen&1)
        {
            if(blolen == 1) return true;
            else return false;
        }
    } while (blolen >0);
    throwout(_("never see this ispowof2 error"));
    return false;
}
/********************************************************/
/**********************************************************/
bool name_enciph(wxString ofnam, longnumber* pky,wxString* pnn)
//neuen filenamen generieren und bei *cryptnam ablegen
{
    longnumber onam;
    wxString mypath,myname,myext,separator,cryptnam;

    separator=wxFileName::GetPathTerminators();
    wxFileName::SplitPath(ofnam, &mypath, &myname, &myext);
    myname += _(".") +myext;

    onam.stortext(&myname);
    //jetzt als nummer interpretieren und verschlüsseln
    onam.lonu_lock(pky);
    onam.writehex(&cryptnam);
    cryptnam=mypath + separator+ cryptnam;
    if(pnn != NULL) *pnn=cryptnam;
    //jetzt umbenennen
    wxRenameFile(ofnam, cryptnam);
    return true;
}
/**********************************************************/
bool testrenaming()
{
    longnumber key;
    wxString newnam,fnam=sec_path + _("/DiesIstMeinFilenameTest.txt");
    key.storlong(12345678);
    if(!wxFileExists(fnam)){throwout(_("File not found cannot test")); return false;}
    //verschlüsseln
    name_enciph(fnam,&key,&newnam);
    name_deciph(newnam,&key);
    if(!wxFileExists(fnam)){throwout(_("Could not find backconvert")); return false;}
    else return true;
}
/**********************************************************/
bool name_deciph(wxString ofnam,  longnumber* pky)
//neuen filenamen generieren und bei *cryptnam ablegen
{
    longnumber onam;
    wxString mypath,myname,separator,decryptnam;

    separator=wxFileName::GetPathTerminators();
    wxFileName::SplitPath(ofnam, &mypath, &myname,NULL);

    onam.storhex(&myname);
    //jetzt unlocken und characters recovern
    onam.lonu_unlock(pky);
    onam.shrinktofit();
    //onam content nach decryptnam
    onam.writewxstring(&decryptnam,1);
    decryptnam=mypath + separator+ decryptnam;
        //jetzt umbenennen
    wxRenameFile(ofnam, decryptnam);
    return true;
}
/**********************************************************/
bool file_pos_past2(wxString fnam, wxString terminator, ulong32 *firstpast, ulong32 maxcount)
//report first ocurrence of terminator and deposit first position past in *firstpast
//check till maxcount, return true on found, false on not found til maxcount
{
    char c,*buf,*buf2,c4[4];
    unsigned short termlen;
    int i,fl1;
    bool onechar = true, found=false; //symbol is represented as one byte
    ulong32 fpos=0;  //position of next to read char

    //build char array from wx string
    *firstpast=0;
    termlen=terminator.length();
    if(termlen<1){throwout(_("Zero Terminator length - are you nuts?"),3); return false;}
    buf= (char*) malloc(termlen);
    for(i=0;i<(int)termlen;i++)
    {
#if wxCHECK_VERSION(3,0,0)
        onechar=(terminator.GetChar(i)).GetAsChar(c4);
#else
       onechar = true;
       c4[0]=terminator.GetChar(0);
#endif
     if(!onechar)// weird character in delimiter
     {
        throwout(_("Error: Illegal symbol in delimiter\nAborting!"));
        free(buf);
        return false;
     }
     *(buf+i)= c4[0];
    }
    //open file
    wxFFile f1(fnam,_("rb"));
    if(!f1.IsOpened())
    {
        throwout(_("could not open target file"));
        free(buf);
        return false;
    }
    buf2= (char*) malloc(termlen);
    for(i=0;i<(int)termlen;i++) //fill buffer first from file
    {
        fl1=f1.Read(&c,1);
        fpos++;
        if(fl1!=1){free(buf);free(buf2); return false;}
        *(buf2+i)=c;  //put at appropriate place
    }
    while(!found)
    {
        if( memcmp(buf,buf2,termlen) ==0) //match found
        {
            found=true;
            *firstpast = fpos;
            free(buf);
            free(buf2);
            return true;
        }
        else //cycle once more
        {
            for (i=0;i<termlen-1;i++)
            {
                *(buf2+i)=*(buf2+i+1);
            }
        }
        //get new one and fill in
        fl1=f1.Read(&c,1);
        fpos++;
        if(fl1!=1){free(buf);free(buf2); return false;}
        *(buf2+termlen-1)= c;
        if(fpos>maxcount) //limit exceeded
        {
            free(buf);free(buf2); return false;
        }

    }
    throwout(_("Error: should never reach this code in file pos past 2"));
    return false;
}
/**********************************************************/
bool file_pos_past(wxString fnam, wxString terminator, ulong32 *firstpast, ulong32 maxcount)
//report first ocurrence of terminator and deposit first position past in *firstpast
//check till maxcount, return true on found, false on not found til maxcount
{
    int fl1,fl=1,termlen;
    ulong32 pos,fpos=0;
    bool ontrack=true;
    char c,*buf;

    termlen=terminator.length();
    if(termlen==0) return false;

    wxFFile of(fnam, _("rb"));
    if(!of.IsOpened())
    {
        throwout(_("could not open file to seek terminator"));
        return false;
    }
    buf= (char*) malloc(termlen+2);
    *firstpast=0;
    do
    {
        fl1=of.Read(&c,1);
        fpos++;
        if(fl1!=1){free(buf); return false;}
        if(c== terminator.GetChar(0))
        {
            pos=0;
            ontrack=true;
            //put into buffer and check whether complete
            *(buf+pos)=c;
            if(termlen==1)  //if only one char
            {
                free(buf);
                *firstpast= fpos;
                return true;
            }
            do
            {   //naechsten character laden
                pos++;
                fl=of.Read(&c,1);
                if(fl!=1){ free(buf); return false;}
                *(buf+pos)=c;
                if(c==terminator.GetChar(pos))  //noch im Raster
                {
                    if((long)pos+1 == termlen)
                    {
                        *firstpast=fpos+pos;
                        free(buf);
                        return true;
                    }
                    //sonst einfach weitermachen
                }
                //if different, cleanup store first and set one further
                if(c != terminator.GetChar(pos))
                {
                    of.Seek(fpos);
                    ontrack=false;
                 }
             } while(ontrack);
        }
    }while((fl1==1)&&(fl==1)&&(fpos<maxcount));
    free(buf);
    return false;
}
/**********************************************************/

/**********************************************************/
bool splicefile(wxString o_nam, wxString fnam1, wxString fnam2, wxString terminator)
//first terminator char may not repeat in terminator!
{
    wxString currpath, currname, separator,nf1,nf2;
    char c,*buf,*bufref;
    int i,j,   fl,fl1,termlen;
    ulong32 fpos=0;

    termlen=terminator.length();
    if(termlen==0) return false;
    buf= (char*) malloc(4*termlen+2);bufref= (char*) malloc(4*termlen+2);
    //fill bufref with reference chars
    strcpy(bufref, (const char*) terminator.mb_str(wxConvUTF8));
    strcpy(buf, (const char*) terminator.mb_str(wxConvUTF8));
    //separate path of o_nam and filename
    separator=wxFileName::GetPathTerminators();
    wxFileName::SplitPath(o_nam, &currpath, &currname,NULL);
    //openfile o_nam for reading and fnam1 for writing
    wxFFile of(o_nam, _("rb"));
    if(!of.IsOpened())
    {
        throwout(_("could not open file to split"));
        free(buf); free(bufref);
        return false;
    }
    //corrected for blank path!
    if(currpath !=_(""))
       {
           nf1=currpath+separator+fnam1;
           nf2=currpath+separator+fnam2;
       }
      else
      {
           nf1=fnam1;
           nf2=fnam2;
      }
      //open two outputfiles
    wxFFile f1(nf1,_("wb"));
    if(!f1.IsOpened())
    {
        throwout(_("could not open splicehead file"));
        free(buf); free(bufref);
        //of.Close();
        return false;
    }
    wxFFile f2(nf2,_("wb"));
    if(!f2.IsOpened())
    {
        throwout(_("could not open splicetail file"));
        free(buf); free(bufref);
        //of.Close(); f1.Close();
        return false;
    }
    //fill buffer with first termlength characters
    for(i=0;i< termlen;i++)
    {
		fl1=of.Read(&c,1);
		if(fl1!=1) //if finished writeout fragments and exit
		{
			for(j=0;j<i;j++)
			{
			  fl=f1.Write(buf+j,1);
			}
			free(buf); free(bufref);
            //of.Close(); f1.Close(); f2.Close();
            return true;
		}
		buf[i]=c;
        fpos++;
	}
    while((strcmp(buf,bufref)!=0))
    {
	   f1.Write(buf,1);
       memmove(buf,buf+1,termlen-1); //moveone down
       fl1=of.Read(&c,1);  //get next byte
       if(fl1 != 1) //EOF reached or error
       {
		   //writeout buffer
			for(j=0;j<termlen-1;j++)
			{
			  fl=f1.Write(buf+j,1);
			}
			free(buf); free(bufref);
			throwout(_("Warning, Splitmarker not found"),1);
            return true;
       }
       else fpos++;
       *(buf+termlen-1)=c;  //append to compare buffer
   }
   //here the marker has been found, forget buf and spit it all out to f2
   free(buf); free(bufref);
   fl1=of.Read(&c,1);  //get next byte
   while(fl1 == 1)
   {
	   fl=f2.Write(&c,1);
	   if(fl != 1)
	   {
			throwout(_("Warning, couldn't write second outfile"),1);
            return false;
	   }
	   fl1=of.Read(&c,1);  //get next byte
   }
   return true;
}
/***************************************************************/
/**********************************************************/
bool splicefile(wxString o_nam, wxString fnam1, wxString fnam2, wxString terminator, wxString restarter)
//first terminator char may not repeat in terminator!
{
    wxString currpath, currname, separator,nf1,nf2;
    char c,*buf2,*bufref2,*buf,*bufref;
    int i,j,   fl,fl1,termlen, startlen;
    ulong32 fpos=0;

    termlen=terminator.length();
    startlen = restarter.length();
    if((termlen==0)||(startlen==0)) return false;
    buf= (char*) malloc(4*termlen+2);bufref= (char*) malloc(4*termlen+2);
    buf2= (char*) malloc(4*startlen+2);bufref2= (char*) malloc(4*startlen+2);
    //fill bufref with reference chars
    strcpy(bufref, (const char*) terminator.mb_str(wxConvUTF8));
    strcpy(buf, (const char*) terminator.mb_str(wxConvUTF8));
    strcpy(bufref2, (const char*) restarter.mb_str(wxConvUTF8));
    strcpy(buf2, (const char*) restarter.mb_str(wxConvUTF8));
    //separate path of o_nam and filename
    separator=wxFileName::GetPathTerminators();
    wxFileName::SplitPath(o_nam, &currpath, &currname,NULL);
    //openfile o_nam for reading and fnam1 for writing
    wxFFile of(o_nam, _("rb"));
    if(!of.IsOpened())
    {
        throwout(_("could not open file to split"));
        free(buf); free(bufref);
        return false;
    }
    //corrected for blank path!
    if(currpath !=_(""))
       {
           nf1=currpath+separator+fnam1;
           nf2=currpath+separator+fnam2;
       }
      else
      {
           nf1=fnam1;
           nf2=fnam2;
      }
      //open two outputfiles
    wxFFile f1(nf1,_("wb"));
    if(!f1.IsOpened())
    {
        throwout(_("could not open splicehead file"));
        free(buf); free(bufref);
        free(buf2); free(bufref2);
        //of.Close();
        return false;
    }
    wxFFile f2(nf2,_("wb"));
    if(!f2.IsOpened())
    {
        throwout(_("could not open splicetail file"));
        free(buf); free(bufref);
        free(buf2); free(bufref2);
        //of.Close(); f1.Close();
        return false;
    }
    //fill buffer with first termlength characters
    for(i=0;i< termlen;i++)
    {
		fl1=of.Read(&c,1);
		if(fl1!=1) //if finished writeout fragments and exit
		{
			for(j=0;j<i;j++)
			{
			  fl=f1.Write(buf+j,1);
			}
			free(buf); free(bufref);
            free(buf2); free(bufref2);
            //of.Close(); f1.Close(); f2.Close();
            return false;
		}
		buf[i]=c;
        fpos++;
	}
    while((strcmp(buf,bufref)!=0))
    {
	   f1.Write(buf,1);
       memmove(buf,buf+1,termlen-1); //moveone down
       fl1=of.Read(&c,1);  //get next byte
       if(fl1 != 1) //EOF reached or error
       {
		   //writeout buffer
			for(j=0;j<termlen-1;j++)
			{
			  fl=f1.Write(buf+j,1);
			}
			free(buf); free(bufref);
			throwout(_("Warning, Splitmarker 1 not found"),1);
            return false;
       }
       else fpos++;
       *(buf+termlen-1)=c;  //append to compare buffer
   }
   //here the first marker has been found, start next bufref2 search and discard until bufref2 is foundforget buf and spit it all out to f2
   free(buf); free(bufref);

    //fill buffer2 with first startlen characters
    for(i=0;i< startlen;i++)
    {
		fl1=of.Read(&c,1);
		if(fl1!=1) //if finished exit
		{
			free(buf); free(bufref);
			free(buf2); free(bufref2);
            //of.Close(); f1.Close(); f2.Close();
            return false;
		}
		buf2[i]=c;
        fpos++;
	}
	//search loop for restarter
    while((strcmp(buf2,bufref2)!=0))
    {
	   //f1.Write(buf2,1);
       memmove(buf2,buf2+1,startlen-1); //moveone down
       fl1=of.Read(&c,1);  //get next byte
       if(fl1 != 1) //EOF reached or error
       {
			free(buf2); free(bufref2);
			throwout(_("Warning, Restartmarker not found"),1);
            return false;
       }
       else fpos++;
       *(buf2+startlen-1)=c;  //append to compare buffer
   }
   //now restart-sequence has been found, push rest into file 2
   free(buf2); free(bufref2);
   fl1=of.Read(&c,1);  //get next byte
   while(fl1 == 1)
   {
	   fl=f2.Write(&c,1);
	   if(fl != 1)
	   {
			throwout(_("Warning, couldn't write second outfile"),1);
            return false;
	   }
	   fl1=of.Read(&c,1);  //get next byte
   }
   return true;
}
/***************************************************************/

/***************************************************************/
bool fileconcat(wxString fn1, wxString fn2, wxString outnam)
{
    char mybl[1000];
    int fl;

    wxFFile f1(fn1,_("rb"));
    if(!f1.IsOpened()) {throwout(_("pilotfile could not be opened")); return false;}
    wxFFile o(outnam, _("wb"));
    if(!o.IsOpened()){throwout(_("targetfile could not be opened")); return false;}
    do
    {
        fl=f1.Read(&mybl[0],1000);
        if(fl>0) o.Write(&mybl[0],fl);
    } while(fl==1000);
    f1.Close();

    wxFFile f2(fn2,_("rb"));
    if(!f2.IsOpened()){throwout(_("trailing file could not be opened")); return false;}
    do
    {
        fl=f2.Read(&mybl[0],1000);
        if(fl>0) o.Write(&mybl[0],fl);
    } while(fl==1000);
    f2.Close();
    o.Close();
    return true;
}
/***************************************************************/
bool fileconcat(wxString fn1, wxString fn2, wxString outnam, wxString stopsig, ulong32 decoy)
{
    char mybl[1000];
    int fl;
    ulong32 i;

    wxFFile f1(fn1,_("rb"));
    if(!f1.IsOpened()) {throwout(_("pilotfile could not be opened")); return false;}
    wxFFile o(outnam, _("wb"));
    if(!o.IsOpened()){throwout(_("targetfile could not be opened")); return false;}
    do
    {
        fl=f1.Read(&mybl[0],1000);
        if(fl>0) o.Write(&mybl[0],fl);
    } while(fl==1000);
    f1.Close();

    wxFFile f2(fn2,_("rb"));
    if(!f2.IsOpened()){throwout(_("trailing file could not be opened")); return false;}

    o.Write(stopsig);
    //write random decoy
    for(i=0;i<decoy;i++)
    {
        mybl[0]=zuf.get_rbyte();
        o.Write(&mybl[0],1);
    }
    do
    {
        fl=f2.Read(&mybl[0],1000);
        if(fl>0) o.Write(&mybl[0],fl);
    } while(fl==1000);
    f2.Close();
    o.Close();
    return true;
}
/***************************************************************//***************************************************************/
bool develop_a( char* ad, int blolen, int rounds) //neue develop-funktion
//p1: intg,bytxor,comblock,  p2: comblock,bytxor,integ, xor together
{
    char *a2;
    unsigned char offs=17, inc=3;
    bool res;
    int i;

    //clone block block
    a2=(char*) malloc(blolen+1);
    if(a2==NULL){throwout(_("error1 in develop_a!")); return false;}
    memcpy(a2,ad,blolen);

    //work block1
    res=bytinteg(blolen,ad,offs,inc);
        if(!res){throwout(_("error 2 in develop_a!")); free(a2); return false;}
    res= bytxor(blolen,ad,ad,rounds);
        if(!res){throwout(_("error 3 in develop_a!")); free(a2); return false;}
    res= comblock(blolen, ad,rounds);
        if(!res){throwout(_("error 4 in develop_a!")); free(a2); return false;}

    //work block2
    res= comblock(blolen, a2,rounds);
        if(!res){throwout(_("error 5 in develop_a!")); free(a2); return false;}
    res= bytxor(blolen,a2,a2,rounds);
        if(!res){throwout(_("error 6 in develop_a!")); free(a2); return false;}
    res=bytinteg(blolen,a2,offs,inc);
        if(!res){throwout(_("error 7 in develop_a!")); free(a2); return false;}

    //xor to block1
    for(i=0;i<blolen;i++)
    {
        *(ad+i) ^= *(a2+i);
    }

    //kill block2
    free(a2);
    return true;
}
/***************************************************************/
bool develop_b( char* ad, int blolen, int rounds) //neue develop-funktion
//p1: intg,bytperm,comblock,  p2: comblock,bytxor,integ, xor together
{
    char * a2;
    unsigned char offs=17, inc=5;
    bool res;
    int i;

    //clone block block
    a2=(char*) malloc(blolen);
    if(a2==NULL){throwout(_("error1 in develop_b!")); return false;}
    memcpy(a2,ad,blolen);

    //work block1
    res=bytinteg(blolen,ad,offs,inc);
        if(!res){throwout(_("error 2 in develop_b!")); free(a2); return false;}
    res= bytperm(blolen,ad,ad,rounds);
        if(!res){throwout(_("error 3 in develop_b!")); free(a2); return false;}
    res= comblock(blolen, ad,rounds);
        if(!res){throwout(_("error 4 in develop_b!")); free(a2); return false;}

    //work block2
    res= comblock(blolen, a2,rounds);
        if(!res){throwout(_("error 5 in develop_b!")); free(a2); return false;}
    res= bytxor(blolen,a2,a2,rounds);
        if(!res){throwout(_("error 6 in develop_b!")); free(a2); return false;}
    res=bytinteg(blolen,a2,offs,inc);
        if(!res){throwout(_("error 7 in develop_b!")); free(a2); return false;}

    //xor to block1
    for(i=0;i<blolen;i++)
    {
        *(ad+i) ^= *(a2+i);
    }

    //kill block2
    free(a2);
    return true;
}
/***************************************************************/
bool develop_c( char* ad, int blolen, int rounds) //neue develop-funktion
//p1: intg,bytxor,comblock,  p2: comblock,bytperm,integ, xor together
{
    char * a2;
    unsigned char offs=17, inc=7;
    bool res;
    int i;

    //clone block block
    a2= (char *) malloc(blolen);
    if(a2==NULL){throwout(_("error1 in develop_c!")); return false;}
    memcpy(a2,ad,blolen);

    //work block1
    res=bytinteg(blolen,ad,offs,inc);
        if(!res){throwout(_("error 2 in develop_c!")); free(a2); return false;}
    res= bytxor(blolen,ad,ad,rounds);
        if(!res){throwout(_("error 3 in develop_c!")); free(a2); return false;}
    res= comblock(blolen, ad,rounds);
        if(!res){throwout(_("error 4 in develop_c!")); free(a2); return false;}

    //work block2

    res= comblock(blolen, a2,rounds);
        if(!res){throwout(_("error 5 in develop_c!")); free(a2); return false;}
    res= bytperm(blolen,a2,a2,rounds);
        if(!res){throwout(_("error 6 in develop_c!")); free(a2); return false;}
    res=bytinteg(blolen,a2,offs,inc);
        if(!res){throwout(_("error 7 in develop_c!")); free(a2); return false;}

    //xor to block1
    for(i=0;i<blolen;i++)
    {
        *(ad+i) ^= *(a2+i);
    }

    //kill block2
    if(a2 != NULL) free(a2);
    return true;
}
/***************************************************************/
bool develop_d( char* ad, int blolen, int rounds) //neue develop-funktion
//p1: intg,bytperm,comblock,  p2: comblock,bytperm,integ, xor together
{
    char *a2;
    unsigned char offs=17, inc=11;
    bool res;
    int i;

    //clone block block
    a2= (char *) malloc(blolen);
    if(a2==NULL){throwout(_("error1 in develop_a!")); return false;}
    memcpy(a2,ad,blolen);

    //work block1
    res=bytinteg(blolen,ad,offs,inc);
        if(!res){throwout(_("error 2 in develop_a!")); free(a2); return false;}
    res= bytperm(blolen,ad,ad,rounds);
        if(!res){throwout(_("error 3 in develop_a!")); free(a2); return false;}
    res= comblock(blolen, ad,rounds);
        if(!res){throwout(_("error 4 in develop_a!")); free(a2); return false;}

    //work block2
    res= comblock(blolen, a2,rounds);
        if(!res){throwout(_("error 5 in develop_a!")); free(a2); return false;}
    res= bytperm(blolen,a2,a2,rounds);
        if(!res){throwout(_("error 6 in develop_a!")); free(a2); return false;}
    res=bytinteg(blolen,a2,offs,inc);
        if(!res){throwout(_("error 7 in develop_a!")); free(a2); return false;}

    //xor to block1
    for(i=0;i<blolen;i++)
    {
        *(ad+i) ^= *(a2+i);
    }

    //kill block2
    free(a2);
    return true;
}
/***************************************************************/
bool develop_e( char* ad, int blolen, int rounds) //neue develop-funktion
//p1: intg,bytperm,comblock,  p2: comblock,bytperm,integ, xor together
{
    char *a2;
    unsigned char offs=17, inc=11;
    bool res;
    int i;

    //clone block block
    a2= (char *) malloc(blolen);
    if(a2==NULL){throwout(_("error1 in develop_e!")); return false;}
    memcpy(a2,ad,blolen);

    //work block1
    res=bytinteg(blolen,ad,offs,inc);
        if(!res){throwout(_("error 2 in develop_e!")); free(a2); return false;}
    res= bytperm(blolen,ad,ad,rounds);
        if(!res){throwout(_("error 4 in develop_e!")); free(a2); return false;}
    res= bytxor(blolen,ad,ad,rounds);
        if(!res){throwout(_("error 5 in develop_e!")); free(a2); return false;}
    res= comblock(blolen, ad,rounds);
        if(!res){throwout(_("error 6 in develop_e!")); free(a2); return false;}

    //work block2
    res=bytinteg(blolen,a2,offs+2,inc-4);
        if(!res){throwout(_("error 8 in develop_e!")); free(a2); return false;}
    res= bytxor(blolen,a2,a2,rounds);
        if(!res){throwout(_("error 9 in develop_e!")); free(a2); return false;}
    res= bytperm(blolen,a2,a2,rounds);
        if(!res){throwout(_("error 10 in develop_e!")); free(a2); return false;}
    res= comblock(blolen, a2,rounds);
        if(!res){throwout(_("error 11 in develop_e!")); free(a2); return false;}

    //xor to block1
    for(i=0;i<blolen;i++)
    {
        *(ad+i) ^= *(a2+i);
    }

    //kill block2
    free(a2);
    return true;
}
/***************************************************************/
/***************************************************************/
bool develop_f( char* ad, int blolen, int rounds) //neue develop-funktion
//p1: intg,bytperm,comblock,  p2: comblock,bytperm,integ, xor together
{
    char *a2;
    unsigned char offs=17, inc=11;
    bool res;
    int i;

    //clone block block
    a2= (char *) malloc(blolen);
    if(a2==NULL){throwout(_("error1 in develop_e!")); return false;}
    memcpy(a2,ad,blolen);

    //work block1
    res=bytinteg(blolen,ad,offs,inc);
        if(!res){throwout(_("error 2 in develop_e!")); free(a2); return false;}
    res= bytmult(blolen,ad,ad,rounds);
        if(!res){throwout(_("error 4 in develop_e!")); free(a2); return false;}
    res= bytadd(blolen,ad,ad,rounds);
        if(!res){throwout(_("error 5 in develop_e!")); free(a2); return false;}
    res= comblock(blolen, ad,rounds);
        if(!res){throwout(_("error 6 in develop_e!")); free(a2); return false;}

    //work block2
    res=bytinteg(blolen,a2,offs+2,inc-4);
        if(!res){throwout(_("error 8 in develop_e!")); free(a2); return false;}
    res= bytadd(blolen,a2,a2,rounds);
        if(!res){throwout(_("error 9 in develop_e!")); free(a2); return false;}
    res= bytmult(blolen,a2,a2,rounds);
        if(!res){throwout(_("error 10 in develop_e!")); free(a2); return false;}
    res= comblock(blolen, a2,rounds);
        if(!res){throwout(_("error 11 in develop_e!")); free(a2); return false;}

    //xor to block1
    for(i=0;i<blolen;i++)
    {
        *(ad+i) ^= *(a2+i);
    }

    //kill block2
    free(a2);
    return true;
}
/***************************************************************/
bool develop_x1( char* ad, int blolen, int rounds, int paths) //neue develop-funktion
//p1: intg,bytperm,comblock,  p2: comblock,bytperm,integ, xor together
{
    char *a2,*acc;
    unsigned char offs=3, inc=1;
    bool res;
    int i,j;

    if((paths<2)||(paths>128)){throwout(_("path parameter in develop_x\ndoes not make sense")); return false;}
    //clone block block
    a2= (char *) malloc(blolen);
    acc= (char *) malloc(blolen);
    if((a2==NULL)||(acc==NULL)){throwout(_("error1 in develop_e!")); free(a2); free(acc); return false;}
    //memcpy(a2,ad,blolen); -> now done in loop
    for(i=0;i<blolen;i++) *(acc+i)=0; //set accu to null
    //initial
    for(i=0;i<paths;i++)
    {
        //copy initial block
        memcpy(a2,ad,blolen);
        //work block
        res=bytinteg(blolen,a2,offs+i,inc+i);
         if(!res){throwout(_("error 1 in develop_x!")); free(a2); return false;}
        res= bytadd(blolen,a2,a2,rounds);
         if(!res){throwout(_("error 2 in develop_x!")); free(a2); return false;}
        res= bytmult(blolen,a2,a2,rounds);
         if(!res){throwout(_("error 3 in develop_x!")); free(a2); return false;}
        res= comblock(blolen, a2,rounds);
         if(!res){throwout(_("error 4 in develop_x!")); free(a2); return false;}
         //xor to accu
        for(j=0;j<blolen;j++)
        {
          *(acc+j) ^= *(a2+j);
        }
    }
    //overwrite inblock
    memcpy(ad,acc,blolen);
    //kill scratchblocks
    free(a2); free(acc);
    return true;
}
/***************************************************************/
bool co_develop_l( char* ad, int blolen, char* coad, int colen, int rounds) //brand new LEAN! co_developer
{
    char *tmpadd;
    bool retval;

    // parameter hygienie
    if((unsigned int)colen+(unsigned int)blolen>32000)
    {
        colen=32000-blolen;
    }
    if(colen<=0) return(develop_l(ad,blolen,rounds));
    tmpadd = (char *) malloc(blolen+colen+2); //+2 is stupid windows patch
    if(tmpadd==NULL) return false;
    //combine with coad
    memcpy(tmpadd,ad,blolen);
    memcpy(tmpadd+((unsigned int)blolen),coad, colen);
    //develop standard
    retval= develop_l(tmpadd,blolen+colen,rounds);
    //separate out
    memcpy(ad,tmpadd,blolen);
    free(tmpadd);
    return(retval);
}
/***************************************************************/
bool co_develop_ln( char* ad, int blolen, char* coad, int colen, int rounds,int paths, unsigned char spice) //brand new LEAN! co_developer
{
    char *tmpadd;
    bool retval;

    // parameter hygienie
    if((unsigned int)colen+(unsigned int)blolen>32000)
    {
        colen=32000-blolen;
    }
    if(colen<=0) return(develop_ls(ad,blolen,rounds,paths,spice));//paths 3 spice=1
    tmpadd = (char *) malloc(blolen+colen+2); //+2 is stupid windows patch
    if(tmpadd==NULL) return false;
    //combine with coad
    memcpy(tmpadd,ad,blolen);
    memcpy(tmpadd+((unsigned int)blolen),coad, colen);
    //develop standard
    retval= develop_ls(tmpadd,blolen+colen,rounds,paths,spice);//paths 3 spice=1
    //separate out
    memcpy(ad,tmpadd,blolen);
    free(tmpadd);
    return(retval);
}
/***********************************************************************************************/
bool co_develop_ln2( char* ad, int blolen, char* coad, int colen, int rounds,int paths, unsigned char spice) //brand new LEAN! co_developer
{
    char *tmpadd;
    bool retval;

    // parameter hygienie
    if((blolen<=0)||(colen<0)) return false;
    if((unsigned int)colen+(unsigned int)blolen>32000)
    {
        colen=32000-blolen;
    }
    if(colen<=0) return(develop_ln2_mt(ad,blolen,rounds,paths,spice));//paths 3 spice=1

    tmpadd = (char *) malloc(blolen+colen+2); //+2 is stupid windows patch
    if(tmpadd==NULL) return false;
    //combine with coad
    memcpy(tmpadd,ad,blolen);
    memcpy(tmpadd+((unsigned int)blolen),coad, colen);
    //develop standard
    retval= develop_ln2_mt(tmpadd,blolen+colen,rounds,paths,spice);//paths 3 spice=1
    //separate out
    memcpy(ad,tmpadd,blolen);
    free(tmpadd);
    return(retval);
}

/***************************************************************/
bool co_develop_ls( char* ad, int blolen, char* coad, int colen, int rounds,int paths, unsigned char spice) //brand new LEAN! co_developer
{
    char *tmpadd;
    bool retval;

    // parameter hygienie
    if((unsigned int)colen+(unsigned int)blolen>32000)
    {
        colen=32000-blolen;
    }
    if(colen<=0) return(develop_ls(ad,blolen,rounds,paths,spice));//paths 3 spice=1
    tmpadd = (char *) malloc(blolen+colen+2); //+2 is stupid windows patch
    if(tmpadd==NULL) return false;
    //combine with coad
    memcpy(tmpadd,ad,blolen);
    memcpy(tmpadd+((unsigned int)blolen),coad, colen);
    //develop standard
    retval= develop_ls(tmpadd,blolen+colen,rounds,paths,spice);//paths 3 spice=1
    //separate out
    memcpy(ad,tmpadd,blolen);
    free(tmpadd);
    return(retval);
}
/***************************************************************/
bool develop_ls( char* ad, int blolen, int rounds, int paths, unsigned char spice, unsigned char inc) //brand new LEAN! developer
//path: integrate, [bytadd_a], [bytmult_a], comblock, spiced with path number
//algflag 0: bytadd_a solo,  algflag 1: bytmulta_solo, algflag 2 or other: both algorithms
{
    char *a2,*acc,*a3;
    bool res,even=true;
    int i,j;
    unsigned char offs;

    a2= (char *) malloc(blolen);
    a3= (char *) malloc(blolen);
    acc= (char *) malloc(blolen);
    offs=spice;  //enty parameter for integration
    if((paths & 1)==1)even=false;

    if((a2==NULL)||(acc==NULL)||(a3==NULL)){throwout(_("error1 in develop_l!")); free(a2); free(a3); free(acc); return false;}
    //fill acc with first result  zeroes
    //memset(acc,0,blolen);
    memcpy(a2,ad,blolen);
    res=bytinteg(blolen,a2,offs,inc); //once for both, common integration
    memcpy(a3,a2,blolen);
    res=bytinteg(blolen,a3,offs,inc); //anotherone for second
    res= bytadd_a(blolen,a2,a2,rounds,spice,2); //spice is 1, two flea rounds
         if(!res){throwout(_("error add in develop_l!")); free(a2); free(a3); free(acc); return false;}
    res= bytmult_a(blolen,a3,a3,rounds,spice+6,2); //spice is 7, two flea rounds
         if(!res){throwout(_("error mult in develop_l!")); free(a2); free(a3); free(acc); return false;}
    for(j=0;j<blolen;j++)
        {
          *(acc+j) = *(a2+j) ^ *(a3+j);
        }

    for(i=1;i<paths/2;i++)
    {
      memcpy(a2,ad,blolen);
      res=bytinteg(blolen,a2,offs+i,inc+i); //once for both, common integration
      memcpy(a3,a2,blolen);
      //bytmulta one and bytadd_a the other
      res= bytadd_a(blolen,a2,a2,rounds,spice+i,2); //spice is i, two flea rounds
         if(!res){throwout(_("error add in develop_l!")); free(a2); free(a3); free(acc); return false;}
      if(even)
      {
          res=bytinteg(blolen,a3,offs+i,inc+i); //anotherone for second
          res= bytmult_a(blolen,a3,a3,rounds,spice+6+i,2); //spice is 6+i, two flea rounds
          if(!res){throwout(_("error mult in develop_l!")); free(a2); free(a3); free(acc); return false;}
          for(j=0;j<blolen;j++)
           {
             *(acc+j) ^= *(a2+j) ^ *(a3+j);
           }
      }
      else
      {
          for(j=0;j<blolen;j++)
           {
             *(acc+j) ^= *(a2+j);
           }
      }
    }

    //overwrite inblock
    memcpy(ad,acc,blolen);
    //kill scratchblocks
    free(a2); free(acc); free(a3);
    return true;
}
/***************************************************************/
/***************************************************************/
bool develop_ls2( char* ad, int blolen, int rounds, int paths, unsigned char spice, unsigned char inc) //brand new LEAN! developer
//path: integrate, [bytadd_a], [bytmult_a], comblock, spiced with path number
//algflag 0: bytadd_a solo,  algflag 1: bytmulta_solo, algflag 2 or other: both algorithms
{
    char *a2,*acc,*a3;
    bool res,even=true;
    int i,j;
    unsigned char offs;
    badd_thr *p_ad_thr[2];
    bmul_thr *p_mul_thr[2];
    bool resa[2],resm[2];

    a2= (char *) malloc(blolen);
    a3= (char *) malloc(blolen);
    acc= (char *) malloc(blolen);
    offs=spice;  //enty parameter for integration
    if((paths & 1)==1)even=false;

    if((a2==NULL)||(acc==NULL)||(a3==NULL)){throwout(_("error1 in develop_l!")); free(a2); free(a3); free(acc); return false;}
    //fill acc with first result  zeroes
    //memset(acc,0,blolen);
    memcpy(a2,ad,blolen);
    res=bytinteg(blolen,a2,offs,inc); //once for both, common integration
    memcpy(a3,a2,blolen);
    res=bytinteg(blolen,a3,offs,inc); //anotherone for second

//do it in thread
    //res= bytadd_a(blolen,a2,a2,rounds,spice,2); //spice is 1, two flea rounds
         //if(!res){throwout(_("error add in develop_l!")); free(a2); free(a3); free(acc); return false;}

    p_ad_thr[0]=new badd_thr(&resa[0], blolen,a2,a2,rounds,spice,2);
       if ( p_ad_thr[0]->Create() != wxTHREAD_NO_ERROR )
        {
            throwout(_("Can't create the thread!"),3);
            delete p_ad_thr[0];
           p_ad_thr[0] = NULL;
        }
        else
        {
            if (p_ad_thr[0]->Run() != wxTHREAD_NO_ERROR )
            {
                throwout(_("Can't create the thread!"),3);
                delete p_ad_thr[0];
                p_ad_thr[0] = NULL;
            }
        }
        //add thread is fired

    //do it in thread
   // res= bytmult_a(blolen,a3,a3,rounds,spice+6,2); //spice is 7, two flea rounds
         //if(!res){throwout(_("error mult in develop_l!")); free(a2); free(a3); free(acc); return false;}

    p_mul_thr[0]=new bmul_thr(&resm[0], blolen,a3,a3,rounds,spice,2);
       if ( p_mul_thr[0]->Create() != wxTHREAD_NO_ERROR )
        {
            throwout(_("Can't create the thread!"),3);
            delete p_mul_thr[0];
           p_mul_thr[0] = NULL;
        }
        else
        {
            if (p_mul_thr[0]->Run() != wxTHREAD_NO_ERROR )
            {
                throwout(_("Can't create the thread!"),3);
                delete p_mul_thr[0];
                p_mul_thr[0] = NULL;
            }
        }
        //mult thread is fired
//now join threads
        p_ad_thr[0]->Wait();
        p_mul_thr[0]->Wait();
        if((!resa[0])||(!resm[0]))  {throwout(_("error mult in develop_l!")); free(a2); free(a3); free(acc); return false;}



    for(j=0;j<blolen;j++)
        {
          *(acc+j) = *(a2+j) ^ *(a3+j);
        }

    for(i=1;i<(paths+1)/2;i++)
    {
      memcpy(a2,ad,blolen);
      res=bytinteg(blolen,a2,offs+i,inc+i); //once for both, common integration
      memcpy(a3,a2,blolen);
      //bytmulta one and bytadd_a the other
      res= bytadd_a(blolen,a2,a2,rounds,spice+i,2); //spice is i, two flea rounds
         if(!res){throwout(_("error add in develop_l!")); free(a2); free(a3); free(acc); return false;}
      if(even)
      {
          res=bytinteg(blolen,a3,offs+i,inc+i); //anotherone for second
          res= bytmult_a(blolen,a3,a3,rounds,spice+6+i,2); //spice is 6+i, two flea rounds
          if(!res){throwout(_("error mult in develop_l!")); free(a2); free(a3); free(acc); return false;}
          for(j=0;j<blolen;j++)
           {
             *(acc+j) ^= *(a2+j) ^ *(a3+j);
           }
      }
      else
      {
          for(j=0;j<blolen;j++)
           {
             *(acc+j) ^= *(a2+j);
           }
      }
    }

    //overwrite inblock
    memcpy(ad,acc,blolen);
    //kill scratchblocks
    free(a2); free(acc); free(a3);
    return true;
}
/***************************************************************/
/***************************************************************/
bool develop_ln( char* ad, int blolen, int rounds, int paths, unsigned char spice, unsigned char inc) //brand new LEAN! developer
//path: integrate, [bytadd_a], [bytmult_a], comblock, spiced with path number
//one of add and mult can be opted out by setting alg_flag
//algflag 0: bytadd_a solo,  algflag 1: bytmulta_solo, algflag 2 or other: both algorithms
{
    char *a2,*acc,*a3,*a4;
    bool res,even=true;
    int i,j;
    unsigned char offs;

    a2= (char *) malloc(blolen);
    a3= (char *) malloc(blolen);
    a4= (char *) malloc(blolen);
    acc= (char *) malloc(blolen);
    offs=spice;  //enty parameter for integration


    if((a2==NULL)||(acc==NULL)||(a3==NULL)||(a4==NULL)){throwout(_("error1 in develop_ln!")); free(a2); free(a3); free(a4); free(acc); return false;}
    //fill acc with first result  zeroes
    //memset(acc,0,blolen);
    memcpy(a2,ad,blolen);
    res=bytinteg(blolen,a2,offs,inc); //once for both, common integration
    memcpy(a3,a2,blolen);
    res=bytinteg(blolen,a3,offs,inc); //anotherone for second
    if(paths>2) memcpy(a4,a3,blolen);  //store doubly integrated block here
    res= bytadd_a(blolen,a2,a2,rounds,spice,2); //spice is 1, two flea rounds
         if(!res){throwout(_("error add in develop_l!")); free(a2); free(a3); free(a4); free(acc); return false;}
    res= bytmult_a(blolen,a3,a3,rounds,spice+6,2); //spice is 7, two flea rounds
         if(!res){throwout(_("error mult in develop_l!")); free(a2); free(a3); free(a4); free(acc); return false;}
    for(j=0;j<blolen;j++)
        {
          *(acc+j) = *(a2+j) ^ *(a3+j);
        }

    for(i=3;i<=paths;i++)
    {
      even=true;
      if((i & 1)==1)even=false;
      res=bytinteg(blolen,a4,offs+i,inc+i); //another round of integration
      memcpy(a2,a4,blolen); //use a2 as intermediate storage and working memory
      if(even)
      {
        res= bytadd_a(blolen,a2,a2,rounds,spice+i,2); //spice is i, two flea rounds
        if(!res){throwout(_("error add in develop_l!")); free(a2); free(a3); free(a4); free(acc); return false;}
      }
      else
      {
        res= bytmult_a(blolen,a2,a2,rounds,spice+i,2); //spice is i, two flea rounds
        if(!res){throwout(_("error add in develop_l!")); free(a2); free(a3); free(a4); free(acc); return false;}
      }

      for(j=0;j<blolen;j++)
           {
             *(acc+j) ^= *(a2+j);
           }
    }

    //overwrite inblock
    memcpy(ad,acc,blolen);
    //kill scratchblocks
    free(a2); free(acc); free(a3); free(a4);
    return true;
}
/***************************************************************/
bool develop_longblock( char* ad, ulong32 blolen, unsigned char spice)
{
    char *hbl;
    ulong32 i;

    hbl= (char*) malloc(blolen);
    if(hbl== NULL)
    {
        throwout(_("error in dev_longblock!\nmemory shortage?\naborting! "));
        return false;
    }
    memcpy(hbl,ad,blolen);
    if(!dbytadd_a_l(blolen,hbl,hbl,1,spice,1.5))
    {
       free(hbl);
       return false;
    }
    if(!dbytadd_a_l(blolen,ad,ad,1,spice+1,1.5))
    {
       free(hbl);
       return false;
    }
    for(i=0;i< blolen-3;i+=4)
    {
        *((ulong32*)(ad+i)) ^= *((ulong32*)(hbl + i));
    }
    while(i<blolen)
    {
        *(ad + i) ^= *(hbl + i);
        i++;
    }
    free(hbl);
    return true;
}

/***************************************************************/
bool develop_flight( char* ad, int blolen, unsigned char spice)
{
    char *hbl;
    int i;

    hbl= (char*) malloc(blolen);
    if(hbl== NULL)
    {
        throwout(_("error in dev_flight!\naborting! "));
        return false;
    }
    memcpy(hbl,ad,(unsigned int)blolen);
    if(!dbytadd_a(blolen,hbl,hbl,1,spice,1.5))
    {
       free(hbl);
       return false;
    }
    if(!bbytmult_a(blolen,ad,ad,1,spice+1,1.5))
    {
       free(hbl);
       return false;
    }
    /*for(i=0;i<blolen;i++)  //to do: make 4byte variant of this
    {
        *(ad + i) ^= *(hbl + i);
    }*/
    for(i=0;i< blolen-3;i+=4)
    {
        *((ulong32*)(ad+i)) ^= *((ulong32*)(hbl + i));
    }
    while(i<blolen)
    {
        *(ad + i) ^= *(hbl + i);
        i++;
    }
    free(hbl);
    return true;
}
/*****************************************************************************************/
bool develop_flightx_onearm( char* ad, int blolen, unsigned char spice)
//only ever use in conjunction with other xored cipher!, backtrace vulnerabilty in solo use!
{
    char *hbl;
    int i;
    unsigned int blolenx;

    if(blolen&3) blolenx = (blolen - (blolen&3) + 4);
    else blolenx=blolen;
    hbl= (char*) malloc(blolenx);  //make sure to allocate full integer from last address

    if(hbl== NULL)
    {
        throwout(_("error in dev_flight!\naborting! "));
        return false;
    }
    memcpy(hbl,ad,(unsigned int)(blolen)); //only copy what's given
    for(i=blolen;i<blolenx;i++)  //set possible additional bytes to zero
    {
      *(hbl+i)=0;
    }

    if(!dbytadd_ax(blolenx,hbl,hbl,1,spice,1.5))
    {
       free(hbl);
       return false;
    }
   /* if(!dbytadd_ay(blolenx,hbl,hbl,1,spice,1.5))
    {
       free(hbl);
       return false;
    }*/
    for(i=0;i< blolen-3;i+=4)
    {
        *((ulong32*)(ad+i)) ^= *((ulong32*)(hbl + i));
    }
    while(i<blolen)
    {
        *(ad + i) ^= *(hbl + i);
        i++;
    }
    free(hbl);
    return true;
}
/****************************version 2***********************************/
bool develop_flightx( char* ad, int blolen, unsigned char spice, float intensity)
{
    char *hbl;
    int i;
    unsigned int blolenx;

    if(blolen&3) blolenx = (blolen - (blolen&3) + 4);
    else blolenx=blolen;
    hbl= (char*) malloc(blolenx);  //make sure to allocate full integer from last address

    if(hbl== NULL)
    {
        throwout(_("error in dev_flight!\naborting! "));
        return false;
    }
    memcpy(hbl,ad,(unsigned int)(blolen)); //only copy what's given
    for(i=blolen;i<blolenx;i++)  //set possible additional bytes to zero
    {
      *(hbl+i)=0;
    }

    if(!dbytadd_ax(blolenx,hbl,hbl,1,spice,intensity))
    {
       free(hbl);
       return false;
    }
    if(!dbytadd_ay(blolenx,hbl,hbl,1,spice,intensity))
    {
       free(hbl);
       return false;
    }
    for(i=0;i< blolen-3;i+=4)
    {
        *((ulong32*)(ad+i)) ^= *((ulong32*)(hbl + i));
    }
    while(i<blolen)
    {
        *(ad + i) ^= *(hbl + i);
        i++;
    }
    free(hbl);
    return true;
}
/***************************************************************/
/***************************************************************/
bool develop_ln2( char* ad, int blolen, int rounds, int paths, unsigned char spice, unsigned char inc) //brand new LEAN! developer
//path: integrate, [bytadd_a], [bytmult_a], comblock, spiced with path number
//one of add and mult can be opted out by setting alg_flag
//algflag 0: bytadd_a solo,  algflag 1: bytmulta_solo, algflag 2 or other: both algorithms
{
    char *acc;
    char **a;
    bool res;
    unsigned int i,j,max;
    unsigned char offs;


    //for now do it multithreaded
    //return develop_ln2_mt( ad, blolen, rounds, paths, spice, inc); //multithreaded version



    a=(char **)malloc((paths+1)*sizeof(char *));
    if(a==NULL){throwout(_("allocation error  (0) in developln2"),5); return false;}
    for(i=0;i<(unsigned int)paths+1;i++)
    {
        *(a+i) = (char *) (malloc(blolen));
        if(*(a+i) == NULL)
        {
            for(j=0;j<i;j++)
            {
                free( (void *) *(a+i));
            }
            free( (void *) a);
            throwout(_("allocation error  (1) in developln2"),5);
            return false;
        }
    }
    acc= *(a+paths);
    //all  storage blocks allocated

    offs=spice;  //entry parameter for integration
    memcpy(*a,ad,blolen); //deposit in-block
    res=bytinteg(blolen,*a,offs,inc);
    for(i=1;i<(unsigned int)paths;i++)  //set all integrated in-blocks
    {
        memcpy(*(a+i),*(a+i-1),blolen);
        res=bytinteg(blolen,*(a+i),offs+(unsigned char)i,inc+(unsigned char)i);
    }
    //chained integration blocks set!
    for(i=0;i<(unsigned int)paths;i++)
    {
        if((i&1)==1) //odd
        {
                res= bytadd_a(blolen,*(a+i),*(a+i),rounds,spice,2); //spice is 1, two flea rounds
        }
        else  //even
        {
                res= bytmult_a(blolen,*(a+i),*(a+i),rounds,spice+6,2); //spice is 1, two flea rounds
        }
        if(!res)
        {
            for(j=0;j<i;j++)
                    {
                        free( (void *) *(a+i));
                    }
            free( (void *) a);
            throwout(_("bytadd error in develop_ln2"),5);
            return false;
        }
    }
    max=blolen>>2;
    for(j=0;j<max;j++)
    {
        *( (ulong32*) ( (acc+(j<<2)))) = 0;
        for(i=0;i<(unsigned int)paths;i++)
        {
            *( (ulong32*) ( (acc+(j<<2)))) ^=  *((ulong32 *) ( (*(a+i)) +(j<<2)));
        }
    }
    for(j=max<<2;j<(unsigned int)blolen;j++)
    {
        *(acc+j) = 0;
        for(i=0;i<(unsigned int)paths;i++)
        {
                *(acc+j) ^= *((*(a+i)) +j);
        }
    }  //alles gexort - hopefully :-)
    //overwrite inblock
    memcpy(ad,acc,blolen);
    //kill scratchblocks
    for(i=0;i<(unsigned int)paths+1;i++)
    {
        free( (void *) *(a+i));
    }
    free((void *) a);
    return true;
}
/***************************************************************/
bool develop_ln2_mt( char* ad, int blolen, int rounds, int paths, unsigned char spice, unsigned char inc) //brand new LEAN! developer
//path: integrate, [bytadd_a], [bytmult_a], comblock, spiced with path number
//one of add and mult can be opted out by setting alg_flag
//algflag 0: bytadd_a solo,  algflag 1: bytmulta_solo, algflag 2 or other: both algorithms
{
    char *acc;
    char **a;
    bool res;
    unsigned int i,j,max;
    unsigned char offs;
    oneway_thr **ppt;
    bool **prs;
    bool odd;
    unsigned char loc_spice;




    a=(char **)malloc((paths+1)*sizeof(char *));
    if(a==NULL){throwout(_("allocation error  (0) in developln2"),5); return false;}
    ppt= (oneway_thr **) malloc(paths*sizeof(oneway_thr *));
    if(ppt==NULL){throwout(_("allocation error  (0.1) in developln2"),5); free(a); return false;}
    prs= (bool **) malloc(paths*sizeof(bool *));
    if(prs==NULL){throwout(_("allocation error  (0.2) in developln2"),5); free(a); free(ppt); return false;}
    for(i=0;i<(unsigned int)paths+1;i++)
    {
        *(a+i) = (char *) (malloc(blolen));
        if(*(a+i) == NULL)
        {
            for(j=0;j<i;j++)
            {
                free( (void *) *(a+i));
            }
            free( (void *) a);
            free((void *) ppt);
            free((void *) prs);
            throwout(_("allocation error  (1) in developln2"),5);
            return false;
        }
    }
    acc= *(a+paths);
    //all  storage blocks allocated

    offs=spice;  //entry parameter for integration
    memcpy(*a,ad,blolen); //deposit in-block
    res=bytinteg(blolen,*a,offs,inc);
    for(i=1;i<(unsigned int)paths;i++)  //set all integrated in-blocks
    {
        memcpy(*(a+i),*(a+i-1),blolen);
        res=bytinteg(blolen,*(a+i),offs+(unsigned char)i,inc+(unsigned char)i);
    }
    //chained integration blocks set!
    for(i=0;i<(unsigned int)paths;i++)
    {
        *(prs+i) =(bool *) malloc(sizeof(bool));
        if((i&1)==1)
        {
            odd =true;
            loc_spice= spice;
        }
        else
        {
            odd=false;
            loc_spice=spice+6;
        }
        *(ppt+i)=new oneway_thr(odd,*(prs+i), blolen,*(a+i),*(a+i),rounds,loc_spice,2);
        if ( (*(ppt+i))->Create() != wxTHREAD_NO_ERROR )
            {
                throwout(_("Can't create the thread!"),3);
                delete *(ppt+i);
                *(ppt+i)=NULL;
            }
        else
          {
            if ( (*(ppt+i))->Run() != wxTHREAD_NO_ERROR )
            {
                throwout(_("Can't Run the thread!"),3);
                delete *(ppt+i);
                *(ppt+i) = NULL;
            }
          }
        //add thread is fired
                //res= bytadd_a(blolen,*(a+i),*(a+i),rounds,spice,2); //spice is 1, two flea rounds
        if(*(ppt+i)==NULL)  //error
        {
            //kill all allocated pointers and return false aka harakiri!
            for(j=0;j<=i;j++)
            {
                if(j<i) (*(ppt+i))->Wait();
                free(*(a+j)); //buffers
                free(*(prs+i)); //results
                free(a); //root buffer
                free(prs);  //root result
                free(ppt);
                return false;
            }
        }
    }

    max=blolen>>2;
    for(i=0;i<(unsigned int)paths;i++)
    {
        (*(ppt+i))->Wait(); //wait for next thread to finish
        for(j=0;j<max;j++)
        {
           if(i==0) *( (ulong32*) ( (acc+(j<<2)))) = 0;
            *( (ulong32*) ( (acc+(j<<2)))) ^=  *((ulong32 *) ( (*(a+i)) +(j<<2)));
        }
        for(j=max<<2;j<(unsigned int)blolen;j++)
        {
            if(i==0) *(acc+j) = 0;
            *(acc+j) ^= *((*(a+i)) +j);
        }
    }  //alles gexort - hopefully :-)
    //overwrite inblock
    memcpy(ad,acc,blolen);
    //kill scratchblocks
    for(i=0;i<(unsigned int)paths+1;i++)
    {
        free( (void *) *(a+i));
        if(i<(unsigned int)paths) free( (void *) *(prs+i));
    }
    free((void *) a);
    free(prs);
    free(ppt);
    return true;
}
/***************************************************************/
/***************************************************************/
bool develop_l( char* ad, int blolen, int rounds, memblo *mpt) // LEAN! developer
//path: integrate, [bytadd_a], [bytmult_a], comblock, spiced with path number
//one of add and mult can be opted out by setting alg_flag
//algflag 0: bytadd_a solo,  algflag 1: bytmulta_solo, algflag 2 or other: both algorithms
{
    char *a2,*acc;
    unsigned char offs=3, inc=1;
    bool res;
    int j;

    if(mpt==NULL)
    {
        a2= (char *) malloc(blolen);
        acc= (char *) malloc(blolen);
    }
    else
    {
        a2=mpt->get_mb(0);
        acc=mpt->get_mb(1);
        if((a2==NULL)||(acc==NULL)){throwout(_("error1 in develop_l!")); free(a2); free(acc); return false;}
    }
    memcpy(a2,ad,blolen);
    res=bytinteg(blolen,a2,offs,inc); //once for both, common integration
    memcpy(acc,a2,blolen);
    //bytmulta one and bytadd_a the other
    res= bytadd_a(blolen,a2,a2,rounds,1,2); //spice is 1, two flea rounds
         if(!res){throwout(_("error add in develop_l!")); free(a2); return false;}
    res= bytmult_a(blolen,acc,acc,rounds,7,2); //spice is 7, two flea rounds
         if(!res){throwout(_("error mult in develop_l!")); free(a2); return false;}
    for(j=0;j<blolen;j++)
        {
          *(acc+j) ^= *(a2+j);
        }

    //overwrite inblock
    memcpy(ad,acc,blolen);
    //kill scratchblocks
    if(mpt==NULL) //memless call
    {
        free(a2); free(acc);
    }
    return true;
}
/***************************************************************/
bool develop_l2( char* ad, int blolen, int rounds) // LEAN! developer
//path: integrate, [bytadd_a], [bytmult_a], comblock, spiced with path number
//one of add and mult can be opted out by setting alg_flag
//algflag 0: bytadd_a solo,  algflag 1: bytmulta_solo, algflag 2 or other: both algorithms
{
    char *a2,*acc;
    unsigned char offs=3, inc=1;
    bool res;
    int j;

    a2= (char *) malloc(blolen);
    acc= (char *) malloc(blolen);
    if((a2==NULL)||(acc==NULL)){throwout(_("error1 in develop_l!")); free(a2); free(acc); return false;}
    memcpy(a2,ad,blolen);
    res=bytinteg(blolen,a2,offs,inc); //once for both, common integration
    memcpy(acc,a2,blolen);
    //bytmulta one and bytadd_a the other
    res= bytadd_a(blolen,a2,a2,rounds,1,2); //spice is 1, two flea rounds
         if(!res){throwout(_("error add in develop_l!")); free(a2); return false;}
    res= bytmult_a(blolen,acc,acc,rounds,7,2); //spice is 7, two flea rounds
         if(!res){throwout(_("error mult in develop_l!")); free(a2); return false;}
    for(j=0;j<blolen;j++)
        {
          *(acc+j) ^= *(a2+j);
        }

    //overwrite inblock
    memcpy(ad,acc,blolen);
    //kill scratchblocks
    free(a2); free(acc);
    return true;
}
/***************************************************************/
/***************************************************************/
bool develop_o( char* ad, int blolen, int rounds, int paths,char alg_flag) //brand new developer
//path: integrate, [bytadd_a], [bytmult_a], comblock, spiced with path number
//one of add and mult can be opted out by setting alg_flag
//algflag 0: bytadd_a solo,  algflag 1: bytmulta_solo, algflag 2 or other: both algorithms
{
    char *a2,*acc;
    unsigned char offs=3, inc=1;
    bool res;
    int i,j;

    if((paths<2)||(paths>128)){throwout(_("path parameter in develop_x\ndoes not make sense")); return false;}
    //clone block block
    a2= (char *) malloc(blolen);
    acc= (char *) malloc(blolen);
    if((a2==NULL)||(acc==NULL)){throwout(_("error1 in develop_o!")); free(a2); free(acc); return false;}
    //memcpy(a2,ad,blolen); -> now done in loop
    for(i=0;i<blolen;i++) *(acc+i)=0; //set accu to null
    //initial
    for(i=0;i<paths;i++)
    {
        //copy initial block
        memcpy(a2,ad,blolen);
        //work block
        res=bytinteg(blolen,a2,offs+i,inc+i);
         if(!res){throwout(_("error 1 in develop_o!")); free(a2); return false;}
    if(alg_flag != 1)
    {
        res= bytadd_a(blolen,a2,a2,rounds,i);
         if(!res){throwout(_("error 2 in develop_o!")); free(a2); return false;}
    }
    if(alg_flag != 0)
    {
        res= bytmult_a(blolen,a2,a2,rounds,i);
         if(!res){throwout(_("error 3 in develop_o!")); free(a2); return false;}
    }
        res= comblock(blolen, a2,rounds);
         if(!res){throwout(_("error 4 in develop_o!")); free(a2); return false;}
         //xor to accu
        for(j=0;j<blolen;j++)
        {
          *(acc+j) ^= *(a2+j);
        }
    }
    //overwrite inblock
    memcpy(ad,acc,blolen);
    //kill scratchblocks
    free(a2); free(acc);
    return true;
}
/***************************************************************/
bool develop_x( char* ad, int blolen, int rounds, int paths) //neue develop-funktion
//p1: intg,bytperm,comblock,  p2: comblock,bytperm,integ, xor together
{
    char *a2,*acc;
    unsigned char offs=3, inc=1;
    bool res;
    int i,j;

    if((paths<2)||(paths>128)){throwout(_("path parameter in develop_x\ndoes not make sense")); return false;}
    //clone block block
    a2= (char *) malloc(blolen);
    acc= (char *) malloc(blolen);
    if((a2==NULL)||(acc==NULL)){throwout(_("error1 in develop_x!")); free(a2); free(acc); return false;}
    //crypto atomics -> now done in loop
    for(i=0;i<blolen;i++) *(acc+i)=0; //set accu to null
    //initial
    for(i=0;i<paths;i++)
    {
        //copy initial block
        memcpy(a2,ad,blolen);
        //work block
        res=bytinteg(blolen,a2,offs+i,inc+i);
         if(!res){throwout(_("error 1 in develop_x!")); free(a2); return false;}
        res= bytxor(blolen,a2,a2,rounds);
         if(!res){throwout(_("error 2 in develop_x!")); free(a2); return false;}
        res= bytperm(blolen,a2,a2,rounds);
         if(!res){throwout(_("error 3 in develop_x!")); free(a2); return false;}
        res= comblock(blolen, a2,rounds);
         if(!res){throwout(_("error 4 in develop_x!")); free(a2); return false;}
         //xor to accu
        for(j=0;j<blolen;j++)
        {
          *(acc+j) ^= *(a2+j);
        }
    }
    //overwrite inblock
    memcpy(ad,acc,blolen);
    //kill scratchblocks
    free(a2); free(acc);
    return true;
}
/***************************************************************/
bool k_develop_a( char* ad, int blolen, char* keytxt, int keylen, int rounds)
{
   int i;
   char *kbuff;

   //key muss evtl auf blolen gehashed werden
   if(keylen>= blolen) kbuff=bytmixxmalloc(keylen, keytxt, 1, blolen);
   //oder reentrant auffüllen
   else
   {
         kbuff= (char *) malloc(blolen);
   //keyblock kbuff mit key auffüllen
         memcpy(kbuff,keytxt,keylen);
         for(i= keylen;i<blolen;i++)
                *(kbuff+i)=(*(kbuff+(i%keylen)))^((char) i) ;//wird mit xortem mit indx aufgefüllt
   }

   for(i=0;i<blolen;i++)
   {
       *(kbuff+i)= *(ad + i) ^ *(kbuff+i); //kbuff wird mit bladwell gexort
   }
   //nun einfach dvelop aufrufen
   if(! develop_a(kbuff,blolen,rounds)){throwout(_("error in k_develop_a")); return false;}
   //und in den Eingangskanal zurueckschreiben
   memcpy(ad,kbuff,blolen);
   free(kbuff);
   return true;
}
/*********************************************************/
bool k_develop_b( char* ad, int blolen, char* keytxt, int keylen, int rounds) //p1:perm, p2:xor
{
   int i;
   char *kbuff;

   //key muss evtl auf blolen gehashed werden
   if(keylen>= blolen) kbuff=bytmixxmalloc(keylen, keytxt, 1, blolen);
   //oder reentrant auffüllen
   else
   {
         kbuff= (char *) malloc(blolen);
   //keyblock kbuff mit key auffüllen
         memcpy(kbuff,keytxt,keylen);
         for(i= keylen;i<blolen;i++)
                *(kbuff+i)=(*(kbuff+(i%keylen)))^((char) i) ;//wird mit xortem mit indx aufgefüllt
   }

   for(i=0;i<blolen;i++)
   {
       *(kbuff+i)= *(ad + i) ^ *(kbuff+i); //kbuff wird mit bladwell gexort
   }
   //nun einfach dvelop aufrufen
   if(! develop_b(kbuff,blolen,rounds)){throwout(_("error in k_develop_b")); return false;}
   //und in den Eingangskanal zurueckschreiben
   memcpy(ad,kbuff,blolen);
   free(kbuff);
   return true;
}
/*********************************************************/
bool k_develop_c( char* ad, int blolen, char* keytxt, int keylen, int rounds) //p1:xor, p2:perm
{
   int i;
   char *kbuff;

   //key muss evtl auf blolen gehashed werden
   if(keylen>= blolen) kbuff=bytmixxmalloc(keylen, keytxt, 1, blolen);
   //oder reentrant auffüllen
   else
   {
         kbuff= (char *) malloc(blolen);
   //keyblock kbuff mit key auffüllen
         memcpy(kbuff,keytxt,keylen);
         for(i= keylen;i<blolen;i++)
                *(kbuff+i)=(*(kbuff+(i%keylen)))^((char) i) ;//wird mit xortem mit indx aufgefüllt
   }

   for(i=0;i<blolen;i++)
   {
       *(kbuff+i)= *(ad + i) ^ *(kbuff+i); //kbuff wird mit bladwell gexort
   }
   //nun einfach dvelop aufrufen
   if(! develop_c(kbuff,blolen,rounds)){throwout(_("error in k_develop_c")); return false;}
   //und in den Eingangskanal zurueckschreiben
   memcpy(ad,kbuff,blolen);
   free(kbuff);
   return true;
}
/*********************************************************/
bool k_develop_d( char* ad, int blolen, char* keytxt, int keylen, int rounds) //p1:perm, p2: perm
{
   int i;
   char *kbuff;

   //key muss evtl auf blolen gehashed werden
   if(keylen>= blolen) kbuff=bytmixxmalloc(keylen, keytxt, 1, blolen);
   //oder reentrant auffüllen
   else
   {
         kbuff= (char *) malloc(blolen);
   //keyblock kbuff mit key auffüllen
         memcpy(kbuff,keytxt,keylen);
         for(i= keylen;i<blolen;i++)
                *(kbuff+i)=(*(kbuff+(i%keylen)))^((char) i) ;//wird mit xortem mit indx aufgefüllt
   }

   for(i=0;i<blolen;i++)
   {
       *(kbuff+i)= *(ad + i) ^ *(kbuff+i); //kbuff wird mit bladwell gexort
   }
   //nun einfach dvelop aufrufen
   if(! develop_d(kbuff,blolen,rounds)){throwout(_("error in k_develop_d")); return false;}
   //und in den Eingangskanal zurueckschreiben
   memcpy(ad,kbuff,blolen);
   free(kbuff);
   return true;
}
/*********************************************************/
/*********************************************************/
bool counter_chain_freq(ulong32 *oa, ulong32 targetno, int blsiz, wxString tt, int rounds)
//oa is pointerarray (256 els) to put frequencies in, targetno is expected hitno, blocksize, Testtype sets testtype
{
   int j;
    longnumber n;
    unsigned char *bl1;
    ulong32 cnt=0;
    ulong32 lops,i;

    //check for possible testtypes
    if( !( (tt==_("bytadd_solo")) || (tt==_("bytmult_solo")) || (tt==_("mult_add")) || (tt==_("add_mult")) ||
          (tt==_("int_add")) || (tt==_("int_add_combl")) || (tt==_("dev_a_half")) || (tt==_("dev_a_full"))|| (tt==_("dev_e"))
            || (tt==_("bytmult_a_solo")) || (tt==_("bytperm_solo")) || (tt==_("mult_a_add"))|| (tt==_("bytadd_a_solo"))
          || (tt==_("mult_a_add_a"))|| (tt==_("add_a_mult_a")) ))
    {
        throwout(_("unknown test in counter chain freq"));
        return false;
    }

    //set blocksize, set counterstrucutre
    n.storlong(0);
    //calculate lops from targetno
    lops=(targetno*256)/blsiz +1; //end securely reached
    n.resizelonu(blsiz);
    bl1=(unsigned char *)malloc(blsiz);
    for(i=0;i<256;i++) *(oa+i)=0;  //resultarray nullen
      //loop
    for(i=0;i<lops;i++)
      {
          memcpy(bl1,n.ad,blsiz);
          //block for applying appropriate function
          if(tt==_("bytadd_solo"))  bytadd(blsiz,(char *)bl1,(char *)bl1,rounds);
          else if(tt==_("bytadd_a_solo"))  bytadd_a(blsiz,(char *)bl1,(char *)bl1,rounds);
          else if(tt==_("bytmult_solo"))  bytmult(blsiz,(char *)bl1,(char *)bl1,rounds);
          else if(tt==_("bytmult_a_solo"))
                   {bytmult_a(blsiz,(char *)bl1,(char *)bl1,rounds);}
          else if(tt==_("bytperm_solo"))  bytperm(blsiz,(char *)bl1,(char *)bl1,rounds);

          else if(tt==_("mult_add"))
            {
                for(j=0;j<rounds;j++)
                {
                    bytmult(blsiz,(char *)bl1,(char *)bl1,1);
                    bytadd(blsiz,(char *)bl1,(char *)bl1,1);
                }
            }
          else if(tt==_("mult_a_add"))
            {
                for(j=0;j<rounds;j++)
                {
                    bytmult_a(blsiz,(char *)bl1,(char *)bl1,1,5);
                    bytadd(blsiz,(char *)bl1,(char *)bl1,1);
                }
            }
           else if(tt==_("mult_a_add_a"))
            {
                for(j=0;j<rounds;j++)
                {
                    bytmult_a(blsiz,(char *)bl1,(char *)bl1,1,5);
                    bytadd_a(blsiz,(char *)bl1,(char *)bl1,1);
                }
            }
          else if(tt==_("add_mult"))
            {
                for(j=0;j<rounds;j++)
                {
                    bytadd(blsiz,(char *)bl1,(char *)bl1,1);
                    bytmult(blsiz,(char *)bl1,(char *)bl1,1);
                }
            }

          else if(tt==_("add_a_mult_a"))
            {
                for(j=0;j<rounds;j++)
                {
                    bytadd_a(blsiz,(char *)bl1,(char *)bl1,1);
                    bytmult_a(blsiz,(char *)bl1,(char *)bl1,1);
                }
            }
          else if(tt==_("int_add"))
            {
                for(j=0;j<rounds;j++)
                {
                    bytinteg(blsiz, (char*)bl1, 17, 3);
                    bytadd(blsiz,(char *)bl1,(char *)bl1,1);
                }
            }

          else if(tt==_("int_add_combl"))
            {
                for(j=0;j<rounds;j++)
                {
                    bytinteg(blsiz, (char*)bl1, 17, 3);
                    bytadd(blsiz,(char *)bl1,(char *)bl1,1);
                    comblock(blsiz, (char*) bl1,1);
                }
            }
          else if(tt==_("dev_a_half"))
            {
                    bytinteg(blsiz, (char*)bl1, 17, 3);
                    bytxor(blsiz,(char *)bl1,(char *)bl1,rounds);
                    comblock(blsiz, (char*)bl1,rounds);
            }

           else if(tt==_("dev_a_full"))
            {
                    develop_a((char *)bl1,blsiz,rounds);
            }

          else if(tt==_("dev_e"))
            {
                    develop_e((char *)bl1,blsiz,rounds);
            }
         // else if(tt==_(""))  bytperm(blsiz,(char *)bl1,(char *)bl1,rounds);
          else{throwout(_("unknown type")); free(bl1); return false;}


          for(j=0;j<blsiz;j++)
          {
              *(oa+ *(bl1+j) )+=1;
              cnt++;
              if(cnt>=targetno*256){ free((void *) bl1); return true;} //normal exit point
          }
          n.inc();
          n.resizelonu(blsiz);
      }
    free((void *) bl1);
    throwout(_("should not run into this message ?!"));
    return false;
}
/************************************************************/
/*********************************************************/
bool eval_freq_list(ulong32 *oa, float *pchi, float *pbpb)
//oa is pointerarray (256 els) to put frequencies in, targetno is expected hitno, blocksize, Testtype sets testtype
{
   long double accu,pi,erw,h;
   int i;
   ulong32 cnt=0;


      accu=0;
      for(i=0;i<256;i++) cnt+= *(oa+i);  //count hits
      erw=(long double)cnt / 256;

      for(i=0;i<256;i++)
      {
         if(*(oa+i)==0) continue;
         pi=((long double)*(oa+i))/(cnt);
         pi=-pi*log(pi)/log(2.0);
         accu+=pi;
      }
      *pbpb= (float)accu;

      accu=0;
      for(i=0;i<256;i++)
      {
         h=((long double)*(oa+i)-erw)* ((long double)*(oa+i)-erw)/ erw;
         accu+=h;
      }
   *pchi= (float) accu;
   return true;
}
/************************************************************/
int splitstring(wxString in, wxArrayString *p_out, wxChar sep)
//separates in into return value outstrings with sep as separator, deleted in out-array
{
    int f,l,end,i;
    wxString tmps;
    bool drin=false;

    f=0; i=0, l=0;
    end=in.Len();

   (*p_out).Clear();
    for(i=0;i<end;i++)
    {
        if(in.GetChar(i)== sep)
        {
            if(drin)
            {
                l=i;
                (*p_out).Add( in.Mid(f, l-f) );
                drin=false;
            }
            else
            {
                f=i;
            }
        }
        else
        {
            if(!drin)
            {
                f=i; //startindex
                drin=true;
            }
         }
    }
    //last one not added yet
    if(drin) (*p_out).Add( in.Right(end-f));
    return (*p_out).GetCount();
}
/*********************************************************/
int keyid_free(wxString nknam, bool verbose)
//returns bit 0 set if private present, bit 1 set if public, -1 on error
{
    FILE *knamfil;
    wxString h2;
    bool oldname=false;
    char buff1[1602],buff2[1602],buffref[1602];
    int lbl,retval=0;


    //length check
    if((nknam).Len()>400)
         {
           throwout(_("Alert! suggested Key-ID excessively long\n truncating!"));
           nknam.Truncate(400);
         }
    strcpy(buffref, (const char*) nknam.mb_str(wxConvUTF8));

    //open priv keynamfile, append name and id, close keynamfile
    if(S_FIL) unlockinplace(sec_path + _("/ell_knamfile.knf"),&zugang);
    h2= sec_path + _("/ell_knamfile.knf");
    knamfil=fopen((const char*)h2.mb_str(wxConvLocal), "r");
    if(knamfil == NULL)
    {
        throwout(_("Could not find private key name file\naborting!"));
        if(S_FIL) lockinplace(sec_path + _("/ell_knamfile.knf"),&zugang);
        return -1;
    }
    else //gut geoeffnet
    {
     do{
        lbl=fscanf(knamfil, " id: %400s Name: %400s \n",buff1,buff2);
        if(strcmp(buff1,buffref) ==0)
        {
            oldname = true;
            if(verbose) throwout(nknam+_("\nName bereits in Verwendung fuer priv. schluessel!"),2);
            retval |= 1;
            break;
        }
     }while(lbl == 2);
     fclose(knamfil);
     if(S_FIL) lockinplace(sec_path + _("/ell_knamfile.knf"),&zugang);
    }

    //if(!oldname) //wenn noch kein double, nochmal für pubfiles
    {
    //open priv keynamfile
    if(S_FIL) unlockinplace(sec_path + _("/e_ell_knamfile.knf"),&zugang);
    h2= sec_path + _("/e_ell_knamfile.knf");
    knamfil=fopen((const char*)h2.mb_str(wxConvLocal), "r");
    if(knamfil == NULL)
    {
        throwout(_("Could not find public key name file\naborting!"));
        if(S_FIL) lockinplace(sec_path + _("/e_ell_knamfile.knf"),&zugang);
        return -1;
    }
    else //gut geoeffnet
    {
     do{
        lbl=fscanf(knamfil, " id: %400s Name: %400s \n",buff1,buff2);
        if(strcmp(buff1,buffref) ==0)
        {
            oldname = true;
            if(verbose) throwout(nknam +_("\nName wird bereits fuer oefftl. Schluessel benutzt!\nVersuche eine andere ID!"),2);
            retval |= 2;
            break;
        }
       }while(lbl == 2);
       fclose(knamfil);
       if(S_FIL) lockinplace(sec_path + _("/e_ell_knamfile.knf"),&zugang);
     }
    }
    return retval;
}
/*********************************************************/
bool pub_keyid_free(wxString nknam, bool verbose)
{
    FILE *knamfil;
    wxString h2;
    bool oldname=false;
    char buff1[1602],buff2[1602],buffref[1602];
    int lbl;


    //length check
    if((nknam).Len()>400)
         {
           throwout(_("Alert! suggested Key-ID excessively long\n truncating!"));
           nknam.Truncate(400);
         }
    strcpy(buffref, (const char*) nknam.mb_str(wxConvUTF8));

    if(!oldname) //wenn noch kein double, nochmal für pubfiles
    {
    //open public keynamfile, append name and id, close keynamfile
    if(S_FIL) unlockinplace(sec_path + _("/e_ell_knamfile.knf"),&zugang);
    h2= sec_path + _("/e_ell_knamfile.knf");
    knamfil=fopen((const char*)h2.mb_str(wxConvLocal), "r");
    if(knamfil == NULL)
    {
        throwout(_("Could not find public key name file\naborting!"));
        if(S_FIL) lockinplace(sec_path + _("/e_ell_knamfile.knf"),&zugang);
        return false;
    }
    else //gut geoeffnet
    {
     do{
        lbl=fscanf(knamfil, " id: %400s Name: %400s \n",buff1,buff2);
        if(strcmp(buff1,buffref) ==0)
        {
            oldname = true;
            if(verbose) throwout(nknam +_("\nID wird bereits fuer oeffentl. Schluessel benutzt\nversuche eine andere ID!"),2);
            break;
        }
       }while(lbl == 2);
       fclose(knamfil);
       if(S_FIL) lockinplace(sec_path + _("/e_ell_knamfile.knf"),&zugang);
     }
    }
    if(oldname)
     {
         return false;
     }
    else return true; //ID is still available
}
/*************************************************************************//*********************************************************/
bool priv_keyid_free(wxString nknam, bool verbose)
{
    FILE *knamfil;
    wxString h2;
    bool oldname=false;
    char buff1[402],buff2[402],buffref[802];
    int lbl;


    //length check
    if((nknam).Len()>400)
         {
           throwout(_("Alert! suggested Key-ID excessively long\n truncating!"));
           nknam.Truncate(400);
         }
    strcpy(buffref, (const char*) nknam.mb_str(wxConvUTF8));

    //open priv keynamfile, append name and id, close keynamfile
    if(S_FIL) unlockinplace(sec_path + _("/ell_knamfile.knf"),&zugang);
    h2= sec_path + _("/ell_knamfile.knf");
    knamfil=fopen((const char*)h2.mb_str(wxConvLocal), "r");
    if(knamfil == NULL)
    {
        throwout(_("Could not find private key name file\naborting!"));
        if(S_FIL) lockinplace(sec_path + _("/ell_knamfile.knf"),&zugang);
        return false;
    }
    else //gut geoeffnet
    {
     do{
        lbl=fscanf(knamfil, " id: %400s Name: %400s \n",buff1,buff2);
        if(strcmp(buff1,buffref) ==0)
        {
            oldname = true;
            if(verbose) throwout(nknam+_("\nID wird bereits fuer privaten Schluessel benutzt!"),2);
            break;
        }
     }while(lbl == 2);
     fclose(knamfil);
     if(S_FIL) lockinplace(sec_path + _("/ell_knamfile.knf"),&zugang);
    }

    if(oldname)
     {
         return false;
     }
    else return true; //ID is still available
}
/*************************************************************************/
bool filetrunc_head(wxString fn1, unsigned int hedlen)
{
    unsigned int cnt,rno,flen;
    char c;
    bool result;


    wxFFile infil(fn1, _("rb"));
    if(!infil.IsOpened())
    {
        throwout(_("Error\ncould not open file to truncate!\nDo Nothing."),3);
        return false;
    }
    wxFFile outfil(fn1+_("_ex"), _("wb"));
    if(!outfil.IsOpened())
    {
        throwout(_("Error\ncould not open file to truncate-write!\nDo Nothing."),3);
        infil.Close();
        return false;
    }
    //seek position of payload
    infil.Seek(hedlen);
    //put everything into outfile
    cnt=0;
    flen= infil.Length();
    do
    {
        rno=infil.Read(&c,1);
        if( rno==1  )
        {
            rno=outfil.Write(&c,1);
            cnt++;
        }
    }while(rno==1);
    infil.Close();
    outfil.Close();
    result = wxRenameFile(fn1+_("_ex"),fn1); //ovrwrite and rename
    return result;
}
/*************************************************************************/
/*************************************************************************/
bool maketrunc_head(wxString fn_in, wxString fn_out, unsigned int hedlen)
{
    unsigned int cnt,rno,flen;
    char c;


    wxFFile infil(fn_in, _("rb"));
    if(!infil.IsOpened())
    {
        throwout(_("Error\ncould not open file to read_truncate!\nDo Nothing."),3);
        return false;
    }
    wxFFile outfil(fn_out, _("wb"));
    if(!outfil.IsOpened())
    {
        throwout(_("Error\ncould not open file to truncate-write!\nDo Nothing."),3);
        infil.Close();
        return false;
    }
    //seek position of payload
    infil.Seek(hedlen);
    //put everything into outfile
    cnt=0;
    flen= infil.Length();
    do
    {
        rno=infil.Read(&c,1);
        if( rno==1  )
        {
            rno=outfil.Write(&c,1);
            cnt++;
        }
    }while(rno==1);
    infil.Close();
    outfil.Close();
    return true;
}
/*************************************************************************/
bool extr_initials(wxString *pinstr, wxString *poutstr)
{
    wxString sxr1, trunk;
    bool isdone=false;
    int i;


        sxr1=*pinstr;
        sxr1.Trim(false); //trim from left
        sxr1.Trim(); //trim from right
        trunk += sxr1.Left(2); //first two nonwhite Letters
        sxr1= sxr1.Right(sxr1.Len()-2); //Take them away
        //eat away until next blank or empty
        do
        {
          do
          {
           i= sxr1.Find(_(" "));
           if (i== wxNOT_FOUND) {isdone=true;}
           if(isdone) break;
           sxr1=sxr1.Right(sxr1.Len()-i);
           sxr1.Trim(false); //trim from left
           if(sxr1.Len() > 1) trunk += sxr1.Left(2);
           else if(sxr1.Len() == 1) trunk += sxr1.Left(1);
           else isdone=true;
          }while(!isdone);
        }while(!isdone);
        *poutstr=trunk;
        return true;
}
/*********************************************************************************/
// ende meine Hilfsfuns

// ende meine Hilfsfuns
